#!/bin/bash
eval "$(conda shell.bash hook)"
conda activate grinnder

# ------------------------------------
# 1. Load custom paths from text file
# ------------------------------------
TEXT_FILE="custom_info.txt"

CUSTOM_DIR=$(grep '^CUSTOM_DIR='    "$TEXT_FILE" | cut -d'=' -f2)
DATASET_DIR=$(grep '^DATASET_DIR='  "$TEXT_FILE" | cut -d'=' -f2)
CKPT_DIR=$(grep '^CKPT_DIR='        "$TEXT_FILE" | cut -d'=' -f2)
STORAGE_DIR=$(grep '^STORAGE_DIR='  "$TEXT_FILE" | cut -d'=' -f2)

if [ -z "$CUSTOM_DIR" ]; then
  echo "Error: CUSTOM_DIR value not found in text file!"
  exit 1
fi

echo "The custom directory is: $CUSTOM_DIR"

export PYTHONPATH="$PYTHONPATH:$CUSTOM_DIR"
export PYTHONDONTWRITEBYTECODE=1
export PYTHONUNBUFFERED=1

# ------------------------------------
# 2. Experiment grid
# ------------------------------------
DATASETS=("products")               # ("products")
MODELS=("HongTu" "GRD-G" "GRD-GC")  # ("HongTu" "GRD-G" "GRD-GC")
LAYERS=("3" "5")
PARTITION_METHODS=("metis" "random" "grinnder")
N_PARTITIONS=4                      # fixed across runs

LOG_DIR="artifact_results"
rm -rf artifact_results
mkdir -p "$LOG_DIR"

# ------------------------------------
# 3. Run the experiments
# ------------------------------------
for dataset in "${DATASETS[@]}"; do
  for part_method in "${PARTITION_METHODS[@]}"; do
    for model in "${MODELS[@]}"; do
      for layers in "${LAYERS[@]}"; do

        # Pick checkpointing strategy
        if [ "$model" == "HongTu" ]; then
          CHECKPOINTING="cpu"
          STORAGE_OFFLOAD=""
        elif [ "$model" == "GRD-G" ]; then
          CHECKPOINTING="scattered"
          STORAGE_OFFLOAD=""
        else
          CHECKPOINTING="scattered"
          STORAGE_OFFLOAD="--storage_offload"
        fi

        LOG_FILE="${LOG_DIR}/${dataset}_${model}_${layers}layers_${part_method}.log"

        ARGS="
          --dataset $dataset
          --n-partitions $N_PARTITIONS
          --storage_path $STORAGE_DIR
          --dataset_path $DATASET_DIR
          --ckpt_path $CKPT_DIR
          --partition-method $part_method
          --n-epoch 2
          --model gcn
          --n-hidden 256
          --n-conv-layers $layers
          --optimize_dataloader
          --checkpointing_strategy $CHECKPOINTING
          --loguru_path $LOG_FILE
          $STORAGE_OFFLOAD
        "

        echo "Running: $dataset | $model | ${layers}L | $part_method (P=$N_PARTITIONS)"
        python main.py $ARGS

      done
    done
  done
done
