import argparse
import os
from typing import Generator, override

from mow.utils.program import Program


def collect_programs() -> Generator[type[Program], None, None]:
    """
    Collects all the programs available in the CLI.
    This is a placeholder function that should be implemented to collect actual programs.
    """
    for filename in os.listdir(os.path.dirname(__file__)):
        if filename.endswith(".py") and filename not in (
            "__init__.py",
            "main.py",
        ):
            module_name = os.path.splitext(filename)[0]
            module = __import__(f"mow.cli.{module_name}", fromlist=[""])
            for attr_name in dir(module):
                attr = getattr(module, attr_name)
                if (
                    isinstance(attr, type)
                    and issubclass(attr, Program)
                    and not attr == Program
                ):
                    yield attr
                    break
            else:
                raise TypeError(
                    f"mow.cli.{module_name} does not define a valid Program class"
                )


class MainProgram(Program, name="main"):
    @override
    @staticmethod
    def add_arguments(parser: argparse.ArgumentParser):
        """
        Adds the subprograms to the main parser.
        """
        MainProgram.build_subprogram(
            parser, list(collect_programs()), dest="command"
        )

    @override
    @staticmethod
    def main(args: argparse.Namespace):
        MainProgram.run_subprogram(args)


def main():
    parser = argparse.ArgumentParser(
        description="A command-line interface for Mow, a tool for training and evaluating models."
    )
    MainProgram.add_arguments(parser)

    # Parse the arguments
    args = parser.parse_args()

    MainProgram().main(args)


if __name__ == "__main__":
    main()
