import torch


def merge_views(*tensors: torch.Tensor) -> torch.Tensor:
    """Merge multiple tensors into a single tensor with a new view.

    Args:
        *tensors (torch.Tensor): Tensors to merge.

    Returns:
        torch.Tensor: Merged tensor.
    """
    if not tensors:
        raise ValueError("At least one tensor must be provided.")

    batch_size = tensors[0].shape[0]
    num_views = len(tensors)
    return torch.stack(tensors, dim=1).reshape(
        batch_size * num_views, *tensors[0].shape[1:]
    )


def split_views(
    tensor: torch.Tensor, num_views: int
) -> tuple[torch.Tensor, ...]:
    """Split a tensor into multiple views.

    Args:
        tensor (torch.Tensor): Tensor to split.
        num_views (int): Number of views to split into.

    Returns:
        list[torch.Tensor]: List of tensors split into views.
    """
    if num_views <= 0:
        raise ValueError("Number of views must be a positive integer.")

    batch_size = tensor.shape[0] // num_views
    if batch_size * num_views != tensor.shape[0]:
        raise ValueError(
            "The tensor cannot be evenly split into the specified number of views."
        )

    ret = tensor.reshape(batch_size, num_views, *tensor.shape[1:])
    return ret.unbind(dim=1)
