from __future__ import annotations

import argparse
from abc import ABC, abstractmethod
from typing import TypeVar

Args = TypeVar("Args", bound=argparse.Namespace)


class Program[Args](ABC):
    def __init_subclass__(
        cls,
        *,
        name: str,
        args: type[Args] = argparse.Namespace,
        help: str | None = None,
    ) -> None:
        cls.args_type = args
        cls.name = name
        cls.help = help
        cls.dest = None
        cls.subprograms = []
        return super().__init_subclass__()

    @staticmethod
    @abstractmethod
    def add_arguments(parser: argparse.ArgumentParser):
        """
        Parses the command-line arguments for the program.

        Args:
            parser (argparse.ArgumentParser): The argument parser to use.
        """
        raise NotImplementedError

    @staticmethod
    @abstractmethod
    def main(args: Args):
        """
        Main entry point for the program.

        Args:
            args (Args): The parsed command-line arguments.
        """
        raise NotImplementedError

    @classmethod
    def build_subprogram(
        cls,
        parser: argparse.ArgumentParser,
        programs: list[type[Program]],
        *,
        dest: str | None = None,
    ) -> argparse.ArgumentParser:
        """
        Builds a subparser for the program.

        Args:
            parser (argparse.ArgumentParser): The argument parser to use.

        Returns:
            argparse.ArgumentParser: The updated argument parser.
        """
        cls.dest = dest or cls.name
        cls.subprograms.extend(programs)

        subparsers = parser.add_subparsers(
            dest=dest, description=cls.help, required=True
        )
        for program in programs:
            subparser = subparsers.add_parser(program.name, help=program.help)
            program.add_arguments(subparser)

        return parser

    @classmethod
    def run_subprogram(cls, args: Args):
        """
        Runs the subprogram with the given arguments.

        Args:
            args (Args): The parsed command-line arguments.
        """
        key = cls.dest or cls.name
        if not hasattr(args, key):
            raise ValueError(f"No subprogram specified for {cls.name}.")
        subprogram_name = getattr(args, key)
        for program in cls.subprograms:
            if program.name == subprogram_name:
                program.main(args)
                return
        raise ValueError(
            f"Unknown subprogram: {subprogram_name} for {cls.name}."
        )
