"""Parser options."""

import argparse


def options():
    """Construct the central argument parser, filled with useful defaults."""
    parser = argparse.ArgumentParser(description='Options for the CLI tool.')

    # Data
    parser.add_argument('--data', type=str, default='~/data', help='location of the data corpus')
    parser.add_argument('--dataset', type=str, default='Blur', help='Dataset name.')  # Cosine
    parser.add_argument('--noise_level', default=0.00, type=float, help='Amount of noise added to data.')

    # Search space:
    parser.add_argument('--operations', default='all-good-learnable', help='Which operations are included in the search?')  # grad
    parser.add_argument('--layers', type=int, default=10, help='How many layers in the network?')  # 8
    parser.add_argument('--cell_steps', type=int, default=1, help="""Use actual cells with multiple connections if cell_steps > 1.
                                                                  In that case layers is then number of cells and the total number
                                                                  of successive ops can be up to layers x cell_steps.""")
    parser.add_argument('--sublayers', type=int, default=2, help='How many sublayers in each operation?')
    parser.add_argument('--channels', type=int, default=64, help='How many channels pass from operation to operation?')
    parser.add_argument('--recover_signal', action='store_true', help="""Network output is signal instead of noise.
                                                                       Only valid if cell_steps>1.""")

    parser.add_argument('--variant', default='DARTS_1', help='Which variant to use for architecture search?')  # DARTS_1
    parser.add_argument('--test_all_variants', action='store_true', help='Run through many hyperparameter variants.')

    # Hyperparameters:
    parser.add_argument('--epochs', type=int, default=None, help='Overwrite number of training epochs.')
    parser.add_argument('--deep_supervision', action='store_true', help='Run with deep supervision.')
    parser.add_argument('--config_path', type=str, default='configs/darts_1_configspace.json',
                        help='location of the hyperparameter configs')
    parser.add_argument('--randomize_init', action='store_true', help='Randomly initialize initialization alphas.')

    # Debugging and determinism:
    parser.add_argument('--seed', type=int, default=None, help='random seed')
    parser.add_argument('--deterministic', action='store_true', help='Disable CUDNN non-determinism.')
    parser.add_argument('--dryrun', action='store_true', help='Run everything for just one step to test functionality.')
    parser.add_argument('--callback', default=None, type=int, help='Print info every callback many epochs.')

    # BOHB specifics
    parser.add_argument('--bohb', action='store_true', help='Run BOHB.')
    parser.add_argument('--save_dir', type=str, default='bohb_logs', help='Where to save BOHB logs')
    parser.add_argument('--param_lr', type=float, default=None, help='Overwrite parameter learning rate.')
    parser.add_argument('--alpha_lr', type=float, default=None, help='Overwrite alpha learning rate.')
    parser.add_argument('--param_weight_decay', type=float, default=None, help='Overwrite parameter weight decay rate.')
    parser.add_argument('--alpha_weight_decay', type=float, default=None, help='Overwrite alpha weight decayrate.')
    parser.add_argument('--alpha_warmup', choices=['True', 'False'], default=None, help='Overwrite  if alpha warmup.')
    parser.add_argument('--param_warmup', choices=['True', 'False'], default=None, help='Overwrite if parameter warm up.')
    parser.add_argument('--alpha_scheduler', choices=['none', 'linear'], default=None, help='Overwrite the alpha scheduler.')
    parser.add_argument('--alpha_optimizer', choices=['GD', 'Adam'], default=None, help='Overwrite the alpha optimizer.')
    parser.add_argument('--grid_search', action='store_true', help='overwrite default parameter.')
    parser.add_argument('--md', action='store_true', help='overwrite default parameter.')

    # Files and Folders:
    parser.add_argument('--exp_name', type=str, default='EXP', help='experiment name')
    return parser
