import sys
sys.path.insert(0, "./../..")

import os, time, glob
import argparse
import numpy as np

import torch
import torch.nn as nn
from torch import optim
import torchvision.datasets as dset
import torch.backends.cudnn as cudnn

from betty.engine import Engine
from betty.config_template import Config, EngineConfig
from betty.problems import ImplicitProblem

from model_search import Network, Architecture
import utils


parser = argparse.ArgumentParser("cifar")
parser.add_argument('--data', type=str, default='../data', help='location of the data corpus')
parser.add_argument('--batchsz', type=int, default=64, help='batch size')
parser.add_argument('--lr', type=float, default=0.025, help='init learning rate')
parser.add_argument('--lr_min', type=float, default=0.001, help='min learning rate')
parser.add_argument('--momentum', type=float, default=0.9, help='momentum')
parser.add_argument('--wd', type=float, default=3e-4, help='weight decay')
parser.add_argument('--report_freq', type=int, default=100, help='report frequency')
parser.add_argument('--gpu', type=int, default=0, help='gpu device id')
parser.add_argument('--epochs', type=int, default=50, help='num of training epochs')
parser.add_argument('--init_ch', type=int, default=16, help='num of init channels')
parser.add_argument('--layers', type=int, default=8, help='total number of layers')
parser.add_argument('--cutout', action='store_true', default=False, help='use cutout')
parser.add_argument('--cutout_len', type=int, default=16, help='cutout length')
parser.add_argument('--drop_path_prob', type=float, default=0.3, help='drop path probability')
parser.add_argument('--train_portion', type=float, default=0.5, help='portion of training/val splitting')
parser.add_argument('--arch_lr', type=float, default=3e-4, help='learning rate for arch encoding')
parser.add_argument('--arch_wd', type=float, default=1e-3, help='weight decay for arch encoding')
parser.add_argument('--arch_steps', type=int, default=4, help='architecture steps')
parser.add_argument('--unroll_steps', type=int, default=1, help='unrolling steps')
args = parser.parse_args()

os.environ['CUDA_VISIBLE_DEVICES'] = str(args.gpu)
device = torch.device('cuda:0')

train_transform, valid_transform = utils.data_transforms_cifar10(args)
train_data = dset.CIFAR10(root=args.data, train=True, download=True, transform=train_transform)
valid_data = dset.CIFAR10(root=args.data, train=False, download=True, transform=valid_transform)

test_queue = torch.utils.data.DataLoader(
    valid_data, batch_size=args.batchsz, shuffle=False, pin_memory=True, num_workers=2)

num_train = len(train_data) # 50000
indices = list(range(num_train))
split = int(np.floor(args.train_portion * num_train))

train_iters = int(args.epochs * (num_train * args.train_portion // args.batchsz + 1) * args.unroll_steps)
train_queue = torch.utils.data.DataLoader(
    train_data,
    batch_size=args.batchsz,
    sampler=torch.utils.data.sampler.SubsetRandomSampler(indices[:split]),
    pin_memory=True,
    num_workers=2
)
criterion = nn.CrossEntropyLoss().to(device)
network = Network(args.init_ch, 10, args.layers, criterion, steps=args.arch_steps).to(device)
net_optimizer = optim.SGD(network.parameters(),
                          lr=args.lr,
                          momentum=args.momentum,
                          weight_decay=args.wd)
net_scheduler = optim.lr_scheduler.CosineAnnealingLR(net_optimizer,
                                                     float(train_iters // args.unroll_steps),
                                                     eta_min=args.lr_min)

valid_queue = torch.utils.data.DataLoader(
    train_data,
    batch_size=args.batchsz,
    sampler=torch.utils.data.sampler.SubsetRandomSampler(indices[split:]),
    pin_memory=True,
    num_workers=2
)
architecture = Architecture(steps=args.arch_steps).to(device)
arch_optimizer = optim.Adam(architecture.parameters(),
                            lr=args.arch_lr,
                            betas=(0.5, 0.999),
                            weight_decay=args.arch_wd)


class Outer(ImplicitProblem):
    def forward(self):
        return self.module()

    def training_step(self, batch):
        x, target = batch
        x, target = x.to(device), target.to(device, non_blocking=True)

        alphas = self.forward()
        loss = self.inner.module.loss(x, alphas, target)

        if self.count % 50 == 0:
            print(f"step {self.count} || loss: {loss.item()}")

        return loss


class Inner(ImplicitProblem):
    def forward(self, x, alphas):
        return self.module(x, alphas)

    def training_step(self, batch):
        x, target = batch
        x, target = x.to(device), target.to(device, non_blocking=True)

        alphas = self.outer()
        loss = self.module.loss(x, alphas, target)

        return loss


class NASEngine(Engine):
    @torch.no_grad()
    def validation(self):
        corrects = 0
        total = 0
        for x, target in test_queue:
            x, target = x.to(device), target.to(device, non_blocking=True)
            alphas = self.outer()
            _, correct = self.inner.module.loss(x, alphas, target, acc=True)
            corrects += correct
            total += x.size(0)
        acc = corrects / total

        print('[*] Valid Acc.:', acc)
        alphas = self.outer()
        torch.save({'genotype': self.inner.module.genotype(alphas)}, 'genotype.t7')

    def train_step(self):
        for leaf in self.leaves:
            leaf.step(param_update=False)


outer_config = Config(type='darts', step=args.unroll_steps, retain_graph=True, first_order=True)
inner_config = Config(type='torch')
engine_config = EngineConfig(valid_step=args.report_freq*args.unroll_steps, train_iters=train_iters)
outer = Outer(name='outer',
              config=outer_config,
              module=architecture,
              optimizer=arch_optimizer,
              train_data_loader=valid_queue,
              device=device)
inner = Inner(name='inner',
              config=inner_config,
              module=network,
              optimizer=net_optimizer,
              scheduler=net_scheduler,
              train_data_loader=train_queue,
              device=device)

problems = [outer, inner]
l2h = {inner: [outer]}
h2l = {outer: [inner]}
dependencies = {'l2h': l2h, 'h2l': h2l}

engine = NASEngine(config=engine_config, problems=problems, dependencies=dependencies)
engine.run()
