# coding=utf-8
# Copyright 2022 EleutherAI and the HuggingFace Inc. team. All rights reserved.
#
# This code is based on EleutherAI's GPT-NeoX library and the GPT-NeoX
# and OPT implementations in this library. It has been modified from its
# original forms to accommodate minor architectural differences compared
# to GPT-NeoX and OPT used by the Meta AI team that trained the model.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import os
from typing import Callable, List, Literal, Optional, Tuple, Union

import torch
import torch.nn.functional as F
import torch.utils.checkpoint
import triton
from torch import nn
from torch.nn.functional import scaled_dot_product_attention as sdpa
from transformers.activations import ACT2FN
from transformers.cache_utils import Cache, DynamicCache, StaticCache
from transformers.generation import GenerationMixin
from transformers.modeling_attn_mask_utils import AttentionMaskConverter
from transformers.modeling_flash_attention_utils import FlashAttentionKwargs
from transformers.modeling_outputs import (BaseModelOutputWithPast,
                                           CausalLMOutputWithPast,
                                           QuestionAnsweringModelOutput,
                                           SequenceClassifierOutputWithPast,
                                           TokenClassifierOutput)
from transformers.modeling_rope_utils import ROPE_INIT_FUNCTIONS
from transformers.modeling_utils import (ALL_ATTENTION_FUNCTIONS,
                                         PreTrainedModel)
from transformers.models.llama.configuration_llama import LlamaConfig
from transformers.processing_utils import Unpack
from transformers.pytorch_utils import ALL_LAYERNORM_LAYERS
from transformers.utils import (LossKwargs, add_code_sample_docstrings,
                                add_start_docstrings,
                                add_start_docstrings_to_model_forward,
                                is_torch_flex_attn_available, logging,
                                replace_return_docstrings)
from transformers.utils.deprecation import deprecate_kwarg

from hip_research.utils.triton_recompute import attention as recomp_attn

logger = logging.get_logger(__name__)

_CHECKPOINT_FOR_DOC = "meta-llama/Llama-2-7b-hf"
_CONFIG_FOR_DOC = "LlamaConfig"


class LlamaRMSNorm(nn.Module):
    def __init__(self, hidden_size, eps=1e-6):
        """
        LlamaRMSNorm is equivalent to T5LayerNorm
        """
        super().__init__()
        self.weight = nn.Parameter(torch.ones(hidden_size))
        self.variance_epsilon = eps

    def forward(self, hidden_states):
        input_dtype = hidden_states.dtype
        hidden_states = hidden_states.to(torch.float32)
        variance = hidden_states.pow(2).mean(-1, keepdim=True)
        hidden_states = hidden_states * torch.rsqrt(variance + self.variance_epsilon)
        return self.weight.to(input_dtype) * hidden_states.to(input_dtype)

    def extra_repr(self):
        return f"{tuple(self.weight.shape)}, eps={self.variance_epsilon}"


ALL_LAYERNORM_LAYERS.append(LlamaRMSNorm)


class LlamaRotaryEmbedding(nn.Module):
    def __init__(self, config: LlamaConfig, device=None):
        super().__init__()
        # BC: "rope_type" was originally "type"
        if hasattr(config, "rope_scaling") and config.rope_scaling is not None:
            self.rope_type = config.rope_scaling.get(
                "rope_type", config.rope_scaling.get("type")
            )
        else:
            self.rope_type = "default"
        self.max_seq_len_cached = config.max_position_embeddings
        self.original_max_seq_len = config.max_position_embeddings

        self.config = config
        self.rope_init_fn = ROPE_INIT_FUNCTIONS[self.rope_type]

        inv_freq, self.attention_scaling = self.rope_init_fn(self.config, device)
        self.register_buffer("inv_freq", inv_freq, persistent=False)
        self.original_inv_freq = self.inv_freq

    def _dynamic_frequency_update(self, position_ids, device):
        """
        dynamic RoPE layers should recompute `inv_freq` in the following situations:
        1 - growing beyond the cached sequence length (allow scaling)
        2 - the current sequence length is in the original scale (avoid losing precision with small sequences)
        """
        seq_len = torch.max(position_ids) + 1
        if seq_len > self.max_seq_len_cached:  # growth
            inv_freq, self.attention_scaling = self.rope_init_fn(
                self.config, device, seq_len=seq_len
            )
            self.register_buffer(
                "inv_freq", inv_freq, persistent=False
            )  # TODO joao: may break with compilation
            self.max_seq_len_cached = seq_len

        if (
            seq_len < self.original_max_seq_len
            and self.max_seq_len_cached > self.original_max_seq_len
        ):  # reset
            # This .to() is needed if the model has been moved to a device after being initialized (because
            # the buffer is automatically moved, but not the original copy)
            self.original_inv_freq = self.original_inv_freq.to(device)
            self.register_buffer("inv_freq", self.original_inv_freq, persistent=False)
            self.max_seq_len_cached = self.original_max_seq_len

    @torch.no_grad()
    def forward(self, x, position_ids):
        if "dynamic" in self.rope_type:
            self._dynamic_frequency_update(position_ids, device=x.device)

        # Core RoPE block
        inv_freq_expanded = (
            self.inv_freq[None, :, None].float().expand(position_ids.shape[0], -1, 1)
        )
        position_ids_expanded = position_ids[:, None, :].float()
        # Force float32 (see https://github.com/huggingface/transformers/pull/29285)
        device_type = x.device.type
        device_type = (
            device_type
            if isinstance(device_type, str) and device_type != "mps"
            else "cpu"
        )
        with torch.autocast(device_type=device_type, enabled=False):
            freqs = (
                inv_freq_expanded.float().to(x.device) @ position_ids_expanded.float()
            ).transpose(1, 2)
            emb = torch.cat((freqs, freqs), dim=-1)
            cos = emb.cos()
            sin = emb.sin()

        # Advanced RoPE types (e.g. yarn) apply a post-processing scaling factor, equivalent to scaling attention
        cos = cos * self.attention_scaling
        sin = sin * self.attention_scaling

        return cos.to(dtype=x.dtype), sin.to(dtype=x.dtype)


def rotate_half(x):
    """Rotates half the hidden dims of the input."""
    x1 = x[..., : x.shape[-1] // 2]
    x2 = x[..., x.shape[-1] // 2 :]
    return torch.cat((-x2, x1), dim=-1)


def apply_rotary_pos_emb(q, k, cos, sin, position_ids=None, unsqueeze_dim=1):
    """Applies Rotary Position Embedding to the query and key tensors.

    Args:
        q (`torch.Tensor`): The query tensor.
        k (`torch.Tensor`): The key tensor.
        cos (`torch.Tensor`): The cosine part of the rotary embedding.
        sin (`torch.Tensor`): The sine part of the rotary embedding.
        position_ids (`torch.Tensor`, *optional*):
            Deprecated and unused.
        unsqueeze_dim (`int`, *optional*, defaults to 1):
            The 'unsqueeze_dim' argument specifies the dimension along which to unsqueeze cos[position_ids] and
            sin[position_ids] so that they can be properly broadcasted to the dimensions of q and k. For example, note
            that cos[position_ids] and sin[position_ids] have the shape [batch_size, seq_len, head_dim]. Then, if q and
            k have the shape [batch_size, heads, seq_len, head_dim], then setting unsqueeze_dim=1 makes
            cos[position_ids] and sin[position_ids] broadcastable to the shapes of q and k. Similarly, if q and k have
            the shape [batch_size, seq_len, heads, head_dim], then set unsqueeze_dim=2.
    Returns:
        `tuple(torch.Tensor)` comprising of the query and key tensors rotated using the Rotary Position Embedding.
    """
    cos = cos.unsqueeze(unsqueeze_dim)
    sin = sin.unsqueeze(unsqueeze_dim)
    q_embed = (q * cos) + (rotate_half(q) * sin)
    k_embed = (k * cos) + (rotate_half(k) * sin)
    return q_embed, k_embed


class LlamaMLP(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.config = config
        self.hidden_size = config.hidden_size
        self.intermediate_size = config.intermediate_size
        self.gate_proj = nn.Linear(
            self.hidden_size, self.intermediate_size, bias=config.mlp_bias
        )
        self.up_proj = nn.Linear(
            self.hidden_size, self.intermediate_size, bias=config.mlp_bias
        )
        self.down_proj = nn.Linear(
            self.intermediate_size, self.hidden_size, bias=config.mlp_bias
        )
        self.act_fn = ACT2FN[config.hidden_act]

    def forward(self, x):
        down_proj = self.down_proj(self.act_fn(self.gate_proj(x)) * self.up_proj(x))
        assert down_proj.dtype == x.dtype
        return down_proj


def repeat_kv(hidden_states: torch.Tensor, n_rep: int) -> torch.Tensor:
    """
    This is the equivalent of torch.repeat_interleave(x, dim=1, repeats=n_rep). The hidden states go from (batch,
    num_key_value_heads, seqlen, head_dim) to (batch, num_attention_heads, seqlen, head_dim)
    """
    batch, num_key_value_heads, slen, head_dim = hidden_states.shape
    if n_rep == 1:
        return hidden_states
    hidden_states = hidden_states[:, :, None, :, :].expand(
        batch, num_key_value_heads, n_rep, slen, head_dim
    )
    return hidden_states.reshape(batch, num_key_value_heads * n_rep, slen, head_dim)


def eager_attention_forward(
    module: nn.Module,
    query: torch.Tensor,
    key: torch.Tensor,
    value: torch.Tensor,
    attention_mask: Optional[torch.Tensor],
    scaling: float,
    dropout: float = 0.0,
    **kwargs,
):
    key_states = repeat_kv(key, module.num_key_value_groups)
    value_states = repeat_kv(value, module.num_key_value_groups)

    attn_weights = torch.matmul(query, key_states.transpose(2, 3)) * scaling
    if attention_mask is not None:
        causal_mask = attention_mask[:, :, :, : key_states.shape[-2]]
        attn_weights = attn_weights + causal_mask

    attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32).to(
        query.dtype
    )
    attn_weights = nn.functional.dropout(
        attn_weights, p=dropout, training=module.training
    )
    attn_output = torch.matmul(attn_weights, value_states)
    attn_output = attn_output.transpose(1, 2).contiguous()

    return attn_output, attn_weights


class SimpleTokenPooler(torch.nn.Module):
    def __init__(self):
        super().__init__()

    def forward(
        self,
        x: torch.Tensor,
        dim: int,
        i_stage: int,
        tensor_type: Literal["query", "key", "value"],
    ):
        assert tensor_type in ["query", "key", "value"]

        # x_pooled = x.mean(dim=dim) # 23.774122

        # x_pooled = (x.amax(dim=dim) + x.amin(dim=dim)) / 2

        x_pooled = x.median(dim=dim).values  # 23.764080

        return x_pooled


class SimpleOutputUnpooler(torch.nn.Module):
    def __init__(self, dim=2):
        super().__init__()
        self.pooling_dim = dim

    def forward(self, x: torch.Tensor, rate: int):
        return x.repeat_interleave(rate, dim=self.pooling_dim)


class DummyGateProbEstimator(nn.Module):
    def __init__(self, config: LlamaConfig):
        super().__init__()

        self.config = config

        num_stages = 3
        t = (
            torch.randn(
                # NOTE: [BSZ, N_Q, HEAD, N_GATES]
                (
                    1,
                    1,
                    self.config.num_attention_heads,
                    num_stages + 1,
                ),
            )
            - 10.0
        )
        t[..., -1] = 10

        self.gate_probs = nn.Parameter(t)

    def forward(self, query: torch.Tensor):
        return self.gate_probs.sigmoid().to(query.dtype)


class LoraAdopter(torch.nn.Module):
    def __init__(self, ch: int, lora_r: int):
        super().__init__()

        # FIXME: init lora well
        self.dropout = nn.Dropout(0.15)
        self.lora_a = nn.Linear(ch, lora_r, bias=False).weight
        self.lora_b = nn.Linear(lora_r, ch, bias=False).weight
        self.scaling = (lora_r // 2) / lora_r

        # print(self.lora_a, self.lora_a.dtype)
        # assert not torch.any(torch.isnan(self.lora_a))

    def _init_weights(self):
        nn.init.kaiming_uniform_(self.lora_a, a=5**0.5)
        # nn.init.normal_(self.lora_a)
        # self.lora_a.data.normal_()
        nn.init.zeros_(self.lora_b)

    def forward(self, x: torch.Tensor):
        assert x.dtype == torch.bfloat16
        # assert not torch.any(torch.isnan(x))
        # assert not torch.any(torch.isnan(self.lora_a))
        # assert not torch.any(torch.isnan(self.lora_b))
        y = self.dropout(x)
        y = F.linear(y, self.lora_a)
        # assert not torch.any(torch.isnan(y))
        # print(self.lora_a)
        # print(self.lora_b)
        # print(self.scaling)
        y = F.linear(y, self.lora_b) * self.scaling
        # assert not torch.any(torch.isnan(y))
        return x + y


class LoraTokenPooler(torch.nn.Module):
    def __init__(self, config: LlamaConfig, lora_r: int):
        super().__init__()

        self.lora_r = lora_r
        self.num_stages = 3
        self.pool_func = config.pooler_config["pool_func"]

        adopters = {}

        for i in range(self.num_stages):
            for name in ["query", "key", "value"]:
                if name == "query":
                    ch = config.hidden_size
                else:
                    ch = config.head_dim * config.num_key_value_heads
                layer_pre = LoraAdopter(ch, self.lora_r)
                layer_post = LoraAdopter(ch, self.lora_r)
                adopters[f"{name}_{i}_pre"] = layer_pre
                adopters[f"{name}_{i}_post"] = layer_post

        self.adopters = nn.ModuleDict(adopters)

    def pool(self, x, dim):
        if self.pool_func == "mean":
            return x.mean(dim=dim)
        elif self.pool_func == "max":
            return x.amax(dim=dim)
        else:
            raise NotImplementedError(f"{self.pool_func=} not implemented")

    def forward(
        self,
        x: torch.Tensor,
        dim: int,
        i_stage: int,
        tensor_type: Literal["query", "key", "value"],
    ):
        assert tensor_type in ["query", "key", "value"]
        x_shape = x.shape
        x = x.flatten(-2, -1)
        x = self.adopters[f"{tensor_type}_{i_stage}_pre"](x)
        x = self.pool(x, dim)
        x = self.adopters[f"{tensor_type}_{i_stage}_post"](x)
        x = x.view(x.shape[:-1] + x_shape[-2:])
        return x


class LoraOutputUnpooler(torch.nn.Module):
    def __init__(self, config: LlamaConfig, lora_r: int, dim=2):
        super().__init__()
        self.pooling_dim = dim

        self.lora_r = lora_r
        self.adopter = LoraAdopter(config.hidden_size, self.lora_r)

    def forward(self, x: torch.Tensor, rate: int):
        x_shape = x.shape
        x = x.flatten(-2, -1)
        x = self.adopter(x)
        x = x.view(x.shape[:-1] + x_shape[-2:])
        return x.repeat_interleave(rate, dim=self.pooling_dim)


class QueryDependentGateProbEstimator(nn.Module):
    def __init__(self, config: LlamaConfig):
        super().__init__()

        self.config = config

        self.num_heads = self.config.num_attention_heads
        self.num_stages = 3
        t = (
            torch.randn(
                # NOTE: [BSZ, N_Q, HEAD, N_GATES]
                (
                    1,
                    1,
                    self.config.num_attention_heads,
                    self.num_stages + 1,
                ),
            )
            - 4.0
        )
        t[..., -1] = 4

        self.gate_prob_weight = nn.Linear(
            self.config.hidden_size,
            self.num_heads * (self.num_stages + 1),
            bias=False,
        )
        self.gate_prob_bias = nn.Parameter(t)

    def forward(self, query: torch.Tensor):
        BSZ, TDST, HEAD, HID = query.shape

        query = query.view(BSZ, TDST, HEAD * HID)
        probs = self.gate_prob_weight(query)
        probs = probs.view(BSZ, TDST, HEAD, -1) + self.gate_prob_bias

        return probs.sigmoid().to(query.dtype)


class LlamaAttention(nn.Module):
    """Multi-headed attention from 'Attention Is All You Need' paper"""

    def __init__(self, config: LlamaConfig, layer_idx: int):
        super().__init__()
        self.config = config
        self.layer_idx = layer_idx
        self.head_dim = getattr(
            config, "head_dim", config.hidden_size // config.num_attention_heads
        )
        self.num_key_value_groups = (
            config.num_attention_heads // config.num_key_value_heads
        )
        self.scaling = self.head_dim**-0.5
        self.attention_dropout = config.attention_dropout
        self.is_causal = True

        self.q_proj = nn.Linear(
            config.hidden_size,
            config.num_attention_heads * self.head_dim,
            bias=config.attention_bias,
        )
        self.k_proj = nn.Linear(
            config.hidden_size,
            config.num_key_value_heads * self.head_dim,
            bias=config.attention_bias,
        )
        self.v_proj = nn.Linear(
            config.hidden_size,
            config.num_key_value_heads * self.head_dim,
            bias=config.attention_bias,
        )
        self.o_proj = nn.Linear(
            config.num_attention_heads * self.head_dim,
            config.hidden_size,
            bias=config.attention_bias,
        )

        from hip_attn.v1_3.metadata import HiPAttentionArgs

        self.hip_attn_args: HiPAttentionArgs = None
        self.force_sliding_window: bool = False

        if config.pooler_method == "dummy":
            self.token_pooler_fn = SimpleTokenPooler()
            self.output_unpooler_fn = SimpleOutputUnpooler()
            self.gate_prob_estimater = DummyGateProbEstimator(config=self.config)
        elif config.pooler_method == "lora":
            self.token_pooler_fn = LoraTokenPooler(
                config=self.config,
                lora_r=config.pooler_config["lora_r"],
            )
            self.output_unpooler_fn = LoraOutputUnpooler(
                config=self.config,
                lora_r=config.pooler_config["lora_r"],
            )
            self.gate_prob_estimater = QueryDependentGateProbEstimator(
                config=self.config,
            )
        else:
            raise Exception(f"unknown pooler method {config.pooler_method=}")

    def is_need_apply_rope(self):
        return (
            (self.hip_attn_args is not None)
            and (not self.hip_attn_args.need_apply_rope)
        ) or (
            self.config._attn_implementation
            not in [
                "hip_attention",
                # "sdpa_rectangle",
                "recompute",
                # "recompute_dense",
                "dense",
            ]
        )

    def hip_attention_interface(
        self,
        module: torch.nn.Module,
        query: torch.Tensor,
        key: torch.Tensor,
        value: torch.Tensor,
        attention_mask: Optional[torch.Tensor],
        dropout: float = 0.0,
        scaling: Optional[float] = None,
        sliding_window: Optional[int] = None,
        softcap: Optional[float] = None,
        position_ids: Optional[torch.Tensor] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        rope_cos: Optional[torch.Tensor] = None,
        rope_sin: Optional[torch.Tensor] = None,
    ):
        original_dtype = query.dtype
        if query.dtype == torch.float32:
            query = query.to(torch.bfloat16)
            key = key.to(torch.bfloat16)
            value = value.to(torch.bfloat16)
            rope_sin = rope_sin.to(torch.bfloat16)
            rope_cos = rope_cos.to(torch.bfloat16)

        assert query.dtype in [torch.bfloat16, torch.float16], query.dtype
        assert self.hip_attn_args is not None

        query = query.permute(0, 2, 1, 3).contiguous()
        key = key.permute(0, 2, 1, 3).contiguous()
        value = value.permute(0, 2, 1, 3).contiguous()

        args = self.hip_attn_args.clone()
        if (position_ids is not None) and (position_ids.shape[0] < query.shape[0]):
            position_ids = position_ids.repeat(query.shape[0], 1)
        args.position_ids = position_ids
        args.sm_scale = scaling
        args.rope_cos = rope_cos
        args.rope_sin = rope_sin

        if isinstance(sliding_window, int) or self.force_sliding_window:
            from hip_attn.v1_3.kernels.block_sparse_attention import \
                block_sparse_attention

            args = args.clone()
            args.block_size_q = args.block_sparse_block_size_q
            args.block_size_k = args.stages[-1].stage_chunk_size
            args.second_stage_k = 0
            args.sink_token_size = 1024
            args.sliding_window_size = (
                sliding_window if sliding_window is not None else 1024
            )
            args.sliding_window_indices = None
            assert args.sliding_window_size > 0

            BSZ, TDST, HEAD, HID = query.shape
            _, TSRC, HEAD_KV, _ = key.shape
            assert key.shape == value.shape
            BDST = triton.cdiv(TDST, args.block_size_q)
            BH = BSZ * HEAD

            indices = torch.zeros((BH, BDST, 0), dtype=torch.int64, device=query.device)
            ks = torch.zeros((BH, BDST), dtype=torch.int64, device=query.device)
            ks_count = ks.unsqueeze(-1)
            ks_start_end = torch.zeros(
                (BH, BDST, 2), dtype=torch.int64, device=query.device
            )

            output, _scores = block_sparse_attention(
                q=query,
                k=key,
                v=value,
                seq_lens=args.position_ids + 1,
                indices=indices,
                ks=ks,
                ks_count=ks_count,
                ks_start_end=ks_start_end,
                access_counter=None,
                cache_miss_counter=None,
                extend_backend=args.sa_extend_backend,
                model_context_length=self.config.max_position_embeddings,
                extend_context_length=self.config.max_position_embeddings * 10,
                offload_update_cache=False,
                return_attention_scores=False,
                output_attention_scores_reduce_method="max",
                args=args,
            )
        elif self.hip_attn_args.enable_hip_tune:
            assert False
            from hip_attn.v1_3.attention_tune import hip_tune_attention

            assert sliding_window is None, f"{sliding_window=}"
            assert softcap is None, f"{softcap=}"
            assert not output_attentions, f"{output_attentions=}"

            args = args.clone()

            args.token_pooler_fn = self.token_pooler_fn
            args.output_unpooler_fn = self.output_unpooler_fn
            args.gate_probs = self.gate_prob_estimater(query).to(query.dtype)
            if args.disable_gate_prob:
                args.gate_probs.fill_(0)
                args.gate_probs[..., -1].fill_(1)

            assert query.dtype in [torch.float16, torch.bfloat16]

            output, _metadata = hip_tune_attention(
                q=query,
                k=key,
                v=value,
                args=args,
            )

            assert output.dtype == query.dtype
        else:
            use_v12 = args.fallback_to_v12
            if not use_v12:
                from hip_attn.v1_3.attention import hip_attention
            else:
                from hip_attn.v1_2.attention_extend import \
                    dual_stage_quadratic_hip_attention as hip_attention

            assert sliding_window is None, f"{sliding_window=}"
            assert softcap is None, f"{softcap=}"
            assert not output_attentions, f"{output_attentions=}"

            args = args.clone()
            args.enable_hip_tune = False

            assert not args.using_extend
            assert not args.need_apply_rope
            assert use_v12
            
            chunk_size = 32768
            if query.shape[1] < chunk_size:
                output, _metadata = hip_attention(
                    q=((query * args.sm_scale).to(query.dtype) if use_v12 else query),
                    k=key,
                    v=value,
                    args=args,
                )
            else:
                outputs = []
                for i_start in range(0, query.shape[1], chunk_size):
                    i_end = i_start + chunk_size
                    
                    args_chunk = args.clone()
                    query_chunk = query[:, i_start:i_end].contiguous()
                    args_chunk.position_ids = args_chunk.position_ids[:, i_start:i_end].contiguous()
                    
                    output, _metadata = hip_attention(
                        q=((query_chunk * args_chunk.sm_scale).to(query.dtype) if use_v12 else query_chunk),
                        k=key,
                        v=value,
                        args=args_chunk,
                    )
                    outputs.append(output)
                output = torch.cat(outputs, dim=1)

        if original_dtype != output.dtype:
            output = output.to(original_dtype)

        return output, None

    def sdpa_rectangle_forward(
        self,
        module: nn.Module,
        query_states: torch.Tensor,
        key_states: torch.Tensor,
        value_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        dropout: float = 0.0,
        scaling: float = 1.0,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:

        b, _, s, d = query_states.size()

        if not module.is_need_apply_rope():
            cos, sin = kwargs["cos"], kwargs["sin"]
            cos = cos.view(1, 1, cos.shape[-2], cos.shape[-1])
            sin = sin.view(1, 1, sin.shape[-2], sin.shape[-1])

            key_states = (key_states * cos[..., : key_states.shape[-2], :]) + (
                rotate_half(key_states) * sin[..., : key_states.shape[-2], :]
            )

            query_states = (query_states * cos[..., -s:, :]) + (
                rotate_half(query_states) * sin[..., -s:, :]
            )

        key_states = repeat_kv(key_states, self.num_key_value_groups)
        value_states = repeat_kv(value_states, self.num_key_value_groups)

        _, _, k, _ = key_states.size()
        mask = torch.ones(b, 1, s, k, device=query_states.device, dtype=torch.bool)
        mask = mask.tril(k - s)

        attn_output = sdpa(
            query_states,
            key_states,
            value_states,
            attn_mask=mask,
            is_causal=False,
            scale=scaling,
            dropout_p=0.0,
        )
        attn_output = attn_output.transpose(1, 2).contiguous()
        return attn_output, None

    def recompute_forward(
        self,
        module: nn.Module,
        query_states: torch.Tensor,
        key_states: torch.Tensor,
        value_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        dropout: float = 0.0,
        scaling: float = 1.0,
        output_attentions: bool = False,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:

        cos, sin = kwargs["cos"], kwargs["sin"]
        idx = kwargs["recompute_idx"]

        if not module.is_need_apply_rope():
            cos = cos.view(1, 1, cos.shape[-2], cos.shape[-1])
            sin = sin.view(1, 1, sin.shape[-2], sin.shape[-1])

            key_states = (key_states * cos[..., : key_states.shape[-2], :]) + (
                rotate_half(key_states) * sin[..., : key_states.shape[-2], :]
            )

            cos = torch.index_select(cos, 2, idx)
            sin = torch.index_select(sin, 2, idx)
            query_states = (query_states * cos) + (rotate_half(query_states) * sin)

        key_states = repeat_kv(key_states, module.num_key_value_groups)
        value_states = repeat_kv(value_states, module.num_key_value_groups)

        if not output_attentions:
            attn_output = recomp_attn(
                query_states, key_states, value_states, idx.unsqueeze(0), scaling
            )
            attn_output = attn_output.transpose(1, 2).contiguous()
            attn_score = None
        else:
            # torch.Size([1, 32, 1024, 64]) torch.Size([1, 32, 76131, 64]) torch.Size([1, 32, 76131, 64]) torch.Size([1024])
            
            chunk_size = 128
            
            attn_outputs = []
            attn_scores = []
            
            for i_tdst in range(0, query_states.shape[2], chunk_size):
                i_tdst_end = i_tdst + chunk_size
                scores = (
                    torch.matmul(
                        query_states[:, :, i_tdst:i_tdst_end], 
                        key_states.permute(0, 1, 3, 2)
                    )
                    * scaling
                ).to(query_states.dtype)
                scores = torch.where(
                    idx[None, None, i_tdst:i_tdst_end, None]
                    >= torch.arange(0, key_states.shape[-2], device=query_states.device)[None, None, None, :],
                    scores,
                    float('-inf'),
                )
                
                attn_scores.append(scores.cpu())
                
                scores = torch.softmax(scores, dim=-1, dtype=torch.float32).to(query_states.dtype)
                attn_output = scores @ value_states
                
                attn_outputs.append(attn_output)
            
            attn_output = torch.cat(attn_outputs, dim=2).permute(0, 2, 1, 3).contiguous()
            attn_score = torch.cat(attn_scores, dim=2)

        return attn_output, attn_score

    def dense_forward(
        self,
        module: nn.Module,
        query_states: torch.Tensor,
        key_states: torch.Tensor,
        value_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        dropout: float = 0.0,
        scaling: float = 1.0,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:

        cos, sin = kwargs["cos"], kwargs["sin"]

        if not module.is_need_apply_rope():
            cos = cos.view(1, 1, cos.shape[-2], cos.shape[-1])
            sin = sin.view(1, 1, sin.shape[-2], sin.shape[-1])

            key_states = (key_states * cos[..., : key_states.shape[-2], :]) + (
                rotate_half(key_states) * sin[..., : key_states.shape[-2], :]
            )

            query_states = (query_states * cos[..., -query_states.shape[-2] :, :]) + (
                rotate_half(query_states) * sin[..., -query_states.shape[-2] :, :]
            )

        key_states = repeat_kv(key_states, module.num_key_value_groups)
        value_states = repeat_kv(value_states, module.num_key_value_groups)

        attn_weights = torch.matmul(query_states, key_states.transpose(2, 3)) * scaling

        b, _, k, _ = key_states.size()
        _, _, s, _ = query_states.size()
        mask = torch.full(
            (b, 1, s, k),
            torch.finfo(query_states.dtype).min,
            device=query_states.device,
            dtype=torch.bool,
        )
        mask = mask.triu(k - s + 1)

        attn_weights = nn.functional.softmax(
            attn_weights + mask, dim=-1, dtype=torch.float32
        ).to(query_states.dtype)

        attn_output = torch.matmul(attn_weights, value_states)
        attn_output = attn_output.transpose(1, 2).contiguous()

        return attn_output, attn_weights.sum(dim=(2, 1))

    def recompute_dense_forward(
        self,
        module: nn.Module,
        query_states: torch.Tensor,
        key_states: torch.Tensor,
        value_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        dropout: float = 0.0,
        scaling: float = 1.0,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:

        # hip forward
        assert not self.hip_attn_args.need_apply_rope

        # postfix_recompute_dense-window_[size:int]-diff_[1/0]-w_[size:int]
        postfix_string = os.environ.get("USE_ATTN_POSTFIX", "0")
        window = int(postfix_string.split("-")[1].split("_")[1])
        diff = int(postfix_string.split("-")[2].split("_")[1])
        w = int(postfix_string.split("-")[3].split("_")[1])
        if window == 0:
            window = None

        b, h, s, d = query_states.size()
        assert h == 32
        sparse_attn_output, _ = self.hip_attention_interface(
            self,
            query_states,
            key_states,
            value_states,
            attention_mask,
            dropout=dropout,
            scaling=scaling,
            sliding_window=window,
            **kwargs,
        )

        # gather the necessary indices from queries and hip out to calculate diff
        cut_n = s % w + max(128, w)
        s_p = s - cut_n

        if "JUST_RETURN" in postfix_string:
            # ======================================================
            # save dense out; save sparse out; use for plotting differences later
            if os.environ.get("SAVE_TENSORS_FOR_PLOTTING", "0") == "1":
                os.makedirs("./saved_tensors", exist_ok=True)

                postfix = os.environ.get("USE_ATTN_POSTFIX")
                if not os.path.exists(
                    f"./saved_tensors/{postfix}-layer-{self.layer_idx}.pt"
                ):
                    print(f"saving")
                    torch.save(
                        sparse_attn_output[:, :s_p].reshape(b, s_p // w, w, h, d),
                        f"./saved_tensors/{postfix}-layer-{self.layer_idx}.pt",
                    )
                    for name, tns in zip(
                        ("q", "k", "v"), (query_states, key_states, value_states)
                    ):
                        torch.save(
                            tns,
                            f"./saved_tensors/{postfix}-layer-{self.layer_idx}-{name}.pt",
                        )
            # ======================================================
            return sparse_attn_output, None

        sparse_attn_output = sparse_attn_output[:, :s_p]
        if cut_n > 0:
            idx = torch.arange(0, s_p, step=w, device=query_states.device)
            sparse_out_diff_states = sparse_attn_output[:, idx]

            idx = torch.cat((idx, torch.arange(s_p, s, device=query_states.device)))
            recomp_query_states = query_states[:, :, idx]

        # do the strided dense attention
        # key_states = repeat_kv(key_states, module.num_key_value_groups)
        # value_states = repeat_kv(value_states, module.num_key_value_groups)
        
        dense_attn_output = recomp_attn(
            recomp_query_states, key_states, value_states, idx.unsqueeze(0), scaling
        )
        
        dense_attn_output = dense_attn_output.transpose(1, 2)

        if diff == 0:
            # we are not doing the diff method, just scatter the outputs and return
            out = torch.zeros_like(query_states).transpose(1, 2)
            out[:, :s_p] = sparse_attn_output
            out[:, idx] = dense_attn_output

            # ======================================================
            # save dense out; save sparse out; use for plotting differences later
            if os.environ.get("SAVE_TENSORS_FOR_PLOTTING", "0") == "1":
                os.makedirs("./saved_tensors", exist_ok=True)

                postfix = os.environ.get("USE_ATTN_POSTFIX")
                if not os.path.exists(
                    f"./saved_tensors/{postfix}-layer-{self.layer_idx}.pt"
                ):
                    print(f"saving")
                    torch.save(
                        out[:, :s_p].reshape(b, s_p // w, w, h, d),
                        f"./saved_tensors/{postfix}-layer-{self.layer_idx}.pt",
                    )
                    for name, tns in zip(
                        ("q", "k", "v"), (query_states, key_states, value_states)
                    ):
                        torch.save(
                            tns,
                            f"./saved_tensors/{postfix}-layer-{self.layer_idx}-{name}.pt",
                        )
            # ======================================================
            return out, None

        # save last states for concat later
        dense_attn_output, last_states = (
            dense_attn_output[:, :-cut_n],
            dense_attn_output[:, -cut_n:],
        )

        # take difference
        dense_sparse_diff = dense_attn_output - sparse_out_diff_states

        dense_sparse_diff = dense_sparse_diff.reshape(b, s_p // w, 1, h, d)
        sparse_attn_output = sparse_attn_output.reshape(b, s_p // w, w, h, d)

        # ======================================================
        # save dense out; save sparse out; use for plotting differences later
        if os.environ.get("SAVE_TENSORS_FOR_PLOTTING", "0") == "1":
            os.makedirs("./saved_tensors", exist_ok=True)

            postfix = os.environ.get("USE_ATTN_POSTFIX")
            if not os.path.exists(
                f"./saved_tensors/{postfix}-layer-{self.layer_idx}.pt"
            ):
                print(f"saving")
                if "flash_attn_plot" in postfix:
                    full_dense_out, _ = ALL_ATTENTION_FUNCTIONS["flash_attention_2"](
                        self,
                        query_states[:, :, :-cut_n],
                        key_states[:, :, :-cut_n],
                        value_states[:, :, :-cut_n],
                        attention_mask,
                        dropout=dropout,
                        scaling=scaling,
                    )
                    full_dense_out = full_dense_out.reshape(b, s_p // w, w, h, d)
                    print(f"{full_dense_out.size()=}")

                    # return pure flash attention out in this case for plotting purposes
                    torch.save(
                        full_dense_out,
                        f"./saved_tensors/{postfix}-layer-{self.layer_idx}.pt",
                    )
                    for name, tns in zip(
                        ("q", "k", "v"), (query_states, key_states, value_states)
                    ):
                        torch.save(
                            tns,
                            f"./saved_tensors/{postfix}-layer-{self.layer_idx}-{name}.pt",
                        )

                    out = torch.cat(
                        (full_dense_out.reshape(b, s_p, h, d), last_states), dim=1
                    )
                    return out, None
                else:
                    torch.save(
                        dense_sparse_diff + sparse_attn_output,
                        f"./saved_tensors/{postfix}-layer-{self.layer_idx}.pt",
                    )
                    for name, tns in zip(
                        ("q", "k", "v"), (query_states, key_states, value_states)
                    ):
                        torch.save(
                            tns,
                            f"./saved_tensors/{postfix}-layer-{self.layer_idx}-{name}.pt",
                        )
        # ======================================================

        sparse_attn_output = (dense_sparse_diff + sparse_attn_output).reshape(
            b, s_p, h, d
        )

        window_attn_output = torch.cat((sparse_attn_output, last_states), dim=1)

        return window_attn_output, None

    def recompute_dense_forward_decode(
        self,
        module: nn.Module,
        query_states: torch.Tensor,
        key_states: torch.Tensor,
        value_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        dropout: float = 0.0,
        scaling: float = 1.0,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:

        if not hasattr(self, "decode_steps"):
            self.decode_steps = 0

        b, h, s, d = query_states.size()
        assert not self.hip_attn_args.need_apply_rope
        assert query_states.size(2) == 1
        assert h == 32

        # postfix_recompute_dense-window_[size:int]-diff_[1/0]-w_[size:int]
        postfix_string = os.environ.get("USE_ATTN_POSTFIX", "0")
        window = int(postfix_string.split("-")[1].split("_")[1])
        diff = int(postfix_string.split("-")[2].split("_")[1])
        w = int(postfix_string.split("-")[3].split("_")[1])
        self.decode_steps += 1
        if window == 0:
            window = None

        if (self.decode_steps - 1) % w == 0:
            # do dense
            if self.layer_idx == 0:
                print(f"doing dense")

            dense_attn_output, _ = self.sdpa_rectangle_forward(
                self,
                query_states,
                key_states,
                value_states,
                dropout=dropout,
                scaling=self.scaling,
            )

            # do window
            sparse_attn_output, _ = self.hip_attention_interface(
                self,
                query_states,
                key_states,
                value_states,
                attention_mask,
                dropout=dropout,
                scaling=scaling,
                sliding_window=window,
                **kwargs,
            )

            # get diff
            self.dense_sparse_diff = dense_attn_output - sparse_attn_output
            return dense_attn_output, None
        else:
            # do window
            if self.layer_idx == 0:
                print(f"doing sparse")

            sparse_attn_output, _ = self.hip_attention_interface(
                self,
                query_states,
                key_states,
                value_states,
                attention_mask,
                dropout=dropout,
                scaling=scaling,
                sliding_window=window,
                **kwargs,
            )

            # add stored dense difference vector to window output
            sparse_attn_output = sparse_attn_output + self.dense_sparse_diff

            return sparse_attn_output, None

    def forward(
        self,
        hidden_states: torch.Tensor,
        position_embeddings: Tuple[torch.Tensor, torch.Tensor],
        attention_mask: Optional[torch.Tensor],
        past_key_value: Optional[Cache] = None,
        cache_position: Optional[torch.LongTensor] = None,
        **kwargs: Unpack[FlashAttentionKwargs],
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:

        input_shape = hidden_states.shape[:-1]
        hidden_shape = (*input_shape, -1, self.head_dim)

        query_states = self.q_proj(hidden_states).view(hidden_shape).transpose(1, 2)
        key_states = self.k_proj(hidden_states).view(hidden_shape).transpose(1, 2)
        value_states = self.v_proj(hidden_states).view(hidden_shape).transpose(1, 2)

        cos, sin = position_embeddings
        if self.is_need_apply_rope():
            query_states, key_states = apply_rotary_pos_emb(
                query_states, key_states, cos, sin
            )

        recompute = "recompute" in self.config._attn_implementation
        if past_key_value is not None:
            # sin and cos are specific to RoPE models; cache_position needed for the static cache
            cache_kwargs = {"sin": sin, "cos": cos, "cache_position": cache_position}
            if past_key_value is not None:
                key_states, value_states = past_key_value.update(
                    key_states, value_states, self.layer_idx, cache_kwargs
                )
        elif past_key_value is not None:
            assert False
            # sin and cos are specific to RoPE models; cache_position needed for the static cache
            cache_kwargs = {"sin": sin, "cos": cos, "cache_position": cache_position}

            b, h, _, d = key_states.size()
            idx = past_key_value.recompute_idx
            idx = idx.view(1, 1, -1, 1).repeat(b, h, 1, d)
            past_key_value.key_cache[self.layer_idx].scatter_(2, idx, key_states)
            past_key_value.value_cache[self.layer_idx].scatter_(2, idx, value_states)

            key_states = past_key_value.key_cache[self.layer_idx]
            value_states = past_key_value.value_cache[self.layer_idx]

        attention_interface: Callable = eager_attention_forward
        if self.config._attn_implementation != "eager":
            if self.config._attn_implementation == "sdpa" and kwargs.get(
                "output_attentions", False
            ):
                logger.warning_once(
                    "`torch.nn.functional.scaled_dot_product_attention` does not support `output_attentions=True`. Falling back to "
                    'eager attention. This warning can be removed using the argument `attn_implementation="eager"` when loading the model.'
                )
            if self.config._attn_implementation == "hip_attention":
                attention_interface = self.hip_attention_interface
                kwargs.update(
                    {
                        "rope_sin": sin.squeeze(0),
                        "rope_cos": cos.squeeze(0),
                    }
                )
            elif self.config._attn_implementation == "sdpa_rectangle":
                attention_interface = self.sdpa_rectangle_forward
                kwargs.update({"sin": sin, "cos": cos})
            elif self.config._attn_implementation == "dense":
                attention_interface = self.dense_forward
                kwargs.update({"sin": sin, "cos": cos})
            elif self.config._attn_implementation == "recompute_dense":
                attention_interface = self.recompute_dense_forward
                kwargs.update(
                    {
                        "rope_sin": sin.squeeze(0),
                        "rope_cos": cos.squeeze(0),
                    }
                )
            elif self.config._attn_implementation == "recompute_dense_decode":
                attention_interface = self.recompute_dense_forward_decode
                kwargs.update(
                    {
                        "rope_sin": sin.squeeze(0),
                        "rope_cos": cos.squeeze(0),
                    }
                )
            else:
                attention_interface = ALL_ATTENTION_FUNCTIONS[
                    self.config._attn_implementation
                ]

        attn_output, attn_weights = attention_interface(
            self,
            query_states,
            key_states,
            value_states,
            attention_mask,
            dropout=0.0 if not self.training else self.attention_dropout,
            scaling=self.scaling,
            **kwargs,
        )

        assert attn_output.shape[-2] == 32, attn_output.shape
        b, s, _, _ = attn_output.size()
        attn_output = attn_output.reshape(b, s, -1).contiguous()
        # attn_output = attn_output.reshape(*input_shape, -1).contiguous()

        attn_output = self.o_proj(attn_output)
        return attn_output, attn_weights


class LlamaDecoderLayer(nn.Module):
    def __init__(self, config: LlamaConfig, layer_idx: int):
        super().__init__()
        self.hidden_size = config.hidden_size

        self.self_attn = LlamaAttention(config=config, layer_idx=layer_idx)

        self.mlp = LlamaMLP(config)
        self.input_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
        self.post_attention_layernorm = LlamaRMSNorm(
            config.hidden_size, eps=config.rms_norm_eps
        )

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Cache] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        cache_position: Optional[torch.LongTensor] = None,
        position_embeddings: Optional[
            Tuple[torch.Tensor, torch.Tensor]
        ] = None,  # necessary, but kept here for BC
        **kwargs: Unpack[FlashAttentionKwargs],
    ) -> Tuple[
        torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]
    ]:
        assert hidden_states.dtype in [torch.float16, torch.bfloat16]
        residual = hidden_states

        hidden_states = self.input_layernorm(hidden_states)
        assert hidden_states.dtype in [torch.float16, torch.bfloat16]

        # Self Attention
        hidden_states, self_attn_weights = self.self_attn(
            hidden_states=hidden_states,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_value=past_key_value,
            output_attentions=output_attentions,
            use_cache=use_cache,
            cache_position=cache_position,
            position_embeddings=position_embeddings,
            **kwargs,
        )
        assert hidden_states.dtype in [torch.float16, torch.bfloat16]
        hidden_states = residual + hidden_states
        assert hidden_states.dtype in [torch.float16, torch.bfloat16]

        # Fully Connected
        residual = hidden_states
        hidden_states = self.post_attention_layernorm(hidden_states)
        hidden_states = self.mlp(hidden_states)
        hidden_states = residual + hidden_states

        assert hidden_states.dtype in [torch.float16, torch.bfloat16]

        outputs = (hidden_states,)
        if output_attentions:
            outputs += (self_attn_weights,)

        return outputs


LLAMA_START_DOCSTRING = r"""
    This model inherits from [`PreTrainedModel`]. Check the superclass documentation for the generic methods the
    library implements for all its model (such as downloading or saving, resizing the input embeddings, pruning heads
    etc.)

    This model is also a PyTorch [torch.nn.Module](https://pytorch.org/docs/stable/nn.html#torch.nn.Module) subclass.
    Use it as a regular PyTorch Module and refer to the PyTorch documentation for all matter related to general usage
    and behavior.

    Parameters:
        config ([`LlamaConfig`]):
            Model configuration class with all the parameters of the model. Initializing with a config file does not
            load the weights associated with the model, only the configuration. Check out the
            [`~PreTrainedModel.from_pretrained`] method to load the model weights.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class LlamaPreTrainedModel(PreTrainedModel):
    config_class = LlamaConfig
    base_model_prefix = "model"
    supports_gradient_checkpointing = True
    _no_split_modules = ["LlamaDecoderLayer"]
    _skip_keys_device_placement = ["past_key_values"]
    _supports_flash_attn_2 = True
    _supports_sdpa = True
    _supports_flex_attn = True
    _supports_cache_class = True
    _supports_quantized_cache = False
    _supports_static_cache = False
    _supports_attention_backend = True

    def _init_weights(self, module):
        std = self.config.initializer_range
        if isinstance(module, nn.Linear):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.bias is not None:
                module.bias.data.zero_()
        elif isinstance(module, nn.Embedding):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.padding_idx is not None:
                module.weight.data[module.padding_idx].zero_()
        elif isinstance(module, LoraAdopter):
            print("init", module.lora_a.device)
            module._init_weights()


LLAMA_INPUTS_DOCSTRING = r"""
    Args:
        input_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`):
            Indices of input sequence tokens in the vocabulary. Padding will be ignored by default should you provide
            it.

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            [What are input IDs?](../glossary#input-ids)
        attention_mask (`torch.Tensor` of shape `(batch_size, sequence_length)`, *optional*):
            Mask to avoid performing attention on padding token indices. Mask values selected in `[0, 1]`:

            - 1 for tokens that are **not masked**,
            - 0 for tokens that are **masked**.

            [What are attention masks?](../glossary#attention-mask)

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            If `past_key_values` is used, optionally only the last `input_ids` have to be input (see
            `past_key_values`).

            If you want to change padding behavior, you should read [`modeling_opt._prepare_decoder_attention_mask`]
            and modify to your needs. See diagram 1 in [the paper](https://arxiv.org/abs/1910.13461) for more
            information on the default strategy.

            - 1 indicates the head is **not masked**,
            - 0 indicates the head is **masked**.
        position_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
            Indices of positions of each input sequence tokens in the position embeddings. Selected in the range `[0,
            config.n_positions - 1]`.

            [What are position IDs?](../glossary#position-ids)
        past_key_values (`Cache` or `tuple(tuple(torch.FloatTensor))`, *optional*):
            Pre-computed hidden-states (key and values in the self-attention blocks and in the cross-attention
            blocks) that can be used to speed up sequential decoding. This typically consists in the `past_key_values`
            returned by the model at a previous stage of decoding, when `use_cache=True` or `config.use_cache=True`.

            Two formats are allowed:
            - a [`~cache_utils.Cache`] instance, see our
            [kv cache guide](https://huggingface.co/docs/transformers/en/kv_cache);
            - Tuple of `tuple(torch.FloatTensor)` of length `config.n_layers`, with each tuple having 2 tensors of
            shape `(batch_size, num_heads, sequence_length, embed_size_per_head)`). This is also known as the legacy
            cache format.

            The model will output the same cache format that is fed as input. If no `past_key_values` are passed, the
            legacy cache format will be returned.

            If `past_key_values` are used, the user can optionally input only the last `input_ids` (those that don't
            have their past key value states given to this model) of shape `(batch_size, 1)` instead of all `input_ids`
            of shape `(batch_size, sequence_length)`.
        inputs_embeds (`torch.FloatTensor` of shape `(batch_size, sequence_length, hidden_size)`, *optional*):
            Optionally, instead of passing `input_ids` you can choose to directly pass an embedded representation. This
            is useful if you want more control over how to convert `input_ids` indices into associated vectors than the
            model's internal embedding lookup matrix.
        use_cache (`bool`, *optional*):
            If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding (see
            `past_key_values`).
        output_attentions (`bool`, *optional*):
            Whether or not to return the attentions tensors of all attention layers. See `attentions` under returned
            tensors for more detail.
        output_hidden_states (`bool`, *optional*):
            Whether or not to return the hidden states of all layers. See `hidden_states` under returned tensors for
            more detail.
        return_dict (`bool`, *optional*):
            Whether or not to return a [`~utils.ModelOutput`] instead of a plain tuple.
        cache_position (`torch.LongTensor` of shape `(sequence_length)`, *optional*):
            Indices depicting the position of the input sequence tokens in the sequence. Contrarily to `position_ids`,
            this tensor is not affected by padding. It is used to update the cache in the correct position and to infer
            the complete sequence length.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class LlamaModel(LlamaPreTrainedModel):
    """
    Transformer decoder consisting of *config.num_hidden_layers* layers. Each layer is a [`LlamaDecoderLayer`]

    Args:
        config: LlamaConfig
    """

    def __init__(self, config: LlamaConfig):
        super().__init__(config)
        self.padding_idx = config.pad_token_id
        self.vocab_size = config.vocab_size

        self.embed_tokens = nn.Embedding(
            config.vocab_size, config.hidden_size, self.padding_idx
        )
        self.layers = nn.ModuleList(
            [
                LlamaDecoderLayer(config, layer_idx)
                for layer_idx in range(config.num_hidden_layers)
            ]
        )
        self.norm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
        self.rotary_emb = LlamaRotaryEmbedding(config=config)
        self.gradient_checkpointing = False

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.embed_tokens

    def set_input_embeddings(self, value):
        self.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Cache] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        cache_position: Optional[torch.LongTensor] = None,
        token_dropout_p: Optional[float] = 0.0,
        labels: Optional[torch.Tensor] = None,
        **flash_attn_kwargs: Unpack[FlashAttentionKwargs],
    ) -> Union[Tuple, BaseModelOutputWithPast]:
        output_attentions = (
            output_attentions
            if output_attentions is not None
            else self.config.output_attentions
        )
        output_hidden_states = (
            output_hidden_states
            if output_hidden_states is not None
            else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache
        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        if (input_ids is None) ^ (inputs_embeds is not None):
            raise ValueError(
                "You must specify exactly one of input_ids or inputs_embeds"
            )

        if self.gradient_checkpointing and self.training and use_cache:
            logger.warning_once(
                "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`."
            )
            use_cache = False

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)

        if use_cache and past_key_values is None:
            past_key_values = DynamicCache()

        if cache_position is None:
            if not hasattr(past_key_values, "recompute_idx"):
                past_seen_tokens = (
                    past_key_values.get_seq_length()
                    if past_key_values is not None
                    else 0
                )
                cache_position = torch.arange(
                    past_seen_tokens,
                    past_seen_tokens + inputs_embeds.shape[1],
                    device=inputs_embeds.device,
                )
            else:
                cache_position = torch.arange(
                    0,
                    past_key_values.get_seq_length(),
                    device=inputs_embeds.device,
                )

        if position_ids is None:
            position_ids = cache_position.unsqueeze(0)

        causal_mask = self._update_causal_mask(
            attention_mask,
            inputs_embeds,
            cache_position,
            past_key_values,
            output_attentions,
        )

        hidden_states = inputs_embeds
        if self.training:
            hidden_states.requires_grad_(True)

        # create position embeddings to be shared across the decoder layers
        rope_ids = position_ids
        if (past_key_values is not None) and (
            not self.layers[0].self_attn.is_need_apply_rope()
            and not hasattr(past_key_values, "recompute_idx")
        ):
            assert isinstance(past_key_values, Cache)
            max_rope_len = past_key_values.get_max_cache_shape()
            if max_rope_len is None:
                max_rope_len = (
                    past_key_values.get_seq_length(0) + hidden_states.shape[1]
                )
            rope_ids = torch.arange(0, max_rope_len, device=hidden_states.device)[
                None, :
            ]
        position_embeddings = self.rotary_emb(hidden_states, rope_ids)

        cos, sin = position_embeddings

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None

        for decoder_layer in self.layers[: self.config.num_hidden_layers]:
            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    decoder_layer.__call__,
                    hidden_states,
                    causal_mask,
                    position_ids,
                    past_key_values,
                    output_attentions,
                    use_cache,
                    cache_position,
                    position_embeddings,
                )
            else:
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=causal_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_values,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                    cache_position=cache_position,
                    position_embeddings=position_embeddings,
                    **flash_attn_kwargs,
                )

            hidden_states = layer_outputs[0]
            assert hidden_states.dtype in [torch.float16, torch.bfloat16]

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        output = BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=past_key_values if use_cache else None,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
        )
        return output if return_dict else output.to_tuple()

    def _update_causal_mask(
        self,
        attention_mask: torch.Tensor,
        input_tensor: torch.Tensor,
        cache_position: torch.Tensor,
        past_key_values: Cache,
        output_attentions: bool,
    ):
        if self.config._attn_implementation in [
            "flash_attention_2",
            "hip_attention",
            "sdpa_rectangle",
            "recompute",
            "recompute_dense",
            "recompute_dense_decode",
            "dense",
        ]:
            if attention_mask is not None and (attention_mask == 0.0).any():
                return attention_mask
            return None
        if self.config._attn_implementation == "flex_attention":
            if isinstance(attention_mask, torch.Tensor):
                attention_mask = make_flex_block_causal_mask(attention_mask)
            if isinstance(attention_mask, BlockMask):
                return attention_mask

        # For SDPA, when possible, we will rely on its `is_causal` argument instead of its `attn_mask` argument, in
        # order to dispatch on Flash Attention 2. This feature is not compatible with static cache, as SDPA will fail
        # to infer the attention mask.
        past_seen_tokens = (
            past_key_values.get_seq_length() if past_key_values is not None else 0
        )
        using_static_cache = isinstance(past_key_values, StaticCache)

        # When output attentions is True, sdpa implementation's forward method calls the eager implementation's forward
        if (
            self.config._attn_implementation == "sdpa"
            and not using_static_cache
            and not output_attentions
        ):
            if AttentionMaskConverter._ignore_causal_mask_sdpa(
                attention_mask,
                inputs_embeds=input_tensor,
                past_key_values_length=past_seen_tokens,
                is_training=self.training,
            ):
                return None

        dtype, device = input_tensor.dtype, input_tensor.device
        sequence_length = input_tensor.shape[1]
        if using_static_cache:
            target_length = past_key_values.get_max_cache_shape()
        else:
            target_length = (
                attention_mask.shape[-1]
                if isinstance(attention_mask, torch.Tensor)
                else past_seen_tokens + sequence_length + 1
            )

        # In case the provided `attention` mask is 2D, we generate a causal mask here (4D).
        causal_mask = self._prepare_4d_causal_attention_mask_with_cache_position(
            attention_mask,
            sequence_length=sequence_length,
            target_length=target_length,
            dtype=dtype,
            device=device,
            cache_position=cache_position,
            batch_size=input_tensor.shape[0],
        )

        if (
            self.config._attn_implementation == "sdpa"
            and attention_mask is not None
            and attention_mask.device.type in ["cuda", "xpu"]
            and not output_attentions
        ):
            # Attend to all tokens in fully masked rows in the causal_mask, for example the relevant first rows when
            # using left padding. This is required by F.scaled_dot_product_attention memory-efficient attention path.
            # Details: https://github.com/pytorch/pytorch/issues/110213
            min_dtype = torch.finfo(dtype).min
            causal_mask = AttentionMaskConverter._unmask_unattended(
                causal_mask, min_dtype
            )

        return causal_mask

    @staticmethod
    def _prepare_4d_causal_attention_mask_with_cache_position(
        attention_mask: torch.Tensor,
        sequence_length: int,
        target_length: int,
        dtype: torch.dtype,
        device: torch.device,
        cache_position: torch.Tensor,
        batch_size: int,
        **kwargs,
    ):
        """
        Creates a causal 4D mask of shape `(batch_size, 1, query_length, key_value_length)` from a 2D mask of shape
        `(batch_size, key_value_length)`, or if the input `attention_mask` is already 4D, do nothing.

        Args:
            attention_mask (`torch.Tensor`):
                A 2D attention mask of shape `(batch_size, key_value_length)` or a 4D attention mask of shape
                `(batch_size, 1, query_length, key_value_length)`.
            sequence_length (`int`):
                The sequence length being processed.
            target_length (`int`):
                The target length: when generating with static cache, the mask should be as long as the static cache,
                to account for the 0 padding, the part of the cache that is not filled yet.
            dtype (`torch.dtype`):
                The dtype to use for the 4D attention mask.
            device (`torch.device`):
                The device to place the 4D attention mask on.
            cache_position (`torch.Tensor`):
                Indices depicting the position of the input sequence tokens in the sequence.
            batch_size (`torch.Tensor`):
                Batch size.
        """
        if attention_mask is not None and attention_mask.dim() == 4:
            # In this case we assume that the mask comes already in inverted form and requires no inversion or slicing.
            causal_mask = attention_mask
        else:
            min_dtype = torch.finfo(dtype).min
            causal_mask = torch.full(
                (sequence_length, target_length),
                fill_value=min_dtype,
                dtype=dtype,
                device=device,
            )
            if sequence_length != 1:
                causal_mask = torch.triu(causal_mask, diagonal=1)
            causal_mask *= torch.arange(
                target_length, device=device
            ) > cache_position.reshape(-1, 1)
            causal_mask = causal_mask[None, None, :, :].expand(batch_size, 1, -1, -1)
            if attention_mask is not None:
                causal_mask = (
                    causal_mask.clone()
                )  # copy to contiguous memory for in-place edit
                mask_length = attention_mask.shape[-1]
                padding_mask = causal_mask[:, :, :, :mask_length] + attention_mask[
                    :, None, None, :
                ].to(causal_mask.device)
                padding_mask = padding_mask == 0
                causal_mask[:, :, :, :mask_length] = causal_mask[
                    :, :, :, :mask_length
                ].masked_fill(padding_mask, min_dtype)

        return causal_mask


class KwargsForCausalLM(FlashAttentionKwargs, LossKwargs): ...


class LlamaForCausalLM(LlamaPreTrainedModel, GenerationMixin):
    _tied_weights_keys = ["lm_head.weight"]
    _tp_plan = {"lm_head": "colwise_rep"}
    _pp_plan = {"lm_head": (["hidden_states"], ["logits"])}

    def __init__(self, config):
        super().__init__(config)
        self.model = LlamaModel(config)
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    def get_output_embeddings(self):
        return self.lm_head

    def set_output_embeddings(self, new_embeddings):
        self.lm_head = new_embeddings

    def set_decoder(self, decoder):
        self.model = decoder

    def get_decoder(self):
        return self.model

    @deprecate_kwarg("num_logits_to_keep", version="4.50", new_name="logits_to_keep")
    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    @replace_return_docstrings(
        output_type=CausalLMOutputWithPast, config_class=_CONFIG_FOR_DOC
    )
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Union[Cache, List[torch.FloatTensor]]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        cache_position: Optional[torch.LongTensor] = None,
        logits_to_keep: Union[int, torch.Tensor] = 0,
        output_logits: bool = True,
        **kwargs: Unpack[KwargsForCausalLM],
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        r"""
            labels (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
                Labels for computing the masked language modeling loss. Indices should either be in `[0, ...,
                config.vocab_size]` or -100 (see `input_ids` docstring). Tokens with indices set to `-100` are ignored
                (masked), the loss is only computed for the tokens with labels in `[0, ..., config.vocab_size]`.

            logits_to_keep (`int` or `torch.Tensor`, *optional*):
                If an `int`, compute logits for the last `logits_to_keep` tokens. If `0`, calculate logits for all
                `input_ids` (special case). Only last token logits are needed for generation, and calculating them only for that
                token can save memory, which becomes pretty significant for long sequences or large vocabulary size.
                If a `torch.Tensor`, must be 1D corresponding to the indices to keep in the sequence length dimension.
                This is useful when using packed tensor format (single dimension for batch and sequence length).

        Returns:

        Example:

        ```python
        >>> from transformers import AutoTokenizer, LlamaForCausalLM

        >>> model = LlamaForCausalLM.from_pretrained("meta-llama/Llama-2-7b-hf")
        >>> tokenizer = AutoTokenizer.from_pretrained("meta-llama/Llama-2-7b-hf")

        >>> prompt = "Hey, are you conscious? Can you talk to me?"
        >>> inputs = tokenizer(prompt, return_tensors="pt")

        >>> # Generate
        >>> generate_ids = model.generate(inputs.input_ids, max_length=30)
        >>> tokenizer.batch_decode(generate_ids, skip_special_tokens=True, clean_up_tokenization_spaces=False)[0]
        "Hey, are you conscious? Can you talk to me?\nI'm not conscious, but I can talk to you."
        ```"""
        # assert output_logits

        output_attentions = (
            output_attentions
            if output_attentions is not None
            else self.config.output_attentions
        )
        output_hidden_states = (
            output_hidden_states
            if output_hidden_states is not None
            else self.config.output_hidden_states
        )
        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
        outputs = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
            cache_position=cache_position,
            labels=labels,
            **kwargs,
        )

        hidden_states = outputs[0]
        assert hidden_states.dtype in [torch.bfloat16, torch.float16]
        # Only compute necessary logits, and do not upcast them to float if we are not computing the loss
        slice_indices = (
            slice(-logits_to_keep, None)
            if isinstance(logits_to_keep, int)
            else logits_to_keep
        )

        loss = None
        if hasattr(self, "hip_train_config") or hasattr(self, "no_lm_head"):
            # if we are training with hip, we need to do an
            # efficient computation of logits and long ce to avoid
            # pytting all logits into memory
            logits = hidden_states
        else:
            if output_logits:
                logits = self.lm_head(hidden_states[:, slice_indices, :])

                if labels is not None:
                    loss = self.loss_function(
                        logits=logits,
                        labels=labels,
                        vocab_size=self.config.vocab_size,
                        **kwargs,
                    )
            else:
                if labels is not None:
                    from hip_attn.utils.memory_efficient_llm_ce import \
                        memory_efficient_llm_ce

                    first_skip = 0
                    # Shift so that tokens < n predict n
                    shift_states = hidden_states[..., first_skip:-1, :].contiguous()
                    shift_labels = labels[..., first_skip + 1 :].contiguous()
                    # Flatten the tokens
                    shift_states = shift_states.view(-1, shift_states.shape[-1])
                    shift_labels = shift_labels.view(-1)
                    # Enable model parallelism
                    shift_labels = shift_labels.to(shift_states.device)

                    loss = memory_efficient_llm_ce(
                        shift_states,
                        self.lm_head.weight,
                        shift_labels,
                    )
                logits = None

        if not return_dict:
            output = (logits,) + outputs[1:]
            return (loss,) + output if loss is not None else output

        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=outputs.past_key_values,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )


@add_start_docstrings(
    """
    The LLaMa Model transformer with a sequence classification head on top (linear layer).

    [`LlamaForSequenceClassification`] uses the last token in order to do the classification, as other causal models
    (e.g. GPT-2) do.

    Since it does classification on the last token, it requires to know the position of the last token. If a
    `pad_token_id` is defined in the configuration, it finds the last token that is not a padding token in each row. If
    no `pad_token_id` is defined, it simply takes the last value in each row of the batch. Since it cannot guess the
    padding tokens when `inputs_embeds` are passed instead of `input_ids`, it does the same (take the last value in
    each row of the batch).
    """,
    LLAMA_START_DOCSTRING,
)
class LlamaForSequenceClassification(LlamaPreTrainedModel):
    def __init__(self, config):
        super().__init__(config)
        self.num_labels = config.num_labels
        self.model = LlamaModel(config)
        self.score = nn.Linear(config.hidden_size, self.num_labels, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: Optional[torch.LongTensor] = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Union[Cache, List[torch.FloatTensor]]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, SequenceClassifierOutputWithPast]:
        r"""
        labels (`torch.LongTensor` of shape `(batch_size,)`, *optional*):
            Labels for computing the sequence classification/regression loss. Indices should be in `[0, ...,
            config.num_labels - 1]`. If `config.num_labels == 1` a regression loss is computed (Mean-Square loss), If
            `config.num_labels > 1` a classification loss is computed (Cross-Entropy).
        """
        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        transformer_outputs = self.model(
            input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        hidden_states = transformer_outputs[0]
        logits = self.score(hidden_states)

        if input_ids is not None:
            batch_size = input_ids.shape[0]
        else:
            batch_size = inputs_embeds.shape[0]

        if self.config.pad_token_id is None and batch_size != 1:
            raise ValueError(
                "Cannot handle batch sizes > 1 if no padding token is defined."
            )
        if self.config.pad_token_id is None:
            last_non_pad_token = -1
        elif input_ids is not None:
            # To handle both left- and right- padding, we take the rightmost token that is not equal to pad_token_id
            non_pad_mask = (input_ids != self.config.pad_token_id).to(
                logits.device, torch.int32
            )
            token_indices = torch.arange(input_ids.shape[-1], device=logits.device)
            last_non_pad_token = (token_indices * non_pad_mask).argmax(-1)
        else:
            last_non_pad_token = -1
            logger.warning_once(
                f"{self.__class__.__name__} will not detect padding tokens in `inputs_embeds`. Results may be "
                "unexpected if using padding tokens in conjunction with `inputs_embeds.`"
            )

        pooled_logits = logits[
            torch.arange(batch_size, device=logits.device), last_non_pad_token
        ]

        loss = None
        if labels is not None:
            loss = self.loss_function(
                logits=logits,
                labels=labels,
                pooled_logits=pooled_logits,
                config=self.config,
            )

        if not return_dict:
            output = (pooled_logits,) + transformer_outputs[1:]
            return ((loss,) + output) if loss is not None else output

        return SequenceClassifierOutputWithPast(
            loss=loss,
            logits=pooled_logits,
            past_key_values=transformer_outputs.past_key_values,
            hidden_states=transformer_outputs.hidden_states,
            attentions=transformer_outputs.attentions,
        )


@add_start_docstrings(
    """
The Llama Model transformer with a span classification head on top for extractive question-answering tasks like
SQuAD (a linear layer on top of the hidden-states output to compute `span start logits` and `span end logits`).
    """,
    LLAMA_START_DOCSTRING,
)
class LlamaForQuestionAnswering(LlamaPreTrainedModel):
    base_model_prefix = "transformer"

    # Copied from transformers.models.bloom.modeling_bloom.BloomForQuestionAnswering.__init__ with Bloom->Llama
    def __init__(self, config):
        super().__init__(config)
        self.transformer = LlamaModel(config)
        self.qa_outputs = nn.Linear(config.hidden_size, 2)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.transformer.embed_tokens

    def set_input_embeddings(self, value):
        self.transformer.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: Optional[torch.LongTensor] = None,
        attention_mask: Optional[torch.FloatTensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Union[Cache, List[torch.FloatTensor]]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        start_positions: Optional[torch.LongTensor] = None,
        end_positions: Optional[torch.LongTensor] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        **kwargs,
    ) -> Union[Tuple, QuestionAnsweringModelOutput]:
        r"""
        start_positions (`torch.LongTensor` of shape `(batch_size,)`, *optional*):
            Labels for position (index) of the start of the labelled span for computing the token classification loss.
            Positions are clamped to the length of the sequence (`sequence_length`). Position outside of the sequence
            are not taken into account for computing the loss.
        end_positions (`torch.LongTensor` of shape `(batch_size,)`, *optional*):
            Labels for position (index) of the end of the labelled span for computing the token classification loss.
            Positions are clamped to the length of the sequence (`sequence_length`). Position outside of the sequence
            are not taken into account for computing the loss.
        """
        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        outputs = self.transformer(
            input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )

        sequence_output = outputs[0]

        logits = self.qa_outputs(sequence_output)
        start_logits, end_logits = logits.split(1, dim=-1)
        start_logits = start_logits.squeeze(-1).contiguous()
        end_logits = end_logits.squeeze(-1).contiguous()

        loss = None
        if start_positions is not None and end_positions is not None:
            loss = self.loss_function(
                start_logits, end_logits, start_positions, end_positions, **kwargs
            )

        if not return_dict:
            output = (start_logits, end_logits) + outputs[2:]
            return ((loss,) + output) if loss is not None else output

        return QuestionAnsweringModelOutput(
            loss=loss,
            start_logits=start_logits,
            end_logits=end_logits,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )


@add_start_docstrings(
    """
    The Llama Model transformer with a token classification head on top (a linear layer on top of the hidden-states
    output) e.g. for Named-Entity-Recognition (NER) tasks.
    """,
    LLAMA_START_DOCSTRING,
)
class LlamaForTokenClassification(LlamaPreTrainedModel):
    def __init__(self, config):
        super().__init__(config)
        self.num_labels = config.num_labels
        self.model = LlamaModel(config)
        if getattr(config, "classifier_dropout", None) is not None:
            classifier_dropout = config.classifier_dropout
        elif getattr(config, "hidden_dropout", None) is not None:
            classifier_dropout = config.hidden_dropout
        else:
            classifier_dropout = 0.1
        self.dropout = nn.Dropout(classifier_dropout)
        self.score = nn.Linear(config.hidden_size, config.num_labels)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    @add_code_sample_docstrings(
        checkpoint=_CHECKPOINT_FOR_DOC,
        output_type=TokenClassifierOutput,
        config_class=_CONFIG_FOR_DOC,
    )
    def forward(
        self,
        input_ids: Optional[torch.LongTensor] = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, TokenClassifierOutput]:
        r"""
        labels (`torch.LongTensor` of shape `(batch_size,)`, *optional*):
            Labels for computing the sequence classification/regression loss. Indices should be in `[0, ...,
            config.num_labels - 1]`. If `config.num_labels == 1` a regression loss is computed (Mean-Square loss), If
            `config.num_labels > 1` a classification loss is computed (Cross-Entropy).
        """
        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        outputs = self.model(
            input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        sequence_output = outputs[0]
        sequence_output = self.dropout(sequence_output)
        logits = self.score(sequence_output)

        loss = None
        if labels is not None:
            loss = self.loss_function(logits, labels, self.config)

        if not return_dict:
            output = (logits,) + outputs[2:]
            return ((loss,) + output) if loss is not None else output

        return TokenClassifierOutput(
            loss=loss,
            logits=logits,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )


__all__ = [
    "LlamaForCausalLM",
    "LlamaModel",
    "LlamaPreTrainedModel",
    "LlamaForSequenceClassification",
    "LlamaForQuestionAnswering",
    "LlamaForTokenClassification",
]
