import math
import os
import types
from typing import Any, Dict, Literal, Optional, Union

import datasets
import numpy as np
import torch
import torch.nn.functional as F
import transformers
from argparse_dataclass import ArgumentParser
from sklearn.model_selection import train_test_split
from torch import nn
from transformers.cache_utils import Cache
from transformers.generation.configuration_utils import GenerationConfig
from transformers.generation.logits_process import LogitsProcessorList
from transformers.generation.stopping_criteria import StoppingCriteriaList
from transformers.generation.streamers import BaseStreamer
from transformers.generation.utils import GenerateNonBeamOutput
from transformers.modeling_utils import ALL_ATTENTION_FUNCTIONS

import wandb
from hip_attn.v1_3.attention import HiPAttentionArgs, ScanStage
from hip_attn.v1_3.models.llama import (LlamaAttention, LlamaConfig,
                                        LlamaForCausalLM)
from hip_research.dataset.openwebtext import OpenWebTextDataset
from hip_research.dataset.pg19 import PG19Dataset
from hip_research.dataset.pg19_long_qa import PG19LongQA
from hip_research.main.jobs.ppl import PplArgs, job_ppl
from hip_research.main.long_eval_experimental import get_hip_config, long_ppl
from hip_research.utils.long_train import Config, get_logger

log = get_logger()

TRITON_DEBUG = os.getenv("TRITON_DEBUG", "0") == "1"
WANDB_DISABLED = os.getenv("WANDB_MODE", "none") == "disabled"


def evaluate(
    config,
    model,
    tokenizer,
    text,
    ignore_keys=None,
    metric_key_prefix: str = "eval",
):

    model.eval()

    with torch.no_grad():

        text = f"""
        I will give you a story. I want you to summarize it in 100 words.
        Story:\n{text}\n
        What is this story about? Be detailed, use 100 words. This story is about
        """

        inputs = tokenizer(text, return_tensors="pt")["input_ids"]
        print(f"{inputs.size()=}")
        inputs = inputs.cuda()

        print(f"eval: {inputs.size()=}")

        generated = eval(inputs, model, tokenizer, config)
        exit(f"{generated=}")


def eval(inputs, model, tokenizer, config):
    """
    long ce from https://arxiv.org/pdf/2410.23771 (equation 7 on page 6)
    """

    model.eval()

    setattr(model, "no_lm_head", True)
    gen_config = GenerationConfig(max_new_tokens=100, min_new_tokens=25)
    output = model.generate(inputs, generation_config=gen_config)
    output = output[0, inputs.size(1) :]
    out = tokenizer.decode(output)
    print(f"{inputs.size()=} {out=}")


def _sample(
    self,
    input_ids: torch.LongTensor,
    logits_processor: LogitsProcessorList,
    stopping_criteria: StoppingCriteriaList,
    generation_config: GenerationConfig,
    synced_gpus: bool,
    streamer: Optional["BaseStreamer"],
    **model_kwargs,
) -> Union[GenerateNonBeamOutput, torch.LongTensor]:
    r"""
    Generates sequences of token ids for models with a language modeling head using **multinomial sampling** and
    can be used for text-decoder, text-to-text, speech-to-text, and vision-to-text models.

    Parameters:
        input_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`):
            The sequence used as a prompt for the generation.
        logits_processor (`LogitsProcessorList`):
            An instance of [`LogitsProcessorList`]. List of instances of class derived from [`LogitsProcessor`]
            used to modify the prediction scores of the language modeling head applied at each generation step.
        stopping_criteria (`StoppingCriteriaList`):
            An instance of [`StoppingCriteriaList`]. List of instances of class derived from [`StoppingCriteria`]
            used to tell if the generation loop should stop.
        generation_config ([`~generation.GenerationConfig`]):
            The generation configuration to be used as parametrization of the decoding method.
        synced_gpus (`bool`):
            Whether to continue running the while loop until max_length (needed to avoid deadlocking with
            `FullyShardedDataParallel` and DeepSpeed ZeRO Stage 3).
        streamer (`BaseStreamer`, *optional*):
            Streamer object that will be used to stream the generated sequences. Generated tokens are passed
            through `streamer.put(token_ids)` and the streamer is responsible for any further processing.
        model_kwargs:
            Additional model specific kwargs will be forwarded to the `forward` function of the model. If model is
            an encoder-decoder model the kwargs should include `encoder_outputs`.

    Return:
        [`~generation.GenerateDecoderOnlyOutput`], [`~generation.GenerateEncoderDecoderOutput`] or `torch.LongTensor`:
        A `torch.LongTensor` containing the generated tokens (default behaviour) or a
        [`~generation.GenerateDecoderOnlyOutput`] if `model.config.is_encoder_decoder=False` and
        `return_dict_in_generate=True` or a [`~generation.GenerateEncoderDecoderOutput`] if
        `model.config.is_encoder_decoder=True`.
    """
    # init values
    pad_token_id = generation_config._pad_token_tensor
    output_attentions = generation_config.output_attentions
    output_hidden_states = generation_config.output_hidden_states
    output_scores = generation_config.output_scores
    output_logits = generation_config.output_logits
    return_dict_in_generate = generation_config.return_dict_in_generate
    max_length = generation_config.max_length
    has_eos_stopping_criteria = any(
        hasattr(criteria, "eos_token_id") for criteria in stopping_criteria
    )
    do_sample = generation_config.do_sample

    # init attention / hidden states / scores tuples
    scores = () if (return_dict_in_generate and output_scores) else None
    raw_logits = () if (return_dict_in_generate and output_logits) else None
    decoder_attentions = () if (return_dict_in_generate and output_attentions) else None
    cross_attentions = () if (return_dict_in_generate and output_attentions) else None
    decoder_hidden_states = (
        () if (return_dict_in_generate and output_hidden_states) else None
    )

    # if model is an encoder-decoder, retrieve encoder attention weights and hidden states
    if return_dict_in_generate and self.config.is_encoder_decoder:
        encoder_attentions = (
            model_kwargs["encoder_outputs"].get("attentions")
            if output_attentions
            else None
        )
        encoder_hidden_states = (
            model_kwargs["encoder_outputs"].get("hidden_states")
            if output_hidden_states
            else None
        )

    # keep track of which sequences are already finished
    batch_size, cur_len = input_ids.shape
    this_peer_finished = False
    unfinished_sequences = torch.ones(
        batch_size, dtype=torch.long, device=input_ids.device
    )
    model_kwargs = self._get_initial_cache_position(input_ids, model_kwargs)

    model_forward = self.__call__
    if isinstance(model_kwargs.get("past_key_values"), Cache):
        is_compileable = (
            model_kwargs["past_key_values"].is_compileable
            and self._supports_static_cache
        )
        is_compileable = is_compileable and not self.generation_config.disable_compile
        if is_compileable and (
            self.device.type == "cuda"
            or generation_config.compile_config._compile_all_devices
        ):
            os.environ["TOKENIZERS_PARALLELISM"] = "0"
            model_forward = self.get_compiled_call(generation_config.compile_config)

    is_prefill = True
    while self._has_unfinished_sequences(
        this_peer_finished,
        synced_gpus,
        device=input_ids.device,
        cur_len=cur_len,
        max_length=max_length,
    ):
        # prepare model inputs
        model_inputs = self.prepare_inputs_for_generation(input_ids, **model_kwargs)

        # prepare variable output controls (note: some models won't accept all output controls)
        model_inputs.update(
            {"output_attentions": output_attentions} if output_attentions else {}
        )
        model_inputs.update(
            {"output_hidden_states": output_hidden_states}
            if output_hidden_states
            else {}
        )

        setattr(self, "no_lm_head", True)
        if is_prefill:
            if os.environ.get("USE_ATTN_POSTFIX", "0") == "long-ppl":  # long ppl style
                inputs = model_inputs["input_ids"]

                # hip forward ====================================
                previous_attn_backend = self.config._attn_implementation
                self.config._attn_implementation = "hip_attention"
                outputs = self(inputs, use_cache=True)
                self.config._attn_implementation = previous_attn_backend
                long_hidden = outputs.logits
                past_key_values = outputs.past_key_values

                # window forward ==================================
                previous_attn_backend = self.config._attn_implementation
                self.config._attn_implementation = "hip_attention"
                short_outputs = self(
                    inputs,
                    sliding_window=self.args.long_ce_k,
                    use_cache=False,
                )
                self.config._attn_implementation = previous_attn_backend
                short_hidden = short_outputs.logits

                # calculate indices for dense recomputation of specific tokens.
                long_ppl_scores = []
                long_ce_block_size = self.args.long_ce_block_size
                long_ppl_alpha, long_ppl_beta = (
                    self.args.long_ppl_alpha,
                    self.args.long_ppl_beta,
                )
                recompute_n = self.args.recompute_n
                for b in range(0, inputs.size(1), long_ce_block_size):
                    long_logits = self.lm_head(
                        long_hidden[:, b : b + long_ce_block_size]
                    )
                    short_logits = self.lm_head(
                        short_hidden[:, b : b + long_ce_block_size]
                    )
                    labels = inputs.to(long_logits.device)[
                        :, b : b + long_ce_block_size + 1
                    ]

                    out = long_ppl(
                        self,
                        long_logits,
                        short_logits,
                        labels,
                        idx_select=True,
                        alpha=long_ppl_alpha,
                        beta=long_ppl_beta,
                    )
                    long_ppl_scores.append(out)

                long_ppl_scores = torch.cat(long_ppl_scores, dim=0)
                sorted_idx = torch.argsort(long_ppl_scores, dim=-1, descending=True)
                refine_idx, other_idx = (
                    sorted_idx[:recompute_n],
                    sorted_idx[recompute_n:],
                )
                refine_idx = torch.sort(refine_idx, dim=-1).values
                other_idx = torch.sort(other_idx, dim=-1).values

                # recompute forward =====================================
                previous_attn_backend = self.config._attn_implementation
                self.config._attn_implementation = "recompute"
                past_key_values.recompute_idx = refine_idx
                recompute_inputs = inputs[:, refine_idx]
                recompute_outputs = self(
                    recompute_inputs,
                    past_key_values=past_key_values,
                    use_cache=True,
                )
                self.config._attn_implementation = previous_attn_backend
                past_key_values = recompute_outputs.past_key_values
                delattr(past_key_values, "recompute_idx")

                self.config._attn_implementation = "hip_attention"
                outputs.past_key_values = past_key_values
            elif "recompute_dense" in os.environ.get(
                "USE_ATTN_POSTFIX", "0"
            ):  # snapkv style

                inputs = model_inputs["input_ids"]
                self.config._attn_implementation = "recompute_dense"
                outputs = self(inputs, use_cache=True)

                postfix_string = os.environ.get("USE_ATTN_POSTFIX")
                decode_mode = postfix_string.split("-")[4].split("_")[1]

                if decode_mode == "dense":
                    self.config._attn_implementation = "sdpa_rectangle"
                elif decode_mode == "sparse":
                    self.config._attn_implementation = "recompute_dense_decode"
                else:
                    raise NotImplementedError(
                        f"deocde_mode: {decode_mode} is not implemented"
                    )

            elif (
                os.environ.get("USE_ATTN_POSTFIX", "0") == "dense-sanity-check"
            ):  # recompute all for sanity check
                inputs = model_inputs["input_ids"]

                # hip forward ====================================
                previous_attn_backend = self.config._attn_implementation
                self.config._attn_implementation = "hip_attention"
                outputs = self(inputs, use_cache=True)
                self.config._attn_implementation = previous_attn_backend
                long_hidden = outputs.logits
                past_key_values = outputs.past_key_values

                refine_idx = torch.arange(inputs.size(1), device=inputs.device)

                # recompute forward =====================================
                previous_attn_backend = self.config._attn_implementation
                self.config._attn_implementation = "recompute"
                past_key_values.recompute_idx = refine_idx
                recompute_inputs = inputs[:, refine_idx]
                recompute_outputs = self(
                    recompute_inputs,
                    past_key_values=past_key_values,
                    use_cache=True,
                )
                self.config._attn_implementation = previous_attn_backend
                past_key_values = recompute_outputs.past_key_values
                delattr(past_key_values, "recompute_idx")

                self.config._attn_implementation = "sdpa_rectangle"
                outputs.past_key_values = past_key_values
            else:
                outputs = self(**model_inputs, return_dict=True)

            is_prefill = False
        else:
            outputs = model_forward(**model_inputs, return_dict=True)

        # synced_gpus: don't waste resources running the code we don't need; kwargs must be updated before skipping
        model_kwargs = self._update_model_kwargs_for_generation(
            outputs,
            model_kwargs,
            is_encoder_decoder=self.config.is_encoder_decoder,
        )
        if synced_gpus and this_peer_finished:
            continue

        # Clone is needed to avoid keeping a hanging ref to outputs.logits which may be very large for first iteration
        # (the clone itself is always small)
        next_token_logits = outputs.logits[:, -1, :].clone()
        next_token_logits = self.lm_head(next_token_logits).float()
        next_token_logits = next_token_logits.to(input_ids.device)

        # pre-process distribution
        next_token_scores = logits_processor(input_ids, next_token_logits)

        # Store scores, attentions and hidden_states when required
        if return_dict_in_generate:
            if output_scores:
                scores += (next_token_scores,)
            if output_logits:
                raw_logits += (next_token_logits,)
            if output_attentions:
                decoder_attentions += (
                    (outputs.decoder_attentions,)
                    if self.config.is_encoder_decoder
                    else (outputs.attentions,)
                )
                if self.config.is_encoder_decoder:
                    cross_attentions += (outputs.cross_attentions,)

            if output_hidden_states:
                decoder_hidden_states += (
                    (outputs.decoder_hidden_states,)
                    if self.config.is_encoder_decoder
                    else (outputs.hidden_states,)
                )

        # token selection
        if do_sample:
            probs = nn.functional.softmax(next_token_scores, dim=-1)
            # TODO (joao): this OP throws "skipping cudagraphs due to ['incompatible ops']", find solution
            next_tokens = torch.multinomial(probs, num_samples=1).squeeze(1)
        else:
            next_tokens = torch.argmax(next_token_scores, dim=-1)

        # finished sentences should have their next token be a padding token
        if has_eos_stopping_criteria:
            next_tokens = next_tokens * unfinished_sequences + pad_token_id * (
                1 - unfinished_sequences
            )

        # update generated ids, model inputs, and length for next step
        input_ids = torch.cat([input_ids, next_tokens[:, None]], dim=-1)
        if streamer is not None:
            streamer.put(next_tokens.cpu())

        unfinished_sequences = unfinished_sequences & ~stopping_criteria(
            input_ids, scores
        )
        this_peer_finished = unfinished_sequences.max() == 0
        cur_len += 1

        # This is needed to properly delete outputs.logits which may be very large for first iteration
        # Otherwise a reference to outputs is kept which keeps the logits alive in the next iteration
        del outputs

    if streamer is not None:
        streamer.end()

    if return_dict_in_generate:
        if self.config.is_encoder_decoder:
            return GenerateEncoderDecoderOutput(
                sequences=input_ids,
                scores=scores,
                logits=raw_logits,
                encoder_attentions=encoder_attentions,
                encoder_hidden_states=encoder_hidden_states,
                decoder_attentions=decoder_attentions,
                cross_attentions=cross_attentions,
                decoder_hidden_states=decoder_hidden_states,
                past_key_values=model_kwargs.get("past_key_values"),
            )
        else:
            return GenerateDecoderOnlyOutput(
                sequences=input_ids,
                scores=scores,
                logits=raw_logits,
                attentions=decoder_attentions,
                hidden_states=decoder_hidden_states,
                past_key_values=model_kwargs.get("past_key_values"),
            )
    else:
        return input_ids


def get_hf_dataset(ds):
    def gen():
        for idx in range(len(ds)):
            inputs, targets = ds[idx]
            yield {"input_ids": inputs, "labels": targets}

    return datasets.IterableDataset.from_generator(gen)


def parse_args():
    parser = ArgumentParser(Config)
    train_config = parser.parse_args()
    log.info(train_config)
    return train_config


def modify_hip_args(model, turn_on, config):
    layer_idx = 0
    for m in model.modules():
        if isinstance(m, LlamaAttention):
            hip_attn_config = get_hip_config(config, layer_idx)
            if turn_on:
                hip_attn_config.using_extend = True
                hip_attn_config.need_apply_rope = True
            else:
                hip_attn_config.using_extend = False
                hip_attn_config.need_apply_rope = False

            m.hip_attn_args = hip_attn_config
            m.attention_method = config.method

            layer_idx += 1


def init_model(config: Config):
    device = "cuda:0"

    ALL_ATTENTION_FUNCTIONS.update({"hip_attention": (lambda x: x)})
    ALL_ATTENTION_FUNCTIONS.update({"sdpa_rectangle": (lambda x: x)})
    ALL_ATTENTION_FUNCTIONS.update({"recompute": (lambda x: x)})

    tokenizer = transformers.AutoTokenizer.from_pretrained(config.model)

    attn_implementation = os.environ.get("ATTN_IMPLEMENTATION", "hip_attention")

    model_config = LlamaConfig.from_pretrained(
        config.model,
        attn_implementation=attn_implementation,
        torch_dtype=torch.bfloat16,
    )

    model_config.pooler_method = config.pooler_method
    model_config.pooler_config = config.pooler_config
    model = LlamaForCausalLM.from_pretrained(
        config.model,
        config=model_config,
        torch_dtype=torch.bfloat16,
    )

    model.args = config

    layer_idx = 0
    for m in model.modules():
        if isinstance(m, LlamaAttention):
            m.args = config

            hip_attn_config = get_hip_config(config, layer_idx)
            if attn_implementation != "hip_attention":
                hip_attn_config.using_extend = False
                hip_attn_config.need_apply_rope = False
            m.hip_attn_args = hip_attn_config
            m.attention_method = config.method

            layer_idx += 1

    model._sample = types.MethodType(_sample, model)
    return model, tokenizer


def main(config: Config):
    if os.environ.get("DETECT_ANOMALY", "0") == "1":
        torch.autograd.set_detect_anomaly(True)

    # torch.set_float32_matmul_precision('high')
    torch.backends.cudnn.benchmark = True

    os.environ["WANDB_PROJECT"] = "quick_extend"

    if os.environ.get("CUDA_LAUNCH_BLOCKING", "0") == "1":
        log.info(
            "WARNING: CUDA_LAUNCH_BLOCKING is set to 1, this will slow down the training.",
        )

    filename = f'{config.model.replace("/", "_")}-{config.name}-{config.dataset}-{config.seq_len}-{config.get_hash()}'

    model, tokenizer = init_model(config=config)
    tokenizer.pad_token = tokenizer.eos_token

    model = model.cuda()
    evaluate(config, model, tokenizer, text, metric_key_prefix="test")


def run():
    seed()
    main(parse_args())


def seed(seed=42):
    import random

    import numpy as np

    torch.manual_seed(seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    np.random.seed(seed)
    random.seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)  # if use multi-GPU
    os.environ["PYTHONHASHSEED"] = str(seed)


text = """Call me Ishmael. Some years ago—never mind how long precisely—having
little or no money in my purse, and nothing particular to interest me
on shore, I thought I would sail about a little and see the watery part
of the world. It is a way I have of driving off the spleen and
regulating the circulation. Whenever I find myself growing grim about
the mouth; whenever it is a damp, drizzly November in my soul; whenever
I find myself involuntarily pausing before coffin warehouses, and
bringing up the rear of every funeral I meet; and especially whenever
my hypos get such an upper hand of me, that it requires a strong moral
principle to prevent me from deliberately stepping into the street, and
methodically knocking people’s hats off—then, I account it high time to
get to sea as soon as I can. This is my substitute for pistol and ball.
With a philosophical flourish Cato throws himself upon his sword; I
quietly take to the ship. There is nothing surprising in this. If they
but knew it, almost all men in their degree, some time or other,
cherish very nearly the same feelings towards the ocean with me.

There now is your insular city of the Manhattoes, belted round by
wharves as Indian isles by coral reefs—commerce surrounds it with her
surf. Right and left, the streets take you waterward. Its extreme
downtown is the battery, where that noble mole is washed by waves, and
cooled by breezes, which a few hours previous were out of sight of
land. Look at the crowds of water-gazers there.

Circumambulate the city of a dreamy Sabbath afternoon. Go from Corlears
Hook to Coenties Slip, and from thence, by Whitehall, northward. What
do you see?—Posted like silent sentinels all around the town, stand
thousands upon thousands of mortal men fixed in ocean reveries. Some
leaning against the spiles; some seated upon the pier-heads; some
looking over the bulwarks of ships from China; some high aloft in the
rigging, as if striving to get a still better seaward peep. But these
are all landsmen; of week days pent up in lath and plaster—tied to
counters, nailed to benches, clinched to desks. How then is this? Are
the green fields gone? What do they here?

But look! here come more crowds, pacing straight for the water, and
seemingly bound for a dive. Strange! Nothing will content them but the
extremest limit of the land; loitering under the shady lee of yonder
warehouses will not suffice. No. They must get just as nigh the water
as they possibly can without falling in. And there they stand—miles of
them—leagues. Inlanders all, they come from lanes and alleys, streets
and avenues—north, east, south, and west. Yet here they all unite. Tell
me, does the magnetic virtue of the needles of the compasses of all
those ships attract them thither?

Once more. Say you are in the country; in some high land of lakes. Take
almost any path you please, and ten to one it carries you down in a
dale, and leaves you there by a pool in the stream. There is magic in
it. Let the most absent-minded of men be plunged in his deepest
reveries—stand that man on his legs, set his feet a-going, and he will
infallibly lead you to water, if water there be in all that region.
Should you ever be athirst in the great American desert, try this
experiment, if your caravan happen to be supplied with a metaphysical
professor. Yes, as every one knows, meditation and water are wedded for
ever.

But here is an artist. He desires to paint you the dreamiest, shadiest,
quietest, most enchanting bit of romantic landscape in all the valley
of the Saco. What is the chief element he employs? There stand his
trees, each with a hollow trunk, as if a hermit and a crucifix were
within; and here sleeps his meadow, and there sleep his cattle; and up
from yonder cottage goes a sleepy smoke. Deep into distant woodlands
winds a mazy way, reaching to overlapping spurs of mountains bathed in
their hill-side blue. But though the picture lies thus tranced, and
though this pine-tree shakes down its sighs like leaves upon this
shepherd’s head, yet all were vain, unless the shepherd’s eye were
fixed upon the magic stream before him. Go visit the Prairies in June,
when for scores on scores of miles you wade knee-deep among
Tiger-lilies—what is the one charm wanting?—Water—there is not a drop
of water there! Were Niagara but a cataract of sand, would you travel
your thousand miles to see it? Why did the poor poet of Tennessee, upon
suddenly receiving two handfuls of silver, deliberate whether to buy
him a coat, which he sadly needed, or invest his money in a pedestrian
trip to Rockaway Beach? Why is almost every robust healthy boy with a
robust healthy soul in him, at some time or other crazy to go to sea?
Why upon your first voyage as a passenger, did you yourself feel such a
mystical vibration, when first told that you and your ship were now out
of sight of land? Why did the old Persians hold the sea holy? Why did
the Greeks give it a separate deity, and own brother of Jove? Surely
all this is not without meaning. And still deeper the meaning of that
story of Narcissus, who because he could not grasp the tormenting, mild
image he saw in the fountain, plunged into it and was drowned. But that
same image, we ourselves see in all rivers and oceans. It is the image
of the ungraspable phantom of life; and this is the key to it all.

Now, when I say that I am in the habit of going to sea whenever I begin
to grow hazy about the eyes, and begin to be over conscious of my
lungs, I do not mean to have it inferred that I ever go to sea as a
passenger. For to go as a passenger you must needs have a purse, and a
purse is but a rag unless you have something in it. Besides, passengers
get sea-sick—grow quarrelsome—don’t sleep of nights—do not enjoy
themselves much, as a general thing;—no, I never go as a passenger;
nor, though I am something of a salt, do I ever go to sea as a
Commodore, or a Captain, or a Cook. I abandon the glory and distinction
of such offices to those who like them. For my part, I abominate all
honorable respectable toils, trials, and tribulations of every kind
whatsoever. It is quite as much as I can do to take care of myself,
without taking care of ships, barques, brigs, schooners, and what not.
And as for going as cook,—though I confess there is considerable glory
in that, a cook being a sort of officer on ship-board—yet, somehow, I
never fancied broiling fowls;—though once broiled, judiciously
buttered, and judgmatically salted and peppered, there is no one who
will speak more respectfully, not to say reverentially, of a broiled
fowl than I will. It is out of the idolatrous dotings of the old
Egyptians upon broiled ibis and roasted river horse, that you see the
mummies of those creatures in their huge bake-houses the pyramids.

No, when I go to sea, I go as a simple sailor, right before the mast,
plumb down into the forecastle, aloft there to the royal mast-head.
True, they rather order me about some, and make me jump from spar to
spar, like a grasshopper in a May meadow. And at first, this sort of
thing is unpleasant enough. It touches one’s sense of honor,
particularly if you come of an old established family in the land, the
Van Rensselaers, or Randolphs, or Hardicanutes. And more than all, if
just previous to putting your hand into the tar-pot, you have been
lording it as a country schoolmaster, making the tallest boys stand in
awe of you. The transition is a keen one, I assure you, from a
schoolmaster to a sailor, and requires a strong decoction of Seneca and
the Stoics to enable you to grin and bear it. But even this wears off
in time.

What of it, if some old hunks of a sea-captain orders me to get a broom
and sweep down the decks? What does that indignity amount to, weighed,
I mean, in the scales of the New Testament? Do you think the archangel
Gabriel thinks anything the less of me, because I promptly and
respectfully obey that old hunks in that particular instance? Who ain’t
a slave? Tell me that. Well, then, however the old sea-captains may
order me about—however they may thump and punch me about, I have the
satisfaction of knowing that it is all right; that everybody else is
one way or other served in much the same way—either in a physical or
metaphysical point of view, that is; and so the universal thump is
passed round, and all hands should rub each other’s shoulder-blades,
and be content.

Again, I always go to sea as a sailor, because they make a point of
paying me for my trouble, whereas they never pay passengers a single
penny that I ever heard of. On the contrary, passengers themselves must
pay. And there is all the difference in the world between paying and
being paid. The act of paying is perhaps the most uncomfortable
infliction that the two orchard thieves entailed upon us. But _being
paid_,—what will compare with it? The urbane activity with which a man
receives money is really marvellous, considering that we so earnestly
believe money to be the root of all earthly ills, and that on no
account can a monied man enter heaven. Ah! how cheerfully we consign
ourselves to perdition!

Finally, I always go to sea as a sailor, because of the wholesome
exercise and pure air of the fore-castle deck. For as in this world,
head winds are far more prevalent than winds from astern (that is, if
you never violate the Pythagorean maxim), so for the most part the
Commodore on the quarter-deck gets his atmosphere at second hand from
the sailors on the forecastle. He thinks he breathes it first; but not
so. In much the same way do the commonalty lead their leaders in many
other things, at the same time that the leaders little suspect it. But
wherefore it was that after having repeatedly smelt the sea as a
merchant sailor, I should now take it into my head to go on a whaling
voyage; this the invisible police officer of the Fates, who has the
constant surveillance of me, and secretly dogs me, and influences me in
some unaccountable way—he can better answer than any one else. And,
doubtless, my going on this whaling voyage, formed part of the grand
programme of Providence that was drawn up a long time ago. It came in
as a sort of brief interlude and solo between more extensive
performances. I take it that this part of the bill must have run
something like this:

“_Grand Contested Election for the Presidency of the United States._
“WHALING VOYAGE BY ONE ISHMAEL. “BLOODY BATTLE IN AFFGHANISTAN.”

Though I cannot tell why it was exactly that those stage managers, the
Fates, put me down for this shabby part of a whaling voyage, when
others were set down for magnificent parts in high tragedies, and short
and easy parts in genteel comedies, and jolly parts in farces—though I
cannot tell why this was exactly; yet, now that I recall all the
circumstances, I think I can see a little into the springs and motives
which being cunningly presented to me under various disguises, induced
me to set about performing the part I did, besides cajoling me into the
delusion that it was a choice resulting from my own unbiased freewill
and discriminating judgment.

Chief among these motives was the overwhelming idea of the great whale
himself. Such a portentous and mysterious monster roused all my
curiosity. Then the wild and distant seas where he rolled his island
bulk; the undeliverable, nameless perils of the whale; these, with all
the attending marvels of a thousand Patagonian sights and sounds,
helped to sway me to my wish. With other men, perhaps, such things
would not have been inducements; but as for me, I am tormented with an
everlasting itch for things remote. I love to sail forbidden seas, and
land on barbarous coasts. Not ignoring what is good, I am quick to
perceive a horror, and could still be social with it—would they let
me—since it is but well to be on friendly terms with all the inmates of
the place one lodges in.

By reason of these things, then, the whaling voyage was welcome; the
great flood-gates of the wonder-world swung open, and in the wild
conceits that swayed me to my purpose, two and two there floated into
my inmost soul, endless processions of the whale, and, mid most of them
all, one grand hooded phantom, like a snow hill in the air.


CHAPTER 2. The Carpet-Bag.

I stuffed a shirt or two into my old carpet-bag, tucked it under my
arm, and started for Cape Horn and the Pacific. Quitting the good city
of old Manhatto, I duly arrived in New Bedford. It was a Saturday night
in December. Much was I disappointed upon learning that the little
packet for Nantucket had already sailed, and that no way of reaching
that place would offer, till the following Monday.

As most young candidates for the pains and penalties of whaling stop at
this same New Bedford, thence to embark on their voyage, it may as well
be related that I, for one, had no idea of so doing. For my mind was
made up to sail in no other than a Nantucket craft, because there was a
fine, boisterous something about everything connected with that famous
old island, which amazingly pleased me. Besides though New Bedford has
of late been gradually monopolising the business of whaling, and though
in this matter poor old Nantucket is now much behind her, yet Nantucket
was her great original—the Tyre of this Carthage;—the place where the
first dead American whale was stranded. Where else but from Nantucket
did those aboriginal whalemen, the Red-Men, first sally out in canoes
to give chase to the Leviathan? And where but from Nantucket, too, did
that first adventurous little sloop put forth, partly laden with
imported cobblestones—so goes the story—to throw at the whales, in
order to discover when they were nigh enough to risk a harpoon from the
bowsprit?

Now having a night, a day, and still another night following before me
in New Bedford, ere I could embark for my destined port, it became a
matter of concernment where I was to eat and sleep meanwhile. It was a
very dubious-looking, nay, a very dark and dismal night, bitingly cold
and cheerless. I knew no one in the place. With anxious grapnels I had
sounded my pocket, and only brought up a few pieces of silver,—So,
wherever you go, Ishmael, said I to myself, as I stood in the middle of
a dreary street shouldering my bag, and comparing the gloom towards the
north with the darkness towards the south—wherever in your wisdom you
may conclude to lodge for the night, my dear Ishmael, be sure to
inquire the price, and don’t be too particular.

With halting steps I paced the streets, and passed the sign of “The
Crossed Harpoons”—but it looked too expensive and jolly there. Further
on, from the bright red windows of the “Sword-Fish Inn,” there came
such fervent rays, that it seemed to have melted the packed snow and
ice from before the house, for everywhere else the congealed frost lay
ten inches thick in a hard, asphaltic pavement,—rather weary for me,
when I struck my foot against the flinty projections, because from
hard, remorseless service the soles of my boots were in a most
miserable plight. Too expensive and jolly, again thought I, pausing one
moment to watch the broad glare in the street, and hear the sounds of
the tinkling glasses within. But go on, Ishmael, said I at last; don’t
you hear? get away from before the door; your patched boots are
stopping the way. So on I went. I now by instinct followed the streets
that took me waterward, for there, doubtless, were the cheapest, if not
the cheeriest inns.

Such dreary streets! blocks of blackness, not houses, on either hand,
and here and there a candle, like a candle moving about in a tomb. At
this hour of the night, of the last day of the week, that quarter of
the town proved all but deserted. But presently I came to a smoky light
proceeding from a low, wide building, the door of which stood
invitingly open. It had a careless look, as if it were meant for the
uses of the public; so, entering, the first thing I did was to stumble
over an ash-box in the porch. Ha! thought I, ha, as the flying
particles almost choked me, are these ashes from that destroyed city,
Gomorrah? But “The Crossed Harpoons,” and “The Sword-Fish?”—this, then
must needs be the sign of “The Trap.” However, I picked myself up and
hearing a loud voice within, pushed on and opened a second, interior
door.

It seemed the great Black Parliament sitting in Tophet. A hundred black
faces turned round in their rows to peer; and beyond, a black Angel of
Doom was beating a book in a pulpit. It was a negro church; and the
preacher’s text was about the blackness of darkness, and the weeping
and wailing and teeth-gnashing there. Ha, Ishmael, muttered I, backing
out, Wretched entertainment at the sign of ‘The Trap!’

Moving on, I at last came to a dim sort of light not far from the
docks, and heard a forlorn creaking in the air; and looking up, saw a
swinging sign over the door with a white painting upon it, faintly
representing a tall straight jet of misty spray, and these words
underneath—“The Spouter Inn:—Peter Coffin.”

Coffin?—Spouter?—Rather ominous in that particular connexion, thought
I. But it is a common name in Nantucket, they say, and I suppose this
Peter here is an emigrant from there. As the light looked so dim, and
the place, for the time, looked quiet enough, and the dilapidated
little wooden house itself looked as if it might have been carted here
from the ruins of some burnt district, and as the swinging sign had a
poverty-stricken sort of creak to it, I thought that here was the very
spot for cheap lodgings, and the best of pea coffee.

It was a queer sort of place—a gable-ended old house, one side palsied
as it were, and leaning over sadly. It stood on a sharp bleak corner,
where that tempestuous wind Euroclydon kept up a worse howling than
ever it did about poor Paul’s tossed craft. Euroclydon, nevertheless,
is a mighty pleasant zephyr to any one in-doors, with his feet on the
hob quietly toasting for bed. “In judging of that tempestuous wind
called Euroclydon,” says an old writer—of whose works I possess the
only copy extant—“it maketh a marvellous difference, whether thou
lookest out at it from a glass window where the frost is all on the
outside, or whether thou observest it from that sashless window, where
the frost is on both sides, and of which the wight Death is the only
glazier.” True enough, thought I, as this passage occurred to my
mind—old black-letter, thou reasonest well. Yes, these eyes are
windows, and this body of mine is the house. What a pity they didn’t
stop up the chinks and the crannies though, and thrust in a little lint
here and there. But it’s too late to make any improvements now. The
universe is finished; the copestone is on, and the chips were carted
off a million years ago. Poor Lazarus there, chattering his teeth
against the curbstone for his pillow, and shaking off his tatters with
his shiverings, he might plug up both ears with rags, and put a
corn-cob into his mouth, and yet that would not keep out the
tempestuous Euroclydon. Euroclydon! says old Dives, in his red silken
wrapper—(he had a redder one afterwards) pooh, pooh! What a fine frosty
night; how Orion glitters; what northern lights! Let them talk of their
oriental summer climes of everlasting conservatories; give me the
privilege of making my own summer with my own coals.

But what thinks Lazarus? Can he warm his blue hands by holding them up
to the grand northern lights? Would not Lazarus rather be in Sumatra
than here? Would he not far rather lay him down lengthwise along the
line of the equator; yea, ye gods! go down to the fiery pit itself, in
order to keep out this frost?

Now, that Lazarus should lie stranded there on the curbstone before the
door of Dives, this is more wonderful than that an iceberg should be
moored to one of the Moluccas. Yet Dives himself, he too lives like a
Czar in an ice palace made of frozen sighs, and being a president of a
temperance society, he only drinks the tepid tears of orphans.

But no more of this blubbering now, we are going a-whaling, and there
is plenty of that yet to come. Let us scrape the ice from our frosted
feet, and see what sort of a place this “Spouter” may be.


CHAPTER 3. The Spouter-Inn.

Entering that gable-ended Spouter-Inn, you found yourself in a wide,
low, straggling entry with old-fashioned wainscots, reminding one of
the bulwarks of some condemned old craft. On one side hung a very large
oilpainting so thoroughly besmoked, and every way defaced, that in the
unequal crosslights by which you viewed it, it was only by diligent
study and a series of systematic visits to it, and careful inquiry of
the neighbors, that you could any way arrive at an understanding of its
purpose. Such unaccountable masses of shades and shadows, that at first
you almost thought some ambitious young artist, in the time of the New
England hags, had endeavored to delineate chaos bewitched. But by dint
of much and earnest contemplation, and oft repeated ponderings, and
especially by throwing open the little window towards the back of the
entry, you at last come to the conclusion that such an idea, however
wild, might not be altogether unwarranted.

But what most puzzled and confounded you was a long, limber,
portentous, black mass of something hovering in the centre of the
picture over three blue, dim, perpendicular lines floating in a
nameless yeast. A boggy, soggy, squitchy picture truly, enough to drive
a nervous man distracted. Yet was there a sort of indefinite,
half-attained, unimaginable sublimity about it that fairly froze you to
it, till you involuntarily took an oath with yourself to find out what
that marvellous painting meant. Ever and anon a bright, but, alas,
deceptive idea would dart you through.—It’s the Black Sea in a midnight
gale.—It’s the unnatural combat of the four primal elements.—It’s a
blasted heath.—It’s a Hyperborean winter scene.—It’s the breaking-up of
the icebound stream of Time. But at last all these fancies yielded to
that one portentous something in the picture’s midst. _That_ once found
out, and all the rest were plain. But stop; does it not bear a faint
resemblance to a gigantic fish? even the great leviathan himself?

In fact, the artist’s design seemed this: a final theory of my own,
partly based upon the aggregated opinions of many aged persons with
whom I conversed upon the subject. The picture represents a Cape-Horner
in a great hurricane; the half-foundered ship weltering there with its
three dismantled masts alone visible; and an exasperated whale,
purposing to spring clean over the craft, is in the enormous act of
impaling himself upon the three mast-heads.

The opposite wall of this entry was hung all over with a heathenish
array of monstrous clubs and spears. Some were thickly set with
glittering teeth resembling ivory saws; others were tufted with knots
of human hair; and one was sickle-shaped, with a vast handle sweeping
round like the segment made in the new-mown grass by a long-armed
mower. You shuddered as you gazed, and wondered what monstrous cannibal
and savage could ever have gone a death-harvesting with such a hacking,
horrifying implement. Mixed with these were rusty old whaling lances
and harpoons all broken and deformed. Some were storied weapons. With
this once long lance, now wildly elbowed, fifty years ago did Nathan
Swain kill fifteen whales between a sunrise and a sunset. And that
harpoon—so like a corkscrew now—was flung in Javan seas, and run away
with by a whale, years afterwards slain off the Cape of Blanco. The
original iron entered nigh the tail, and, like a restless needle
sojourning in the body of a man, travelled full forty feet, and at last
was found imbedded in the hump.

Crossing this dusky entry, and on through yon low-arched way—cut
through what in old times must have been a great central chimney with
fireplaces all round—you enter the public room. A still duskier place
is this, with such low ponderous beams above, and such old wrinkled
planks beneath, that you would almost fancy you trod some old craft’s
cockpits, especially of such a howling night, when this corner-anchored
old ark rocked so furiously. On one side stood a long, low, shelf-like
table covered with cracked glass cases, filled with dusty rarities
gathered from this wide world’s remotest nooks. Projecting from the
further angle of the room stands a dark-looking den—the bar—a rude
attempt at a right whale’s head. Be that how it may, there stands the
vast arched bone of the whale’s jaw, so wide, a coach might almost
drive beneath it. Within are shabby shelves, ranged round with old
decanters, bottles, flasks; and in those jaws of swift destruction,
like another cursed Jonah (by which name indeed they called him),
bustles a little withered old man, who, for their money, dearly sells
the sailors deliriums and death.

Abominable are the tumblers into which he pours his poison. Though true
cylinders without—within, the villanous green goggling glasses
deceitfully tapered downwards to a cheating bottom. Parallel meridians
rudely pecked into the glass, surround these footpads’ goblets. Fill to
_this_ mark, and your charge is but a penny; to _this_ a penny more;
and so on to the full glass—the Cape Horn measure, which you may gulp
down for a shilling.

Upon entering the place I found a number of young seamen gathered about
a table, examining by a dim light divers specimens of _skrimshander_. I
sought the landlord, and telling him I desired to be accommodated with
a room, received for answer that his house was full—not a bed
unoccupied. “But avast,” he added, tapping his forehead, “you haint no
objections to sharing a harpooneer’s blanket, have ye? I s’pose you are
goin’ a-whalin’, so you’d better get used to that sort of thing.”

I told him that I never liked to sleep two in a bed; that if I should
ever do so, it would depend upon who the harpooneer might be, and that
if he (the landlord) really had no other place for me, and the
harpooneer was not decidedly objectionable, why rather than wander
further about a strange town on so bitter a night, I would put up with
the half of any decent man’s blanket.

“I thought so. All right; take a seat. Supper?—you want supper?
Supper’ll be ready directly.”

I sat down on an old wooden settle, carved all over like a bench on the
Battery. At one end a ruminating tar was still further adorning it with
his jack-knife, stooping over and diligently working away at the space
between his legs. He was trying his hand at a ship under full sail, but
he didn’t make much headway, I thought.

At last some four or five of us were summoned to our meal in an
adjoining room. It was cold as Iceland—no fire at all—the landlord said
he couldn’t afford it. Nothing but two dismal tallow candles, each in a
winding sheet. We were fain to button up our monkey jackets, and hold
to our lips cups of scalding tea with our half frozen fingers. But the
fare was of the most substantial kind—not only meat and potatoes, but
dumplings; good heavens! dumplings for supper! One young fellow in a
green box coat, addressed himself to these dumplings in a most direful
manner.

“My boy,” said the landlord, “you’ll have the nightmare to a dead
sartainty.”

“Landlord,” I whispered, “that aint the harpooneer is it?”

“Oh, no,” said he, looking a sort of diabolically funny, “the
harpooneer is a dark complexioned chap. He never eats dumplings, he
don’t—he eats nothing but steaks, and he likes ’em rare.”

“The devil he does,” says I. “Where is that harpooneer? Is he here?”

“He’ll be here afore long,” was the answer.

I could not help it, but I began to feel suspicious of this “dark
complexioned” harpooneer. At any rate, I made up my mind that if it so
turned out that we should sleep together, he must undress and get into
bed before I did.

Supper over, the company went back to the bar-room, when, knowing not
what else to do with myself, I resolved to spend the rest of the
evening as a looker on.

Presently a rioting noise was heard without. Starting up, the landlord
cried, “That’s the Grampus’s crew. I seed her reported in the offing
this morning; a three years’ voyage, and a full ship. Hurrah, boys; now
we’ll have the latest news from the Feegees.”

A tramping of sea boots was heard in the entry; the door was flung
open, and in rolled a wild set of mariners enough. Enveloped in their
shaggy watch coats, and with their heads muffled in woollen comforters,
all bedarned and ragged, and their beards stiff with icicles, they
seemed an eruption of bears from Labrador. They had just landed from
their boat, and this was the first house they entered. No wonder, then,
that they made a straight wake for the whale’s mouth—the bar—when the
wrinkled little old Jonah, there officiating, soon poured them out
brimmers all round. One complained of a bad cold in his head, upon
which Jonah mixed him a pitch-like potion of gin and molasses, which he
swore was a sovereign cure for all colds and catarrhs whatsoever, never
mind of how long standing, or whether caught off the coast of Labrador,
or on the weather side of an ice-island.

The liquor soon mounted into their heads, as it generally does even
with the arrantest topers newly landed from sea, and they began
capering about most obstreperously.

I observed, however, that one of them held somewhat aloof, and though
he seemed desirous not to spoil the hilarity of his shipmates by his
own sober face, yet upon the whole he refrained from making as much
noise as the rest. This man interested me at once; and since the
sea-gods had ordained that he should soon become my shipmate (though
but a sleeping-partner one, so far as this narrative is concerned), I
will here venture upon a little description of him. He stood full six
feet in height, with noble shoulders, and a chest like a coffer-dam. I
have seldom seen such brawn in a man. His face was deeply brown and
burnt, making his white teeth dazzling by the contrast; while in the
deep shadows of his eyes floated some reminiscences that did not seem
to give him much joy. His voice at once announced that he was a
Southerner, and from his fine stature, I thought he must be one of
those tall mountaineers from the Alleghanian Ridge in Virginia. When
the revelry of his companions had mounted to its height, this man
slipped away unobserved, and I saw no more of him till he became my
comrade on the sea. In a few minutes, however, he was missed by his
shipmates, and being, it seems, for some reason a huge favourite with
them, they raised a cry of “Bulkington! Bulkington! where’s
Bulkington?” and darted out of the house in pursuit of him.

It was now about nine o’clock, and the room seeming almost
supernaturally quiet after these orgies, I began to congratulate myself
upon a little plan that had occurred to me just previous to the
entrance of the seamen.

No man prefers to sleep two in a bed. In fact, you would a good deal
rather not sleep with your own brother. I don’t know how it is, but
people like to be private when they are sleeping. And when it comes to
sleeping with an unknown stranger, in a strange inn, in a strange town,
and that stranger a harpooneer, then your objections indefinitely
multiply. Nor was there any earthly reason why I as a sailor should
sleep two in a bed, more than anybody else; for sailors no more sleep
two in a bed at sea, than bachelor Kings do ashore. To be sure they all
sleep together in one apartment, but you have your own hammock, and
cover yourself with your own blanket, and sleep in your own skin.

The more I pondered over this harpooneer, the more I abominated the
thought of sleeping with him. It was fair to presume that being a
harpooneer, his linen or woollen, as the case might be, would not be of
the tidiest, certanno_1y none of the finest. I began to twitch all over.
Besides, it was getting late, and my decent harpooneer ought to be home
and going bedwards. Suppose now, he should tumble in upon me at
midnight—how could I tell from what vile hole he had been coming?

“Landlord! I’ve changed my mind about that harpooneer.—I shan’t sleep
with him. I’ll try the bench here.”

“Just as you please; I’m sorry I can’t spare ye a tablecloth for a
mattress, and it’s a plaguy rough board here”—feeling of the knots and
notches. “But wait a bit, Skrimshander; I’ve got a carpenter’s plane
there in the bar—wait, I say, and I’ll make ye snug enough.” So saying
he procured the plane; and with his old silk handkerchief first dusting
the bench, vigorously set to planing away at my bed, the while grinning
like an ape. The shavings flew right and left; till at last the
plane-iron came bump against an indestructible knot. The landlord was
near spraining his wrist, and I told him for heaven’s sake to quit—the
bed was soft enough to suit me, and I did not know how all the planing
in the world could make eider down of a pine plank. So gathering up the
shavings with another grin, and throwing them into the great stove in
the middle of the room, he went about his business, and left me in a
brown study.

I now took the measure of the bench, and found that it was a foot too
short; but that could be mended with a chair. But it was a foot too
narrow, and the other bench in the room was about four inches higher
than the planed one—so there was no yoking them. I then placed the
first bench lengthwise along the only clear space against the wall,
leaving a little interval between, for my back to settle down in. But I
soon found that there came such a draught of cold air over me from
under the sill of the window, that this plan would never do at all,
especially as another current from the rickety door met the one from
the window, and both together formed a series of small whirlwinds in
the immediate vicinity of the spot where I had thought to spend the
night.

The devil fetch that harpooneer, thought I, but stop, couldn’t I steal
a march on him—bolt his door inside, and jump into his bed, not to be
wakened by the most violent knockings? It seemed no bad idea; but upon
second thoughts I dismissed it. For who could tell but what the next
morning, so soon as I popped out of the room, the harpooneer might be
standing in the entry, all ready to knock me down!

Still, looking round me again, and seeing no possible chance of
spending a sufferable night unless in some other person’s bed, I began
to think that after all I might be cherishing unwarrantable prejudices
against this unknown harpooneer. Thinks I, I’ll wait awhile; he must be
dropping in before long. I’ll have a good look at him then, and perhaps
we may become jolly good bedfellows after all—there’s no telling.

But though the other boarders kept coming in by ones, twos, and threes,
and going to bed, yet no sign of my harpooneer.

“Landlord!” said I, “what sort of a chap is he—does he always keep such
late hours?” It was now hard upon twelve o’clock.

The landlord chuckled again with his lean chuckle, and seemed to be
mightily tickled at something beyond my comprehension. “No,” he
answered, “generally he’s an early bird—airley to bed and airley to
rise—yes, he’s the bird what catches the worm. But to-night he went out
a peddling, you see, and I don’t see what on airth keeps him so late,
unless, may be, he can’t sell his head.”

“Can’t sell his head?—What sort of a bamboozingly story is this you are
telling me?” getting into a towering rage. “Do you pretend to say,
landlord, that this harpooneer is actually engaged this blessed
Saturday night, or rather Sunday morning, in peddling his head around
this town?”

“That’s precisely it,” said the landlord, “and I told him he couldn’t
sell it here, the market’s overstocked.”

“With what?” shouted I.

“With heads to be sure; ain’t there too many heads in the world?”

“I tell you what it is, landlord,” said I quite calmly, “you’d better
stop spinning that yarn to me—I’m not green.”

“May be not,” taking out a stick and whittling a toothpick, “but I
rayther guess you’ll be done _brown_ if that ere harpooneer hears you a
slanderin’ his head.”

“I’ll break it for him,” said I, now flying into a passion again at
this unaccountable farrago of the landlord’s.

“It’s broke a’ready,” said he.

“Broke,” said I—“_broke_, do you mean?”

“Sartain, and that’s the very reason he can’t sell it, I guess.”

“Landlord,” said I, going up to him as cool as Mt. Hecla in a
snow-storm—“landlord, stop whittling. You and I must understand one
another, and that too without delay. I come to your house and want a
bed; you tell me you can only give me half a one; that the other half
belongs to a certain harpooneer. And about this harpooneer, whom I have
not yet seen, you persist in telling me the most mystifying and
exasperating stories tending to beget in me an uncomfortable feeling
towards the man whom you design for my bedfellow—a sort of connexion,
landlord, which is an intimate and confidential one in the highest
degree. I now demand of you to speak out and tell me who and what this
harpooneer is, and whether I shall be in all respects safe to spend the
night with him. And in the first place, you will be so good as to unsay
that story about selling his head, which if true I take to be good
evidence that this harpooneer is stark mad, and I’ve no idea of
sleeping with a madman; and you, sir, _you_ I mean, landlord, _you_,
sir, by trying to induce me to do so knowingly, would thereby render
yourself liable to a criminal prosecution.”

“Wall,” said the landlord, fetching a long breath, “that’s a purty long
sarmon for a chap that rips a little now and then. But be easy, be
easy, this here harpooneer I have been tellin’ you of has just arrived
from the south seas, where he bought up a lot of ’balmed New Zealand
heads (great curios, you know), and he’s sold all on ’em but one, and
that one he’s trying to sell to-night, cause to-morrow’s Sunday, and it
would not do to be sellin’ human heads about the streets when folks is
goin’ to churches. He wanted to, last Sunday, but I stopped him just as
he was goin’ out of the door with four heads strung on a string, for
all the airth like a string of inions.”

This account cleared up the otherwise unaccountable mystery, and showed
that the landlord, after all, had had no idea of fooling me—but at the
same time what could I think of a harpooneer who stayed out of a
Saturday night clean into the holy Sabbath, engaged in such a cannibal
business as selling the heads of dead idolators?

“Depend upon it, landlord, that harpooneer is a dangerous man.”

“He pays reg’lar,” was the rejoinder. “But come, it’s getting dreadful
late, you had better be turning flukes—it’s a nice bed; Sal and me
slept in that ere bed the night we were spliced. There’s plenty of room
for two to kick about in that bed; it’s an almighty big bed that. Why,
afore we give it up, Sal used to put our Sam and little Johnny in the
foot of it. But I got a dreaming and sprawling about one night, and
somehow, Sam got pitched on the floor, and came near breaking his arm.
Arter that, Sal said it wouldn’t do. Come along here, I’ll give ye a
glim in a jiffy;” and so saying he lighted a candle and held it towards
me, offering to lead the way. But I stood irresolute; when looking at a
clock in the corner, he exclaimed “I vum it’s Sunday—you won’t see that
harpooneer to-night; he’s come to anchor somewhere—come along then;
_do_ come; _won’t_ ye come?”

I considered the matter a moment, and then up stairs we went, and I was
ushered into a small room, cold as a clam, and furnished, sure enough,
with a prodigious bed, almost big enough indeed for any four
harpooneers to sleep abreast.

“There,” said the landlord, placing the candle on a crazy old sea chest
that did double duty as a wash-stand and centre table; “there, make
yourself comfortable now, and good night to ye.” I turned round from
eyeing the bed, but he had disappeared.

Folding back the counterpane, I stooped over the bed. Though none of
the most elegant, it yet stood the scrutiny tolerably well. I then
glanced round the room; and besides the bedstead and centre table,
could see no other furniture belonging to the place, but a rude shelf,
the four walls, and a papered fireboard representing a man striking a
whale. Of things not properly belonging to the room, there was a
hammock lashed up, and thrown upon the floor in one corner; also a
large seaman’s bag, containing the harpooneer’s wardrobe, no doubt in
lieu of a land trunk. Likewise, there was a parcel of outlandish bone
fish hooks on the shelf over the fire-place, and a tall harpoon
standing at the head of the bed.

But what is this on the chest? I took it up, and held it close to the
light, and felt it, and smelt it, and tried every way possible to
arrive at some satisfactory conclusion concerning it. I can compare it
to nothing but a large door mat, ornamented at the edges with little
tinkling tags something like the stained porcupine quills round an
Indian moccasin. There was a hole or slit in the middle of this mat, as
you see the same in South American ponchos. But could it be possible
that any sober harpooneer would get into a door mat, and parade the
streets of any Christian town in that sort of guise? I put it on, to
try it, and it weighed me down like a hamper, being uncommonly shaggy
and thick, and I thought a little damp, as though this mysterious
harpooneer had been wearing it of a rainy day. I went up in it to a bit
of glass stuck against the wall, and I never saw such a sight in my
life. I tore myself out of it in such a hurry that I gave myself a kink
in the neck.

I sat down on the side of the bed, and commenced thinking about this
head-peddling harpooneer, and his door mat. After thinking some time on
the bed-side, I got up and took off my monkey jacket, and then stood in
the middle of the room thinking. I then took off my coat, and thought a
little more in my shirt sleeves. But beginning to feel very cold now,
half undressed as I was, and remembering what the landlord said about
the harpooneer’s not coming home at all that night, it being so very
late, I made no more ado, but jumped out of my pantaloons and boots,
and then blowing out the light tumbled into bed, and commended myself
to the care of heaven.

Whether that mattress was stuffed with corn-cobs or broken crockery,
there is no telling, but I rolled about a good deal, and could not
sleep for a long time. At last I slid off into a light doze, and had
pretty nearly made a good offing towards the land of Nod, when I heard
a heavy footfall in the passage, and saw a glimmer of light come into
the room from under the door.

Lord save me, thinks I, that must be the harpooneer, the infernal
head-peddler. But I lay perfectly still, and resolved not to say a word
till spoken to. Holding a light in one hand, and that identical New
Zealand head in the other, the stranger entered the room, and without
looking towards the bed, placed his candle a good way off from me on
the floor in one corner, and then began working away at the knotted
cords of the large bag I before spoke of as being in the room. I was
all eagerness to see his face, but he kept it averted for some time
while employed in unlacing the bag’s mouth. This accomplished, however,
he turned round—when, good heavens! what a sight! Such a face! It was
of a dark, purplish, yellow colour, here and there stuck over with
large blackish looking squares. Yes, it’s just as I thought, he’s a
terrible bedfellow; he’s been in a fight, got dreadfully cut, and here
he is, just from the suranno_7. But at that moment he chanced to turn his
face so towards the light, that I planno_1y saw they could not be
sticking-plasters at all, those black squares on his cheeks. They were
stains of some sort or other. At first I knew not what to make of this;
but soon an inkling of the truth occurred to me. I remembered a story
of a white man—a whaleman too—who, falling among the cannibals, had
been tattooed by them. I concluded that this harpooneer, in the course
of his distant voyages, must have met with a similar adventure. And
what is it, thought I, after all! It’s only his outside; a man can be
honest in any sort of skin. But then, what to make of his unearthly
complexion, that part of it, I mean, lying round about, and completely
independent of the squares of tattooing. To be sure, it might be
nothing but a good coat of tropical tanning; but I never heard of a hot
sun’s tanning a white man into a purplish yellow one. However, I had
never been in the South Seas; and perhaps the sun there produced these
extraordinary effects upon the skin. Now, while all these ideas were
passing through me like lightning, this harpooneer never noticed me at
all. But, after some difficulty having opened his bag, he commenced
fumbling in it, and presently pulled out a sort of tomahawk, and a
seal-skin wallet with the hair on. Placing these on the old chest in
the middle of the room, he then took the New Zealand head—a ghastly
thing enough—and crammed it down into the bag. He now took off his
hat—a new beaver hat—when I came nigh singing out with fresh surprise.
There was no hair on his head—none to speak of at least—nothing but a
small scalp-knot twisted up on his forehead. His bald purplish head now
looked for all the world like a mildewed skull. Had not the stranger
stood between me and the door, I would have bolted out of it quicker
than ever I bolted a dinner.

Even as it was, I thought something of slipping out of the window, but
it was the second floor back. I am no coward, but what to make of this
head-peddling purple rascal altogether passed my comprehension.
Ignorance is the parent of fear, and being completely nonplussed and
confounded about the stranger, I confess I was now as much afraid of
him as if it was the devil himself who had thus broken into my room at
the dead of night. In fact, I was so afraid of him that I was not game
enough just then to address him, and demand a satisfactory answer
concerning what seemed inexplicable in him.

Meanwhile, he continued the business of undressing, and at last showed
his chest and arms. As I live, these covered parts of him were
checkered with the same squares as his face; his back, too, was all
over the same dark squares; he seemed to have been in a Thirty Years’
War, and just escaped from it with a sticking-plaster shirt. Still
more, his very legs were marked, as if a parcel of dark green frogs
were running up the trunks of young palms. It was now quite plain that
he must be some abominable savage or other shipped aboard of a whaleman
in the South Seas, and so landed in this Christian country. I quaked to
think of it. A peddler of heads too—perhaps the heads of his own
brothers. He might take a fancy to mine—heavens! look at that tomahawk!

But there was no time for shuddering, for now the savage went about
something that completely fascinated my attention, and convinced me
that he must indeed be a heathen. Going to his heavy grego, or wrapall,
or dreadnaught, which he had previously hung on a chair, he fumbled in
the pockets, and produced at length a curious little deformed image
with a hunch on its back, and exactly the colour of a three days’ old
Congo baby. Remembering the embalmed head, at first I almost thought
that this black manikin was a real baby preserved in some similar
manner. But seeing that it was not at all limber, and that it glistened
a good deal like polished ebony, I concluded that it must be nothing
but a wooden idol, which indeed it proved to be. For now the savage
goes up to the empty fire-place, and removing the papered fire-board,
sets up this little hunch-backed image, like a tenpin, between the
andirons. The chimney jambs and all the bricks inside were very sooty,
so that I thought this fire-place made a very appropriate little shrine
or chapel for his Congo idol.

I now screwed my eyes hard towards the half hidden image, feeling but
ill at ease meantime—to see what was next to follow. First he takes
about a double handful of shavings out of his grego pocket, and places
them carefully before the idol; then laying a bit of ship biscuit on
top and applying the flame from the lamp, he kindled the shavings into
a sacrificial blaze. Presently, after many hasty snatches into the
fire, and still hastier withdrawals of his fingers (whereby he seemed
to be scorching them badly), he at last succeeded in drawing out the
biscuit; then blowing off the heat and ashes a little, he made a polite
offer of it to the little negro. But the little devil did not seem to
fancy such dry sort of fare at all; he never moved his lips. All these
strange antics were accompanied by still stranger guttural noises from
the devotee, who seemed to be praying in a sing-song or else singing
some pagan psalmody or other, during which his face twitched about in
the most unnatural manner. At last extinguishing the fire, he took the
idol up very unceremoniously, and bagged it again in his grego pocket
as carelessly as if he were a sportsman bagging a dead woodcock.

All these queer proceedings increased my uncomfortableness, and seeing
him now exhibiting strong symptoms of concluding his business
operations, and jumping into bed with me, I thought it was high time,
now or never, before the light was put out, to break the spell in which
I had so long been bound.

But the interval I spent in deliberating what to say, was a fatal one.
Taking up his tomahawk from the table, he examined the head of it for
an instant, and then holding it to the light, with his mouth at the
handle, he puffed out great clouds of tobacco smoke. The next moment
the light was extinguished, and this wild cannibal, tomahawk between
his teeth, sprang into bed with me. I sang out, I could not help it
now; and giving a sudden grunt of astonishment he began feeling me.

Stammering out something, I knew not what, I rolled away from him
against the wall, and then conjured him, whoever or whatever he might
be, to keep quiet, and let me get up and light the lamp again. But his
guttural responses satisfied me at once that he but ill comprehended my
meaning.

“Who-e debel you?”—he at last said—“you no speak-e, dam-me, I kill-e.”
And so saying the lighted tomahawk began flourishing about me in the
dark.

“Landlord, for God’s sake, Peter Coffin!” shouted I. “Landlord! Watch!
Coffin! Angels! save me!”

“Speak-e! tell-ee me who-ee be, or dam-me, I kill-e!” again growled the
cannibal, while his horrid flourishings of the tomahawk scattered the
hot tobacco ashes about me till I thought my linen would get on fire.
But thank heaven, at that moment the landlord came into the room light
in hand, and leaping from the bed I ran up to him.

“Don’t be afraid now,” said he, grinning again, “Queequeg here wouldn’t
harm a hair of your head.”

“Stop your grinning,” shouted I, “and why didn’t you tell me that that
infernal harpooneer was a cannibal?”

“I thought ye know’d it;—didn’t I tell ye, he was a peddlin’ heads
around town?—but turn flukes again and go to sleep. Queequeg, look
here—you sabbee me, I sabbee—you this man sleepe you—you sabbee?”

“Me sabbee plenty”—grunted Queequeg, puffing away at his pipe and
sitting up in bed.

“You gettee in,” he added, motioning to me with his tomahawk, and
throwing the clothes to one side. He really did this in not only a
civil but a really kind and charitable way. I stood looking at him a
moment. For all his tattooings he was on the whole a clean, comely
looking cannibal. What’s all this fuss I have been making about,
thought I to myself—the man’s a human being just as I am: he has just
as much reason to fear me, as I have to be afraid of him. Better sleep
with a sober cannibal than a drunken Christian.

“Landlord,” said I, “tell him to stash his tomahawk there, or pipe, or
whatever you call it; tell him to stop smoking, in short, and I will
turn in with him. But I don’t fancy having a man smoking in bed with
me. It’s dangerous. Besides, I ain’t insured.”

This being told to Queequeg, he at once complied, and again politely
motioned me to get into bed—rolling over to one side as much as to
say—“I won’t touch a leg of ye.”

“Good night, landlord,” said I, “you may go.”

I turned in, and never slept better in my life."""

if __name__ == "__main__":
    run()
