import os
import json
import pandas as pd

def read_and_aggregate_results(base_path):
    all_results = []
    file_counts = {}
    
    for subdir in os.listdir(base_path):
        subdir_path = os.path.join(base_path, subdir)
        if os.path.isdir(subdir_path):
            try:
                tokens, model = subdir.split('_', 1)
                tokens = int(tokens)
            except (ValueError, TypeError):
                tokens = -1
                model = subdir
            if tokens == -1:
                continue

            for result_file in os.listdir(subdir_path):
                result_file_path = os.path.join(subdir_path, result_file)
                if result_file.endswith('.json'):  
                    idd = result_file.split('_')[0]


                        
                    file_counts.setdefault((model, tokens), 0)
                    file_counts[(model, tokens)] += 1

                    with open(result_file_path, 'r') as file:
                        data = json.load(file)
                        score_data = data.get("Score", {})
                        issue_origin = score_data.get("issue_origin", {})
                        issue_message = score_data.get("issue_message", {})
                        issue_ground = score_data.get("issue_ground", {})

                        
                        def get_or_default(d, key):
                            return d.get(key, 0) if isinstance(d, dict) and d.get(key) is not None and isinstance(d.get(key), (int, float)) else 0

                        issues = {
                            'issue_origin': issue_origin,
                            'issue_message': issue_message,
                            'issue_ground': issue_ground
                        }
                        for issue_name, issue in issues.items():
                            
                            reproducibility = get_or_default(issue, "Reproducibility")
                            relevance = get_or_default(issue, "Relevance")
                            explanation = get_or_default(issue, "Explanation")


                            result = {
                                "Model": score_data.get("model", model),
                                "Tokens": tokens,
                                "Difficulty": score_data.get("Difficulty", ""),
                                "FileName": result_file,
                                "Issue": issue_name,

                                "Reproducibility": reproducibility,
                                "Relevance": relevance,
                                "Explanation": explanation,
                                "Overall": get_or_default(issue, "Overall"),
                            }
                            all_results.append(result)

    
    aggregated_df = pd.DataFrame(all_results)

    
    mean_values = aggregated_df.groupby(['Model', 'Tokens', 'Issue']).mean(numeric_only=True).reset_index()
    
    file_counts_df = pd.DataFrame([
        {"Model": model, "Tokens": tokens, "FileCount": count}
        for (model, tokens), count in file_counts.items()
    ])
    file_counts_df.to_csv('file_counts_by_model_and_token.csv', index=False)
    return aggregated_df, mean_values

def main():
    base_path = 'test_result_new'  
    aggregated_results, mean_values = read_and_aggregate_results(base_path)

    
    aggregated_results.to_csv('aggregated_results.csv', index=False)
    
    mean_values.to_csv('average_results_Issue_by_model_and_token.csv', index=False)
    
    print("Results aggregated and saved to 'aggregated_results.csv'")
    print("Average values calculated and saved to 'average_results_by_model_and_token.csv'")

if __name__ == "__main__":
    main()
