from dataclasses import asdict
from datetime import datetime, timezone
from typing import Any, ClassVar, Dict
from uuid import UUID, uuid4

from pydantic import BaseModel, ConfigDict, Field

from src.messages import BaseMessage, FunctionCallingMessage, OpenAIMessage
from src.types import RoleType, OpenAIBackendRole


class MemoryRecord(BaseModel):
    r"""Basic message record used by the memory layer.

    Attributes:
        message (BaseMessage): The stored message content.
        role_at_backend (OpenAIBackendRole): Backend-facing role associated
            with this message (e.g., for OpenAI-compatible backends). This is
            distinct from any application-level :obj:`RoleType`.
        uuid (UUID, optional): Unique identifier for this record. Defaults to a
            randomly generated UUID.
        extra_info (Dict[str, str], optional): Free-form metadata. Defaults to
            an empty dict.
        timestamp (float, optional): Creation timestamp in seconds since epoch (UTC).
        agent_id (str): Identifier of the agent/process that produced this message.
    """

    model_config = ConfigDict(arbitrary_types_allowed=True)

    message: BaseMessage
    role_at_backend: OpenAIBackendRole
    uuid: UUID = Field(default_factory=uuid4)
    extra_info: Dict[str, str] = Field(default_factory=dict)
    timestamp: float = Field(
        default_factory=lambda: datetime.now(timezone.utc).timestamp()
    )
    agent_id: str = Field(default="")

    _MESSAGE_TYPES: ClassVar[dict] = {
        "BaseMessage": BaseMessage,
        "FunctionCallingMessage": FunctionCallingMessage,
    }

    @classmethod
    def from_dict(cls, record_dict: Dict[str, Any]) -> "MemoryRecord":
        r"""Reconstruct a :obj:`MemoryRecord` from a serialized dict.

        Args:
            record_dict (Dict[str, Any]): A dict generated by :meth:`to_dict`.
        """
        message_cls = cls._MESSAGE_TYPES[record_dict["message"]["__class__"]]
        kwargs: Dict = record_dict["message"].copy()
        kwargs.pop("__class__")
        reconstructed_message = message_cls(**kwargs)
        return cls(
            uuid=UUID(record_dict["uuid"]),
            message=reconstructed_message,
            role_at_backend=record_dict["role_at_backend"],
            extra_info=record_dict["extra_info"],
            timestamp=record_dict["timestamp"],
            agent_id=record_dict["agent_id"],
        )

    def to_dict(self) -> Dict[str, Any]:
        r"""Serialize this record to a dict."""
        return {
            "uuid": str(self.uuid),
            "message": {
                "__class__": self.message.__class__.__name__,
                **asdict(self.message),
            },
            "role_at_backend": self.role_at_backend,
            "extra_info": self.extra_info,
            "timestamp": self.timestamp,
            "agent_id": self.agent_id,
        }

    def to_openai_message(self) -> OpenAIMessage:
        r"""Convert to an :obj:`OpenAIMessage` for OpenAI-compatible backends."""
        return self.message.to_openai_message(self.role_at_backend)


class ContextRecord(BaseModel):
    r"""Container for a retrieved memory item along with similarity metadata."""

    memory_record: MemoryRecord
    score: float
    timestamp: float = Field(
        default_factory=lambda: datetime.now(timezone.utc).timestamp()
    )
