import json
from .base_agent import BaseAgent
from tools import get_tool
from utils import safe_llm_call


class DummyAgent(BaseAgent):
    """
    An LLM agent designed to answer questions about patient EHR data using tool calling.
    Uses built-in dummy FHIR data for testing and demonstration purposes.
    """

    # Dummy FHIR data for testing
    DUMMY_FHIR_DATA = {
        "patient": {
            "resourceType": "Patient",
            "id": "example",
            "name": [{"given": ["John"], "family": ["Doe"]}],
            "gender": "male",
            "birthDate": "1980-01-15",
            "address": [{"line": ["123 Main St"], "city": "Anytown", "state": "CA", "postalCode": "90210"}],
            "telecom": [{"system": "phone", "value": "555-123-4567", "use": "home"}]
        },
        "conditions": [
            {
                "resourceType": "Condition",
                "id": "condition1",
                "code": {"text": "Hypertension"},
                "subject": {"reference": "Patient/example"},
                "onsetDateTime": "2020-03-10"
            },
            {
                "resourceType": "Condition",
                "id": "condition2",
                "code": {"text": "Type 2 Diabetes Mellitus"},
                "subject": {"reference": "Patient/example"},
                "onsetDateTime": "2021-06-01"
            }
        ],
        "observations": [
            {
                "resourceType": "Observation",
                "id": "observation1",
                "code": {"text": "Blood Pressure"},
                "subject": {"reference": "Patient/example"},
                "valueQuantity": {"value": 140, "unit": "mmHg", "code": "mmHg"},
                "referenceRange": [{"high": {"value": 120, "unit": "mmHg"}}],
                "effectiveDateTime": "2023-01-20"
            },
            {
                "resourceType": "Observation",
                "id": "observation2",
                "code": {"text": "Hemoglobin A1c"},
                "subject": {"reference": "Patient/example"},
                "valueQuantity": {"value": 7.5, "unit": "%", "code": "%"},
                "referenceRange": [{"high": {"value": 6.5, "unit": "%"}}],
                "effectiveDateTime": "2023-01-20"
            }
        ],
        "medications": [
            {
                "resourceType": "MedicationRequest",
                "id": "medication1",
                "medicationCodeableConcept": {"text": "Lisinopril"},
                "subject": {"reference": "Patient/example"},
                "dosageInstruction": [{"text": "10 mg daily"}]
            },
            {
                "resourceType": "MedicationRequest",
                "id": "medication2",
                "medicationCodeableConcept": {"text": "Metformin"},
                "subject": {"reference": "Patient/example"},
                "dosageInstruction": [{"text": "500 mg twice daily"}]
            }
        ]
    }

    def __init__(self, model: str, verbose: bool = False, base_url=None):
        super().__init__(model, verbose, base_url)

        system_prompt = """
You are a helpful assistant that can answer questions about patient EHR data.
You can use the following tools to get the patient's EHR data:
- get_sample_fhir_data: Retrieves Electronic Health Record (EHR) data for a given type.

If you cannot find the answer or don't have enough information, clearly state 'I cannot answer this question' in your response.
"""

        self.messages = [{"role": "system", "content": system_prompt}]

        # Define the internal tool
        self.tools = [
            {
                "type": "function",
                "function": {
                    "name": "get_sample_fhir_data",
                    "description": "Retrieves Electronic Health Record (EHR) data for a given type.",
                    "parameters": {
                        "type": "object",
                        "properties": {
                            "data_type": {
                                "type": "string",
                                "description": "The type of EHR data to retrieve (e.g., 'patient', 'conditions', 'observations', 'medications')."
                            }
                        },
                        "required": ["data_type"]
                    }
                }
            }
        ]


    def get_sample_fhir_data(self, data_type: str) -> str:
        """
        Retrieves dummy Electronic Health Record (EHR) data for a given type.
        This method simulates fetching data from an EHR system.

        Args:
            data_type (str): The type of EHR data to retrieve (e.g., "patient", "conditions", "observations", "medications").

        Returns:
            str: A JSON string of the requested dummy FHIR resources.
        """
        if data_type in self.DUMMY_FHIR_DATA:
            return json.dumps({data_type: self.DUMMY_FHIR_DATA[data_type]}, indent=2)
        else:
            return json.dumps({"error": f"Data type '{data_type}' not found. Available types: patient, conditions, observations, medications"})

    def run(self, question: str) -> str:
        """
        Processes a user query about EHR data using the LLM and tool calling.

        The agent interacts with the LLM, detects tool calls, executes them,
        and feeds the tool's output back to the LLM to generate a final answer.

        Args:
            question (str): The question about the patient's EHR data.

        Returns:
            str: The LLM agent's answer based on retrieved EHR data.
        """

        # Add user message to chat history
        self.messages.append({"role": "user", "content": question})

        # Send the user's question to the model with tools enabled
        response_message, error, usage_info = safe_llm_call(
            model=self.model,
            messages=self.messages,
            tools=self.tools,
            base_url=self.base_url
        )

        # Track usage for the first call
        self._update_usage(usage_info)

        if error:
            final_answer = f"Error: {error}"

        if response_message.tool_calls:
            tool_call = response_message.tool_calls[0]
            tool_name = tool_call.function.name
            tool_args = json.loads(tool_call.function.arguments)

            self.messages.append(response_message)
            
            if self.verbose:
                print(f"Agent detected tool call: '{tool_name}' with arguments: {tool_args}")
                
            try:
                if tool_name == "get_sample_fhir_data":
                    tool_output = self.get_sample_fhir_data(**tool_args)

                    if self.verbose:
                        print(f"Tool execution successful. Output snippet: {str(tool_output)[:150]}...")
                    
                    self.messages.append(
                        {
                            "tool_call_id": tool_call.id,
                            "role": "tool",
                            "name": tool_name,
                            "content": tool_output,
                        }
                    )
                    
            except Exception as e:
                error_message = f"Error executing tool '{tool_name}': {e}"
                if self.verbose:
                    print(error_message)
                # Inform the model about the error
                self.messages.append(
                    {
                        "tool_call_id": tool_call.id,
                        "role": "tool",
                        "name": tool_name,
                        "content": {"error": error_message},
                    }
                )                    

        final_message, final_error, final_usage_info = safe_llm_call(
            model=self.model,
            messages=self.messages
        )
        
        self._update_usage(final_usage_info)
        
        final_answer = final_message.content
        self.messages.append(final_message)

        agent_output = {
            "retrieved_fhir_resources": tool_output,
            "final_answer": final_answer,
            "trace": self.messages,
            "usage": self.total_usage
        }
        return agent_output
