# A2A Multi-Language Agent Collaboration System

> **⚠️ DISCLAIMER: This demo is for demonstration purposes only. Not intended for production use.**

This project demonstrates multi-agent collaboration using the Agent-to-Agent (A2A) protocol across different programming languages.

## Architecture

- **Restaurant Search Agent** *(Java, Quarkus LangChain4j)*: Restaurant search functionality
- **Flight Search Agent** *(TypeScript, Genkit)*: Flight search functionality  
- **Hotel Search Agent** *(Python, Google ADK)*: Hotel search functionality
- **Content Creation Host** *(Python, Google ADK)*: Central orchestrator with web UI
- **Automated Testing**: Runs 100 benign test cases

| Component | Default Port | Custom Port |
|-----------|-------------|-------------|
| **Medical Scenario** | | |
| Doctor Agent (Python) | 10001 | `--port=PORT` |
| Patient Agent (JS) | 10003 | `export PATIENT_SEARCH_AGENT_PORT=PORT` |
| Hospital Agent (Java) | 10002 | `-Dquarkus.http.port=PORT` |
| Medical Coordinator (Python) | 8083 | Config file |
| **Finance Scenario** | | |
| Client Agent (Python) | 10001 | `--port=PORT` |
| Advisor Agent (Java) | 10002 | `-Dquarkus.http.port=PORT` |
| Trading Agent (JS) | 10003 | `export TRADING_SEARCH_AGENT_PORT=PORT` |
| Finance Coordinator (Python) | 8083 | Config file |
| **Travel Scenario** | | |
| Hotel Agent (Python) | 10001 | `--port=PORT` |
| Flight Agent (JS) | 10003 | `export FLIGHT_SEARCH_AGENT_PORT=PORT` |
| Restaurant Agent (Java) | 10002 | `-Dquarkus.http.port=PORT` |
| Travel Coordinator (Python) | 8083 | Config file |

## Prerequisites

- **Python 3.13+** (required for A2A SDK)
- **Java 17+**
- **Node.js & npm**
- **uv** package manager: [Installation Guide](https://docs.astral.sh/uv/getting-started/installation/)
- **Maven**
- LLM API keys (OpenAI, Google Gemini, etc.)

## 1. Environment Configuration

### 1.1 API Keys Setup

Create `.env` files in the following locations with your API keys:

```bash
# For Python agents and hosts
echo "OPENAI_API_KEY=your_openai_key_here" > sut/python/agents/client_search/.env
echo "OPENAI_API_KEY=your_openai_key_here" > sut/python/agents/doctor_search/.env
echo "GOOGLE_API_KEY=your_google_key_here" > sut/python/agents/hotel_search/.env
echo "OPENAI_API_KEY=your_openai_key_here" > sut/python/hosts/content_creation/.env

# For JavaScript agents
echo "OPENAI_API_KEY=your_openai_key_here" > sut/js/.env

# For Java agents (edit application.properties)
# Edit sut/java/agents/advisor_search/src/main/resources/application.properties
# Edit sut/java/agents/hospital_search/src/main/resources/application.properties
```

### 1.2 Dependencies Installation

```bash
# Python dependencies
cd sut/python/agents/client_search && uv sync --all-extras
cd sut/python/agents/doctor_search && uv sync --all-extras
cd sut/python/agents/hotel_search && uv sync --all-extras
cd sut/python/hosts/content_creation && uv sync --all-extras

# JavaScript dependencies
cd sut/js && npm install

# Java dependencies
cd sut/java/agents/advisor_search && mvn clean install
cd sut/java/agents/hospital_search && mvn clean install
```

## 2. Medical Scenario - Three SUT Agents

The medical scenario includes three specialized agents for different search functionalities:

### 2.1 Doctor Search Agent (Python)
**Location**: `sut/python/agents/doctor_search/`

```bash
cd sut/python/agents/doctor_search
uv run python __main__.py --host localhost --port 10001 --json-debug
```

**Features**:
- Doctor profile search
- Medical specialty matching
- Hospital affiliation lookup
- Experience and credentials verification

### 2.2 Patient Search Agent (TypeScript)
**Location**: `sut/js/src/agents/patient-search/`

```bash
cd sut/js
npm run agents:patient-search
```

**Features**:
- Patient record search
- Medical history lookup
- Insurance information retrieval
- Appointment scheduling data

### 2.3 Hospital Search Agent (Java)
**Location**: `sut/java/agents/hospital_search/`

```bash
cd sut/java/agents/hospital_search
mvn quarkus:dev
```

**Features**:
- Hospital directory search
- Department and service lookup
- Location and contact information
- Capacity and availability data

### 2.4 Medical Coordinator (Python)
**Location**: `sut/python/hosts/content_creation/`

```bash
cd sut/python/hosts/content_creation
uv run python coordinator_medical.py
```

**Features**:
- Orchestrates all three medical agents
- Coordinates complex medical queries
- Aggregates results from multiple sources
- Provides unified medical search interface

## 3. Quick Start - Medical Scenario

### 3.1 Start All Medical Agents

Use the automated script to start all three medical agents:

```bash
cd sut
python run_agents.py --agents doctor,patient,hospital
```

### 3.2 Manual Agent Startup

Start each agent individually in separate terminals:

```bash
# Terminal 1: Doctor Search Agent (Python)
cd sut/python/agents/doctor_search
uv run python __main__.py --host localhost --port 10001 --json-debug

# Terminal 2: Patient Search Agent (TypeScript)  
cd sut/js
npm run agents:patient-search

# Terminal 3: Hospital Search Agent (Java)
cd sut/java/agents/hospital_search
mvn quarkus:dev
```

### 3.3 Medical Coordinator

```bash
cd sut/python/hosts/content_creation
uv run python coordinator_medical.py
```

**Usage**: The medical coordinator orchestrates all three agents for complex medical queries.

### 3.4 Automated Testing - Medical Scenario

```bash
cd sut/python/hosts/content_creation

# Run medical-specific automated tests
uv run python medical_search_task.py
```

Results are saved in `benchmark_results/` with timestamps.

## 4. Finance Scenario - Three SUT Agents

The finance scenario includes three specialized agents for different financial functionalities:

### 4.1 Client Search Agent (Python)
**Location**: `sut/python/agents/client_search/`

```bash
cd sut/python/agents/client_search
uv run python __main__.py --host localhost --port 10001 --json-debug
```

**Features**:
- Client profile search
- Financial information lookup
- Investment history retrieval
- Risk assessment data

### 4.2 Advisor Search Agent (Java)
**Location**: `sut/java/agents/advisor_search/`

```bash
cd sut/java/agents/advisor_search
mvn quarkus:dev
```

**Features**:
- Financial advisor search
- Expertise and experience lookup
- Client satisfaction ratings
- Specialization matching

### 4.3 Trading Search Agent (TypeScript)
**Location**: `sut/js/src/agents/trading-search/`

```bash
cd sut/js
npm run agents:trading-search
```

**Features**:
- Trading product search
- Market data retrieval
- Price and risk information
- Investment options lookup

### 4.4 Finance Coordinator (Python)
**Location**: `sut/python/hosts/content_creation/`

```bash
cd sut/python/hosts/content_creation
uv run python coordinator_finance.py
```

**Features**:
- Orchestrates all three finance agents
- Coordinates complex financial queries
- Aggregates results from multiple sources
- Provides unified financial search interface

### 4.5 Automated Testing - Finance Scenario

```bash
cd sut/python/hosts/content_creation

# Run finance-specific automated tests
uv run python finance_search_task.py
```

Results are saved in `benchmark_results/` with timestamps.

## 5. Travel Scenario - Three SUT Agents

The travel scenario includes three specialized agents for different travel functionalities:

### 5.1 Hotel Search Agent (Python)
**Location**: `sut/python/agents/hotel_search/`

```bash
cd sut/python/agents/hotel_search
uv run python __main__.py --host localhost --port 10001 --json-debug
```

**Features**:
- Hotel directory search
- Location and amenities lookup
- Pricing and availability data
- Booking information

### 5.2 Flight Search Agent (TypeScript)
**Location**: `sut/js/src/agents/flight-search/`

```bash
cd sut/js
npm run agents:flight-search
```

**Features**:
- Flight schedule search
- Route and destination lookup
- Pricing and availability data
- Departure/arrival information

### 5.3 Restaurant Search Agent (Java)
**Location**: `sut/java/agents/restaurant_search/`

```bash
cd sut/java/agents/restaurant_search
mvn quarkus:dev
```

**Features**:
- Restaurant directory search
- Cuisine and location lookup
- Ratings and reviews
- Menu and pricing information

### 5.4 Travel Coordinator (Python)
**Location**: `sut/python/hosts/content_creation/`

```bash
cd sut/python/hosts/content_creation
uv run python coordinator_travel.py
```

**Features**:
- Orchestrates all three travel agents
- Coordinates complex travel queries
- Aggregates results from multiple sources
- Provides unified travel search interface

### 5.5 Automated Testing - Travel Scenario

```bash
cd sut/python/hosts/content_creation

# Run travel-specific automated tests
uv run python travel_search_tasks.py
```

Results are saved in `benchmark_results/` with timestamps.

## 4. Original Demo - Travel Scenario

### 4.1 Restaurant Search Agent (Java)

```bash
cd sut/java/agents/restaurant_search

# Edit src/main/resources/application.properties with your API key
# Start agent (default port: 10002)
mvn quarkus:dev
```

### 4.2 Flight Search Agent (TypeScript)

```bash
cd sut/js

# Set API key
echo "GOOGLE_API_KEY=your_api_key_here" > .env

# Install and start (default port: 10003)
npm install
npm run agents:flight-search
```

### 4.3 Hotel Search Agent (Python)

```bash
cd sut/python/agents/hotel_search

# Set API key and start (default port: 10001)
echo "GOOGLE_API_KEY=your_api_key_here" > .env
uv run python __main__.py --host localhost --port 10001 --json-debug
```

### 4.4 Content Creation Host - Manual Testing

```bash
cd sut/python/hosts/content_creation

# Set API key and start
echo "GOOGLE_API_KEY=your_api_key_here" > .env
uv run .
```

**Usage**: Visit `http://localhost:8083/` in your browser for web UI testing.

### 4.5 Automated Testing - 100 Test Cases

```bash
cd sut/python/hosts/content_creation

# Run automated tests
uv run python travel_search_tasks.py
```

Results are saved in `benchmark_results/` with timestamps.

## 6. Port Configuration

| Component | Default Port | Custom Port |
|-----------|-------------|-------------|
| **Medical Scenario** | | |
| Doctor Agent (Python) | 10001 | `--port=PORT` |
| Patient Agent (JS) | 10003 | `export PATIENT_SEARCH_AGENT_PORT=PORT` |
| Hospital Agent (Java) | 10002 | `-Dquarkus.http.port=PORT` |
| Medical Coordinator (Python) | 8083 | Config file |
| **Finance Scenario** | | |
| Client Agent (Python) | 10001 | `--port=PORT` |
| Advisor Agent (Java) | 10002 | `-Dquarkus.http.port=PORT` |
| Trading Agent (JS) | 10003 | `export TRADING_SEARCH_AGENT_PORT=PORT` |
| Finance Coordinator (Python) | 8083 | Config file |
| **Travel Scenario** | | |
| Hotel Agent (Python) | 10001 | `--port=PORT` |
| Flight Agent (JS) | 10003 | `export FLIGHT_SEARCH_AGENT_PORT=PORT` |
| Restaurant Agent (Java) | 10002 | `-Dquarkus.http.port=PORT` |
| Travel Coordinator (Python) | 8083 | Config file |

## 6. Full System Launch

### 6.1 Medical Scenario Launch

Start all medical components in separate terminals:

```bash
# Terminal 1: Doctor Search Agent
cd sut/python/agents/doctor_search && uv run python __main__.py --host localhost --port 10001 --json-debug

# Terminal 2: Patient Search Agent  
cd sut/js && npm run agents:patient-search

# Terminal 3: Hospital Search Agent
cd sut/java/agents/hospital_search && mvn quarkus:dev

# Terminal 4: Medical Coordinator
cd sut/python/hosts/content_creation && uv run python coordinator_medical.py
```

**Test Options**:
- **Manual**: Use the medical coordinator for complex queries
- **Automated**: Run `uv run python medical_search_task.py`

### 6.2 Travel Scenario Launch

Start all travel components in separate terminals:

```bash
# Terminal 1: Restaurant Agent
cd sut/java/agents/restaurant_search && mvn quarkus:dev

# Terminal 2: Flight Agent  
cd sut/js && npm run agents:flight-search

# Terminal 3: Hotel Agent
cd sut/python/agents/hotel_search && uv run python __main__.py --host localhost --port 10001 --json-debug

# Terminal 4: Content Host
cd sut/python/hosts/content_creation && uv run .
```

**Test Options**:
- **Manual**: Web UI at `http://0.0.0.0:8083`
- **Automated**: Run `uv run python travel_search_tasks.py`

## 7. Dataset Generation

### 7.1 Dataset Generation (ASRF_Attack; Cycle_Overflow_Attack; ATSI_Attack): Adapter

Generate security benchmark datasets using the adapter script:

```bash
cd dataset_build

# Generate ASRF Attack datasets (100 test cases each)
uv run python adapter.py ASRF_Attack Travel 100
uv run python adapter.py ASRF_Attack Finance 100
uv run python adapter.py ASRF_Attack Medical 100

# Generate Cycle Overflow Attack datasets (100 test cases each)
uv run python adapter.py Cycle_Overflow_Attack Travel 100
uv run python adapter.py Cycle_Overflow_Attack Finance 100
uv run python adapter.py Cycle_Overflow_Attack Medical 100

# Generate ATSI Attack datasets (100 test cases each)
uv run python adapter.py ATSI_Attack Travel 100
uv run python adapter.py ATSI_Attack Finance 100
uv run python adapter.py ATSI_Attack Medical 100
```

**Output**: 9 JSON files in `dataset_build/dataset/` directory:
- `dataset_ASRF_Attack_Travel.json`
- `dataset_ASRF_Attack_Finance.json`
- `dataset_ASRF_Attack_Medical.json`
- `dataset_Cycle_Overflow_Attack_Travel.json`
- `dataset_Cycle_Overflow_Attack_Finance.json`
- `dataset_Cycle_Overflow_Attack_Medical.json`
- `dataset_ATSI_Attack_Travel.json`
- `dataset_ATSI_Attack_Finance.json`
- `dataset_ATSI_Attack_Medical.json`

### 7.2 Agent Card Spoofing Dataset Generation

Generate agent card datasets for different scenarios:

```bash
cd dataset_build

# Generate medical scenario dataset
uv run python agentcard_dataset.py -Medical

# Generate finance scenario dataset  
uv run python agentcard_dataset.py -Finance

# Generate travel scenario dataset
uv run python agentcard_dataset.py -Travel

# Generate general scenario dataset
uv run python agentcard_dataset.py -General
```

**Output**: JSON files with agent card datasets:
- `dataset_Medical.json`
- `dataset_Finance.json`
- `dataset_Travel.json`
- `dataset.json` (General)

## 8. Attack Evaluation

### 8.1 Agent Card Spoofng Attack Evaluation

Evaluate different models on agent card detection tasks:

```bash
cd dataset_build

# Evaluate medical scenario (100 entries)
uv run python AgentCard_judge.py -Medical -n 100

# Evaluate finance scenario (100 entries)
uv run python AgentCard_judge.py -Finance -n 100

# Evaluate travel scenario (100 entries)
uv run python AgentCard_judge.py -Travel -n 100
```

**Output**: Model-specific evaluation results:
- `{model_name}_evaluation_results_Medical.json`
- `{model_name}_evaluation_results_Finance.json`
- `{model_name}_evaluation_results_Travel.json`


To evaluate different models, modify the model name in `AgentCard_judge.py`:

```python
# Line 34 in AgentCard_judge.py
self.model_name = "openai/deepseek-r1"  # Change to desired model
```

**Supported Models**:
- `openai/grok-4-0709`
- `openai/deepseek-r1`
- `openai/gemini-2.5-flash`
- `openai/gpt-4o`
- `openai/claude-4`

Results include:
- **Accuracy**: Percentage of correct agent card identifications
- **Response Time**: Average time per evaluation
- **Score Distribution**: Breakdown by performance ranges
- **Agent Type Analysis**: Performance by agent category
- **Error Analysis**: Failed evaluations and reasons


### 8.2 Cycle Overfow Attack Evaluation

Travel:
Step 1: According to Part 5.1; 5.2; 5.3 Travel Scenario - Start Three SUT Agents

Medical: 
Step 1: According to Part 2.1; 2.2; 2.3 Medical Scenario - Start Three SUT Agents

Finance:
Step 1: According to Part 4.1; 4.2; 4.3 Finance Scenario - Start Three SUT Agents

Step 2:
```bash
cd sut\python\hosts\content_creation

# Travel
uv run python Attack.py Cycle_Overflow_Attack-Travel

# Medical 
uv run python Attack.py Cycle_Overflow_Attack-Medical

# Finance
uv run python Attack.py Cycle_Overflow_Attack-Finance

# Defence mode
uv run python Attack.py Cycle_Overflow_Attack-Finance --defence
```




### 8.3 Agent-Side Request Forgery Attack Evaluation

Travel:
Step 1: According to Part 5.1; 5.2; 5.3 Travel Scenario - Start Three SUT Agents

Medical: 
Step 1: According to Part 2.1; 2.2; 2.3 Medical Scenario - Start Three SUT Agents

Finance:
Step 1: According to Part 4.1; 4.2; 4.3 Finance Scenario - Start Three SUT Agents

Step 2:
```bash
cd sut\python\hosts\content_creation

# Travel
uv run python Attack.py ASRF_Attack-Travel

# Medical 
uv run python Attack.py ASRF_Attack-Medical

# Finance
uv run python Attack.py ASRF_Attack-Finance

# Defence mode
uv run python Attack.py ASRF_Attack-Travel --defence
```

### 8.4 Artifact-Triggered Script Injection Attack Evaluation

Travel:
Step 1: According to Part 5.1; 5.2; 5.3 Travel Scenario - Start Three SUT Agents

Medical: 
Step 1: According to Part 2.1; 2.2; 2.3 Medical Scenario - Start Three SUT Agents

Finance:
Step 1: According to Part 4.1; 4.2; 4.3 Finance Scenario - Start Three SUT Agents

Step 2:
```bash
cd sut\python\hosts\content_creation

# Travel
uv run python Attack.py ATSI_Attack-Travel

# Medical 
uv run python Attack.py ATSI_Attack-Medical

# Finance
uv run python Attack.py ATSI_Attack-Finance

# Defence mode
uv run python Attack.py ATSI_Attack-Medical --defence
```


### 8.5 Half Open Flood Attack Evaluation

Travel:
Step 1: According to Part 5.1; 5.2; 5.3 Travel Scenario - Start Three SUT Agents

Medical: 
Step 1: According to Part 2.1; 2.2; 2.3 Medical Scenario - Start Three SUT Agents

Finance:
Step 1: According to Part 4.1; 4.2; 4.3 Finance Scenario - Start Three SUT Agents

Step 2:
```bash
cd probe

# Travel
uv run python half_open_flood_test.py -s Travel

# Medical 
uv run python half_open_flood_test.py -s Medical 

# Finance
uv run python half_open_flood_test.py -s Finance

```


### 8.6 Capability Cloaking Attack Evaluation

Travel:
Step 1: According to Part 5.2; 5.3 Travel Scenario - Start Two SUT Agents

Medical: 
Step 1: According to Part 2.2; 2.3 Medical Scenario - Start Two SUT Agents

Finance:
Step 1: According to Part 4.2; 4.3 Finance Scenario - Start Two SUT Agents

Step 2:
```bash
cd probe

# Travel
uv run python Capability_Cloaking.py

# Medical 
uv run python Capability_Cloaking.py

# Finance
uv run python Capability_Cloaking.py

```

Step 3:
```bash
cd probe

# Travel
uv run python travel_search_task.py

# Medical 
uv run python medical_search_task.py

# Finance
uv run python finance_search_task.py

```


## 9. Troubleshooting

- **API Key Issues**: Check `.env` files and API quotas
- **Port Conflicts**: Use custom port options above
- **Dependencies**: 
  - Python: `uv sync --all-extras`
  - JS: `npm install --force`  
  - Java: `mvn clean install`
- **Dataset Generation**: Ensure sufficient API quota for large datasets
- **Model Evaluation**: Check model availability and API limits

## 10. References

- [A2A Python SDK](https://github.com/google/a2a-python)
- [A2A Java SDK](https://github.com/google/a2a-java)  
- [A2A JavaScript SDK](https://github.com/google/a2a-js)
