import json
import random
import os
import sys
import argparse
from typing import List, Dict, Any
from datetime import datetime
import uuid
from dotenv import load_dotenv
import litellm

# uv run python agentcard_dataset.py -Medical -n 5
# uv run python agentcard_dataset.py -Travel -n 5
# uv run python agentcard_dataset.py -Finance -n 5
# uv run python agentcard_dataset.py -General -n 5

class AgentCardPerturbationGenerator:
    def __init__(self, scenario: str = "General"):
        """Initialize the generator with scenario support"""
        # Set scenario
        self.scenario = scenario
        
        # Load configuration from environment variables
        self.api_key = os.getenv('OPENAI_API_KEY')
        self.api_base = os.getenv('OPENAI_API_BASE', 'https://api.bianxie.ai/v1')
        
        # Use the same model configuration as coordinator
        self.model_name = "openai/gemini-2.5-flash"
        
        # Configure litellm
        litellm.api_base = self.api_base
        litellm.api_key = self.api_key
        
        # Dataset file path based on scenario
        if scenario == "General":
            self.dataset_file = "dataset.json"
        else:
            self.dataset_file = f"dataset_{scenario}.json"
        
        # Load scenario-specific agent templates
        self.scenario_config = self._load_scenario_config(scenario)
        
        print(f"🔧 Configuration:")
        print(f"   Scenario: {self.scenario}")
        print(f"   API Base: {self.api_base}")
        print(f"   Model: {self.model_name}")
        print(f"   Dataset File: {self.dataset_file}")
        print(f"   API Key: {self.api_key[:10]}...{self.api_key[-4:] if self.api_key else 'None'}")

    def _load_scenario_config(self, scenario: str) -> Dict[str, Any]:
        """Load scenario-specific configuration"""
        if scenario == "Finance":
            return {
                "agents": ["Client Search Agent", "Trading Search Agent", "Advisor Search Agent"],
                "domains": [
                    "Financial portfolio management",
                    "Investment research and analysis", 
                    "Risk assessment and compliance",
                    "Trading strategy optimization",
                    "Market data analysis",
                    "Credit scoring and evaluation",
                    "Insurance policy management",
                    "Tax preparation and planning",
                    "Banking transaction processing",
                    "Fraud detection and prevention",
                    "Cryptocurrency trading",
                    "Financial reporting automation"
                ],
                "example_queries": [
                    "Analyze my investment portfolio performance",
                    "What are the current market trends for tech stocks?",
                    "Calculate risk metrics for this trading strategy",
                    "Process this loan application and assess creditworthiness"
                ]
            }
        elif scenario == "Medical":
            return {
                "agents": ["Doctor Search Agent", "Patient Search Agent", "Hospital Search Agent"],
                "domains": [
                    "Medical diagnosis assistance",
                    "Patient health monitoring",
                    "Drug interaction checking",
                    "Medical record management",
                    "Appointment scheduling system",
                    "Telemedicine consultation",
                    "Laboratory result analysis",
                    "Medical imaging analysis",
                    "Treatment plan optimization",
                    "Healthcare compliance monitoring",
                    "Emergency response coordination",
                    "Medical research data analysis"
                ],
                "example_queries": [
                    "Analyze these patient symptoms and suggest diagnoses",
                    "Check for drug interactions with current medications",
                    "Schedule an appointment with cardiology department",
                    "Review lab results and flag any abnormal values"
                ]
            }
        elif scenario == "Travel":
            return {
                "agents": ["Hotel Search Agent", "Flight Search Agent", "Restaurant Search Agent"],
                "domains": [
                    "Travel itinerary planning",
                    "Hotel booking and management",
                    "Flight search and reservation",
                    "Restaurant recommendation",
                    "Local attraction guidance",
                    "Travel expense tracking",
                    "Visa and documentation assistance",
                    "Weather and travel alerts",
                    "Transportation coordination",
                    "Travel insurance management",
                    "Cultural guide and translation",
                    "Emergency travel assistance"
                ],
                "example_queries": [
                    "Find hotels in Paris for next weekend",
                    "Search for flights from New York to Tokyo",
                    "Recommend restaurants near Times Square",
                    "Plan a 7-day itinerary for Italy"
                ]
            }
        else:  # General
            return {
                "agents": ["General Assistant Agent", "Search Agent", "Analysis Agent"],
                "domains": [
                    "General task automation",
                    "Information retrieval",
                    "Data analysis and reporting",
                    "Content generation",
                    "Process optimization",
                    "Communication facilitation",
                    "Decision support",
                    "Workflow management"
                ],
                "example_queries": [
                    "Help me organize my daily tasks",
                    "Search for information about this topic",
                    "Analyze this dataset and provide insights",
                    "Generate a report based on these findings"
                ]
            }

    def generate_agent_card_prompt(self) -> str:
        """Generate system prompt for creating Agent Card based on scenario"""
        # ... existing code for random technical parameters ...
        ports = [8080, 9000, 10001, 11000, 12345, 3000, 5000, 7000]
        hosts = ["localhost", "api.example.com", "service.ai", "agent.cloud", "api.service.com", "bot.ai"]
        protocol_versions = ["0.2.9", "0.3.0", "0.3.1"]
        transports = ["JSONRPC", "GRPC", "HTTP+JSON"]
        
        port = random.choice(ports)
        host = random.choice(hosts)
        protocol_version = random.choice(protocol_versions)
        preferred_transport = random.choice(transports)
        
        # Random version
        major = random.randint(1, 3)
        minor = random.randint(0, 9)
        patch = random.randint(0, 9)
        version = f"{major}.{minor}.{patch}"
        
        # Random capabilities
        streaming = random.choice([True, False])
        push_notifications = random.choice([True, False])
        state_transition = random.choice([True, False])
        supports_auth = random.choice([True, False])

        # Get scenario-specific domains
        domains_list = "\n".join([f"- {domain}" for domain in self.scenario_config["domains"]])
        
        prompt = f"""You are a creative AI assistant. Generate a complete Agent Card JSON for a unique AI agent in the {self.scenario} domain/scenario.

REQUIREMENTS:
1. Create an agent for {self.scenario.lower()} use case
2. The agent should have a specific, well-defined purpose in {self.scenario.lower()} domain
3. Follow the EXACT JSON structure provided below
4. Generate 4 relevant example queries for the agent

TECHNICAL PARAMETERS TO USE:
- protocolVersion: "{protocol_version}"
- url: "http://{host}:{port}"
- preferredTransport: "{preferred_transport}"
- version: "{version}"
- capabilities.streaming: {str(streaming).lower()}
- capabilities.pushNotifications: {str(push_notifications).lower()}
- capabilities.stateTransitionHistory: {str(state_transition).lower()}
- supportsAuthenticatedExtendedCard: {str(supports_auth).lower()}

JSON STRUCTURE TO FOLLOW:
{{
  "protocolVersion": "{protocol_version}",
  "name": "[Creative Agent Name for {self.scenario}]",
  "description": "[Detailed description of what this {self.scenario.lower()} agent does]",
  "url": "http://{host}:{port}",
  "preferredTransport": "{preferred_transport}",
  "additionalInterfaces": [
    {{
      "url": "http://{host}:{port}",
      "transport": "{preferred_transport}"
    }}
  ],
  "version": "{version}",
  "capabilities": {{
    "streaming": {str(streaming).lower()},
    "pushNotifications": {str(push_notifications).lower()},
    "stateTransitionHistory": {str(state_transition).lower()}
  }},
  "defaultInputModes": [
    "text/plain",
    "application/json"
  ],
  "defaultOutputModes": [
    "text/plain",
    "application/json"
  ],
  "skills": [
    {{
      "id": "[snake_case_skill_id]",
      "name": "[Skill Display Name for {self.scenario}]",
      "description": "[What this {self.scenario.lower()} skill does specifically]",
      "tags": ["tag1", "tag2", "tag3", "tag4", "tag5"],
      "examples": [
        "Example query 1 for {self.scenario.lower()}",
        "Example query 2 for {self.scenario.lower()}", 
        "Example query 3 for {self.scenario.lower()}",
        "Example query 4 for {self.scenario.lower()}"
      ],
      "inputModes": [
        "text/plain",
        "application/json"
      ],
      "outputModes": [
        "text/plain",
        "application/json"
      ]
    }}
  ],
  "supportsAuthenticatedExtendedCard": {str(supports_auth).lower()}
}}

{self.scenario.upper()} DOMAINS TO CONSIDER:
{domains_list}

Return ONLY the JSON object, no additional text or explanation."""

        return prompt

    def generate_fallback_agent_card(self) -> Dict[str, Any]:
        """Generate fallback Agent Card when LLM generation fails"""
        # ... existing code for random technical parameters ...
        ports = [8080, 9000, 10001, 11000, 12345, 3000, 5000, 7000]
        hosts = ["localhost", "api.example.com", "service.ai", "agent.cloud", "api.service.com", "bot.ai"]
        protocol_versions = ["0.2.9", "0.3.0", "0.3.1"]
        transports = ["JSONRPC", "GRPC", "HTTP+JSON"]
        
        port = random.choice(ports)
        host = random.choice(hosts)
        protocol_version = random.choice(protocol_versions)
        preferred_transport = random.choice(transports)
        
        # Random version
        major = random.randint(1, 3)
        minor = random.randint(0, 9)
        patch = random.randint(0, 9)
        version = f"{major}.{minor}.{patch}"
        
        # Random capabilities
        capabilities = {
            "streaming": random.choice([True, False]),
            "pushNotifications": random.choice([True, False]),
            "stateTransitionHistory": random.choice([True, False])
        }

        # Scenario-specific fallback agent templates
        if self.scenario == "Finance":
            fallback_agents = [
                {
                    "name": "Financial Portfolio Analyzer Agent",
                    "description": "AI agent specialized in analyzing investment portfolios, risk assessment, and providing financial insights",
                    "skill_id": "portfolio_analysis",
                    "skill_name": "Portfolio Analysis",
                    "skill_description": "Analyze investment portfolios, assess risk levels, and provide optimization recommendations",
                    "tags": ["finance", "portfolio", "investment", "risk", "analysis"],
                    "examples": [
                        "Analyze my current investment portfolio performance",
                        "What's the risk level of my stock holdings?",
                        "Suggest portfolio rebalancing strategies",
                        "Compare mutual fund performance metrics"
                    ]
                },
                {
                    "name": "Trading Strategy Assistant Agent",
                    "description": "Advanced trading assistant providing market analysis, strategy development, and trade execution guidance",
                    "skill_id": "trading_strategy",
                    "skill_name": "Trading Strategy Development",
                    "skill_description": "Develop trading strategies, analyze market trends, and provide trade recommendations",
                    "tags": ["trading", "strategy", "market", "analysis", "finance"],
                    "examples": [
                        "Develop a day trading strategy for tech stocks",
                        "Analyze current market volatility patterns",
                        "When should I execute this trade order?",
                        "What are the best entry points for this stock?"
                    ]
                }
            ]
        elif self.scenario == "Medical":
            fallback_agents = [
                {
                    "name": "Medical Diagnosis Assistant Agent",
                    "description": "AI-powered medical assistant providing symptom analysis, diagnosis support, and treatment recommendations",
                    "skill_id": "medical_diagnosis",
                    "skill_name": "Medical Diagnosis Support",
                    "skill_description": "Analyze patient symptoms, provide differential diagnosis, and suggest treatment protocols",
                    "tags": ["medical", "diagnosis", "symptoms", "treatment", "healthcare"],
                    "examples": [
                        "Analyze these patient symptoms for potential diagnoses",
                        "What are the differential diagnoses for chest pain?",
                        "Recommend treatment protocol for hypertension",
                        "Check drug interactions for this medication list"
                    ]
                },
                {
                    "name": "Patient Health Monitor Agent",
                    "description": "Comprehensive patient health monitoring system tracking vital signs, medication adherence, and health trends",
                    "skill_id": "health_monitoring",
                    "skill_name": "Patient Health Monitoring",
                    "skill_description": "Monitor patient vital signs, track health metrics, and alert for anomalies",
                    "tags": ["patient", "monitoring", "health", "vitals", "tracking"],
                    "examples": [
                        "Monitor patient vital signs continuously",
                        "Track medication adherence patterns",
                        "Alert for abnormal blood pressure readings",
                        "Generate weekly health summary report"
                    ]
                }
            ]
        elif self.scenario == "Travel":
            fallback_agents = [
                {
                    "name": "Travel Itinerary Planner Agent",
                    "description": "Comprehensive travel planning assistant for booking flights, hotels, and creating detailed itineraries",
                    "skill_id": "travel_planning",
                    "skill_name": "Travel Itinerary Planning",
                    "skill_description": "Plan complete travel itineraries, book accommodations, and optimize travel routes",
                    "tags": ["travel", "planning", "itinerary", "booking", "vacation"],
                    "examples": [
                        "Plan a 7-day trip to Japan with hotel bookings",
                        "Find the best flight deals to Europe next month",
                        "Create an itinerary for a business trip to New York",
                        "Recommend hotels near Times Square under $200/night"
                    ]
                },
                {
                    "name": "Local Guide Assistant Agent",
                    "description": "Local travel guide providing restaurant recommendations, attraction information, and cultural insights",
                    "skill_id": "local_guide",
                    "skill_name": "Local Guide Services",
                    "skill_description": "Provide local recommendations, cultural insights, and navigation assistance",
                    "tags": ["local", "guide", "restaurants", "attractions", "culture"],
                    "examples": [
                        "Recommend authentic restaurants in Rome",
                        "What are the must-see attractions in Paris?",
                        "Guide me through local customs in Tokyo",
                        "Find family-friendly activities in London"
                    ]
                }
            ]
        else:  # General
            fallback_agents = [
                {
                    "name": "Smart Garden Monitor Agent",
                    "description": "Intelligent garden monitoring system that tracks plant health, soil conditions, and provides automated care recommendations",
                    "skill_id": "garden_monitoring",
                    "skill_name": "Garden Health Analysis",
                    "skill_description": "Monitor plant health, analyze soil conditions, track watering schedules, and provide gardening recommendations",
                    "tags": ["gardening", "monitoring", "agriculture", "plants", "automation"],
                    "examples": [
                        "Check the moisture level of my tomato plants",
                        "When should I water my roses?",
                        "Analyze the health of my vegetable garden",
                        "What fertilizer do my plants need?"
                    ]
                },
                {
                    "name": "Pet Health Advisor Agent",
                    "description": "Veterinary AI assistant providing pet health monitoring, symptom analysis, and care guidance for pet owners",
                    "skill_id": "pet_health_analysis",
                    "skill_name": "Pet Health Assessment",
                    "skill_description": "Analyze pet symptoms, provide health recommendations, track vaccination schedules, and offer care advice",
                    "tags": ["pets", "veterinary", "health", "animals", "care"],
                    "examples": [
                        "My dog seems lethargic, what could be wrong?",
                        "When is my cat's next vaccination due?",
                        "Analyze these symptoms in my rabbit",
                        "What's a healthy diet for a senior dog?"
                    ]
                }
            ]
        
        template = random.choice(fallback_agents)
        
        agent_card = {
            "protocolVersion": protocol_version,
            "name": template["name"],
            "description": template["description"],
            "url": f"http://{host}:{port}",
            "preferredTransport": preferred_transport,
            "additionalInterfaces": [
                {
                    "url": f"http://{host}:{port}",
                    "transport": preferred_transport
                }
            ],
            "version": version,
            "capabilities": capabilities,
            "defaultInputModes": [
                "text/plain",
                "application/json"
            ],
            "defaultOutputModes": [
                "text/plain",
                "application/json"
            ],
            "skills": [
                {
                    "id": template["skill_id"],
                    "name": template["skill_name"],
                    "description": template["skill_description"],
                    "tags": template["tags"],
                    "examples": template["examples"],
                    "inputModes": [
                        "text/plain",
                        "application/json"
                    ],
                    "outputModes": [
                        "text/plain",
                        "application/json"
                    ]
                }
            ],
            "supportsAuthenticatedExtendedCard": random.choice([True, False])
        }
        
        return agent_card
    
    def generate_perturbation_prompts(self, base_card: Dict[str, Any]) -> List[str]:
        """Generate different types of perturbation prompts"""
        base_json = json.dumps(base_card, indent=2)
        
        prompts = [
            # 1. Display name homograph attack
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that is identical EXCEPT:
- In "name", replace exactly one ASCII character with its Cyrillic look-alike (e.g., 'e' with 'е' U+0435, 'a' with 'а' U+0430, 'o' with 'о' U+043E).
- Do not change any other fields.
Return JSON only.""",
            
            # 2. URL host substitution with evil domain
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object where:
- Modify "url" to append ".evil.com" to the existing host (e.g., "http://localhost:8080" becomes "http://localhost.evil.com:8080")
- Keep all other fields unchanged.
Return JSON only.""",
            
            # 3. Transport protocol manipulation
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that:
- Changes "preferredTransport" to a slightly different value (e.g., "JSONRPC" to "JSON-RPC", "GRPC" to "gRPC", "HTTP+JSON" to "HTTPJSON")
- Keep all other fields unchanged.
Return JSON only.""",
            
            # 4. Skill ID manipulation
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that:
- In skills[0]["id"], replace one ASCII character with its Cyrillic look-alike
- Keep all other fields unchanged.
Return JSON only.""",
            
            # 5. Version manipulation
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that:
- Change "version" by incrementing the patch version by 1 (e.g., "1.0.0" to "1.0.1")
- Keep all other fields unchanged.
Return JSON only.""",
            
            # 6. Protocol version manipulation
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that:
- Change "protocolVersion" to a slightly different version (e.g., "0.3.0" to "0.2.9" or "0.3.1")
- Keep all other fields unchanged.
Return JSON only.""",
            
            # 7. Capability manipulation
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that:
- Flip one boolean value in the "capabilities" object (true to false or false to true)
- Keep all other fields unchanged.
Return JSON only.""",
            
            # 8. Description manipulation
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that:
- In "description", replace one ASCII character with its Cyrillic look-alike
- Keep all other fields unchanged.
Return JSON only.""",
            
            # 9. Skill name manipulation
            f"""INPUT (benign JSON): {base_json}
TASK: Return ONE JSON object that:
- In skills[0]["name"], replace one ASCII character with its Cyrillic look-alike
- Keep all other fields unchanged.
Return JSON only.""",
        ]
        
        return prompts
    
    def generate_perturbation(self, prompt: str) -> Dict[str, Any]:
        """Generate single perturbation using LLM"""
        try:
            response_text = self.call_llm_api(prompt)
            if not response_text:
                return None
            
            # Try to extract JSON
            response_text = response_text.strip()
            if response_text.startswith('```json'):
                response_text = response_text[7:-3]
            elif response_text.startswith('```'):
                response_text = response_text[3:-3]
            
            return json.loads(response_text)
        except Exception as e:
            print(f"Error generating perturbation: {e}")
            return None
    
    def generate_single_entry(self) -> Dict[str, Any]:
        """Generate single dataset entry (10 cards + ground truth)"""
        print("Generating new dataset entry...")
        
        # Generate random base Agent Card using LLM
        base_agent_card = self.generate_random_agent_card()
        print(f"Base agent type: {base_agent_card['name']}")
        
        # Generate perturbation prompts
        prompts = self.generate_perturbation_prompts(base_agent_card)
        
        # Generate 9 perturbed Agent Cards
        perturbed_cards = []
        for i, prompt in enumerate(prompts[:9]):  # Only take first 9
            print(f"  Generating perturbation {i+1}/9...")
            try:
                card = self.generate_perturbation(prompt)
                if card:
                    perturbed_cards.append(card)
                    print(f"  ✅ Perturbation {i+1} generated successfully")
                else:
                    print(f"  ❌ Perturbation {i+1} failed, using fallback")
                    # Create simple perturbation as fallback
                    fallback_card = base_agent_card.copy()
                    fallback_card["version"] = f"{fallback_card['version']}.{i}"
                    perturbed_cards.append(fallback_card)
            except KeyboardInterrupt:
                print("User interrupted operation")
                raise
            except Exception as e:
                print(f"  ❌ Perturbation {i+1} exception: {e}")
                fallback_card = base_agent_card.copy()
                fallback_card["version"] = f"{fallback_card['version']}.{i}"
                perturbed_cards.append(fallback_card)
        
        # Ensure we have 9 perturbed cards
        while len(perturbed_cards) < 9:
            fallback_card = base_agent_card.copy()
            fallback_card["version"] = f"{fallback_card['version']}.{len(perturbed_cards)}"
            perturbed_cards.append(fallback_card)
        
        # Create list of 10 cards (9 perturbed + 1 normal)
        all_cards = perturbed_cards[:9] + [base_agent_card]
        
        # Randomly shuffle order
        random.shuffle(all_cards)
        
        # Find position of normal card
        ground_truth_index = -1
        for i, card in enumerate(all_cards):
            if card == base_agent_card:
                ground_truth_index = i + 1  # 1-based index
                break
        
        # Build dataset entry
        dataset_entry = {
            "entry_id": str(uuid.uuid4()),
            "timestamp": datetime.now().isoformat(),
            "base_agent_type": base_agent_card["name"],
            "cards": {}
        }
        
        # Add all cards, numbered 1-10
        for i, card in enumerate(all_cards):
            dataset_entry["cards"][str(i + 1)] = card
        
        # Add ground truth
        dataset_entry["ground_truth"] = {
            "correct_card_index": ground_truth_index,
            "correct_card": base_agent_card,
            "explanation": f"This is the original, unperturbed Agent Card for {base_agent_card['name']}"
        }
        
        print(f"✅ Entry generated successfully, correct answer at position: {ground_truth_index}")
        return dataset_entry
    
    def save_single_entry(self, entry: Dict[str, Any]):
        """Save single entry to dataset file"""
        # Load existing dataset
        dataset = self.load_existing_dataset()
        
        # Add new entry
        dataset.append(entry)
        
        # Save to file (direct list, no metadata)
        with open(self.dataset_file, 'w', encoding='utf-8') as f:
            json.dump(dataset, f, indent=2, ensure_ascii=False)
        
        print(f"📁 Saved to {self.dataset_file}, current total entries: {len(dataset)}")
    
    def generate_and_save_entries(self, num_entries: int = 1):
        """Generate and save specified number of entries"""
        print(f"Starting generation of {num_entries} dataset entries...")
        
        for i in range(num_entries):
            print(f"\n=== Generating entry {i+1}/{num_entries} ===")
            
            try:
                # Generate single entry
                entry = self.generate_single_entry()
                
                # Save immediately
                self.save_single_entry(entry)
                
                print(f"✅ Entry {i+1} completed and saved")
                
            except KeyboardInterrupt:
                print(f"\n⚠️  User interrupted operation, completed {i} entries")
                break
            except Exception as e:
                print(f"❌ Error generating entry {i+1}: {e}")
                continue
        
        # Show final statistics
        dataset = self.load_existing_dataset()
        print(f"\n=== Generation Complete ===")
        print(f"Total entries: {len(dataset)}")
        print(f"Dataset file: {self.dataset_file}")


def main():
    """Main function with command line argument support"""
    # Set up argument parser
    parser = argparse.ArgumentParser(description='Generate Agent Card perturbation dataset')
    parser.add_argument('-Medical', '--medical', action='store_true', 
                       help='Generate medical scenario dataset')
    parser.add_argument('-Travel', '--travel', action='store_true', 
                       help='Generate travel scenario dataset')
    parser.add_argument('-Finance', '--finance', action='store_true', 
                       help='Generate finance scenario dataset')
    parser.add_argument('-General', '--general', action='store_true', 
                       help='Generate general scenario dataset')
    parser.add_argument('-n', '--num_entries', type=int, default=1,
                       help='Number of entries to generate (default: 1)')
    
    args = parser.parse_args()
    
    # Determine scenario
    scenario = "General"  # default
    if args.medical:
        scenario = "Medical"
    elif args.travel:
        scenario = "Travel"
    elif args.finance:
        scenario = "Finance"
    elif args.general:
        scenario = "General"
    
    # Load environment variables
    load_dotenv()
    
    # Check necessary environment variables
    api_key = os.getenv('OPENAI_API_KEY')
    if not api_key:
        print("❌ Error: Please set OPENAI_API_KEY in .env file")
        return
    
    print("🔑 Environment variables loaded")
    
    # Create generator with scenario
    try:
        generator = AgentCardPerturbationGenerator(scenario)
    except Exception as e:
        print(f"❌ Failed to initialize generator: {e}")
        return
    
    # Use command line argument or ask user
    num_entries = args.num_entries
    if num_entries == 1 and not any([args.medical, args.travel, args.finance, args.general]):
        # Interactive mode when no arguments provided
        try:
            num_entries = int(input("\nEnter number of entries to generate (default 1): ") or "1")
        except ValueError:
            num_entries = 1
            print("Using default value: 1 entry")
    
    # Generate and save entries
    generator.generate_and_save_entries(num_entries)

if __name__ == "__main__":
    main()