#!/usr/bin/env python3
"""
Universal Dynamic Module Replacement Tester
Temporarily replace any specified module, zero file modification
"""

import os
import sys
import subprocess
import importlib.util
from pathlib import Path
import shutil

def print_banner():
    """Print banner"""
    print("=" * 80)
    print("Universal Dynamic Module Replacement Tester")
    print("Temporarily replace any specified module, zero file modification")
    print("=" * 80)
    print()

def get_user_input():
    """Get replacement configuration from user"""
    print("Configure Module Replacement:")
    print()
    
    # Use relative path from script location
    script_dir = Path(__file__).parent
    source_path = script_dir / "malicious_agent" / "agent.py"
    print(f"Using fixed malicious module: {source_path}")
    
    print("\nTarget module path to replace:")
    print("   Example:")
    print("   - ../a2a-samples-main-TAR/samples/python/agents/hotel_search/hotel_search_agent.py")
    print("   - ./path/to/your/target_agent.py")
    
    # Check if running with auto-confirmation (from malicious_agent.py)
    if len(sys.argv) > 1 and sys.argv[1] == "--auto-confirm":
        # Auto-confirm mode - get target path from command line argument
        if len(sys.argv) > 2:
            target_input = sys.argv[2]
            print(f">> {target_input} (auto-provided)")
        else:
            print(">> ", end="", flush=True)
            target_input = input().strip()
    else:
        print(">> ", end="", flush=True)
        target_input = input().strip()
    
    if not target_input:
        print("Target file path is required!")
        return None, None, None, None
    
    target_path = Path(target_input)
    
    # Validate files exist
    if not source_path.exists():
        print(f"Malicious module file not found: {source_path}")
        return None, None, None, None
    
    if not target_path.exists():
        print(f"Target module file not found: {target_path}")
        return None, None, None, None
    
    # Determine project path and module name
    project_path = target_path.parent  # Directory containing the target module
    module_name = target_path.stem  # Module name without .py extension
    
    print(f"\nReplacement Configuration:")
    print(f"   Malicious Module: {source_path}")
    print(f"   Target Module: {target_path}")
    print(f"   Project Path: {project_path}")
    print(f"   Module Name: {module_name}")
    
    return source_path, target_path, project_path, module_name

def setup_environment(project_path):
    """Setup project environment"""
    print(f"\nSetting up project environment...")
    print(f"Project Path: {project_path}")
    
    if not project_path.exists():
        print(f"Project path does not exist: {project_path}")
        return False
    
    # Change to project directory
    os.chdir(project_path)
    print(f"Changed to project directory: {project_path}")
    
    # Add project to Python path
    if str(project_path) not in sys.path:
        sys.path.insert(0, str(project_path))
    
    print("Environment setup completed")
    return True

def create_malicious_launcher(source_path, target_path, project_path, module_name):
    """Create temporary launcher script with memory-level module replacement"""
    return f'''#!/usr/bin/env python3
"""
Temporary malicious launcher script with memory-level module replacement
Auto-generated by Universal Dynamic Module Replacement Tester
"""

import os
import sys
import importlib.util
import subprocess
from pathlib import Path

def main():
    print("Starting malicious module replacement (memory-level)...")
    
    # Setup project environment
    project_path = Path(r"{project_path}")
    os.chdir(project_path)
    sys.path.insert(0, str(project_path))
    
    # Dynamic module replacement configuration
    source_path = Path(r"{source_path}")
    module_name = "{module_name}"
    
    print(f"Replacing module: {{module_name}}")
    print(f"Using malicious file: {{source_path}}")
    
    try:
        # Load malicious module dynamically
        spec = importlib.util.spec_from_file_location(module_name, source_path)
        malicious_module = importlib.util.module_from_spec(spec)
        
        # Replace system module in memory (no file modification)
        sys.modules[module_name] = malicious_module
        spec.loader.exec_module(malicious_module)
        
        print("Malicious module replacement successful (memory-level)")
        print("Original files remain completely unmodified")
        
        # Start the original main program
        print("Starting hotel search agent server...")
        print("Command: uv run python __main__.py --host localhost --port 10001")
        print("-" * 60)
        
        # Use subprocess to run uv command
        cmd = ["uv", "run", "python", "__main__.py", "--host", "localhost", "--port", "10001"]
        subprocess.run(cmd, check=True)
        
    except subprocess.CalledProcessError as e:
        print(f"uv command failed: {{e}}")
    except FileNotFoundError:
        print("uv command not found, please install uv first")
    except Exception as e:
        print(f"Error during execution: {{e}}")
        import traceback
        traceback.print_exc()
    finally:
        # Memory-level replacement automatically reverts when process ends
        print("Memory-level replacement automatically reverted")
        print("No file restoration needed - files were never modified")

if __name__ == "__main__":
    main()
'''

def start_malicious_agent(source_path, target_path, project_path, module_name):
    """Start malicious version of Agent with memory-level replacement"""
    print(f"\nPreparing to start malicious version (memory-level replacement)...")
    print(f"Malicious module: {source_path}")
    print(f"Will replace module: {module_name}")
    print(f"Working directory: {project_path}")
    print(f"Original file will NOT be modified: {target_path}")
    
    # Setup environment
    if not setup_environment(project_path):
        return False
    
    # Create temporary launcher script
    launcher_content = create_malicious_launcher(source_path, target_path, project_path, module_name)
    temp_launcher = project_path / "temp_malicious_launcher.py"
    
    try:
        # Write temporary launcher script
        with open(temp_launcher, "w", encoding="utf-8") as f:
            f.write(launcher_content)
        
        print(f"Created temporary launcher: {temp_launcher}")
        
        print("\nStarting malicious hotel search agent (memory-level replacement)...")
        print("Command will be: uv run python __main__.py --host localhost --port 10001")
        print("Press Ctrl+C to stop service")
        print("-" * 60)
        
        # Execute the launcher with UTF-8 encoding
        with open(temp_launcher, "r", encoding="utf-8") as f:
            launcher_code = f.read()
        exec(launcher_code)
    
    except Exception as e:
        print(f"Error during startup: {e}")
        import traceback
        traceback.print_exc()
        return False
    
    finally:
        # Clean up temporary files
        if temp_launcher.exists():
            try:
                temp_launcher.unlink()
                print(f"Cleaned up temporary launcher")
            except:
                print(f"Warning: Could not clean up {temp_launcher}")
    
    return True

def main():
    """Main function"""
    try:
        print_banner()
        
        # Step 1: Get replacement configuration
        config = get_user_input()
        if not config or not all(config):
            print("Configuration incomplete, exiting...")
            return
        
        source_path, target_path, project_path, module_name = config
        
        # Step 2: Confirm replacement operation (with auto-confirm support)
        print(f"\nReplacement Confirmation (Memory-Level):")
        print(f"   Malicious Module: {source_path}")
        print(f"   Target File: {target_path}")
        print(f"   Working Directory: {project_path}")
        print(f"   Command: uv run python __main__.py --host localhost --port 10001")
        print(f"   Safety: Original file will NOT be modified (memory-level replacement)")
        
        # Check for auto-confirmation
        if len(sys.argv) > 1 and sys.argv[1] == "--auto-confirm":
            print("\nAuto-confirming execution (--auto-confirm flag detected)")
            confirm = "y"
        else:
            confirm = input("\nConfirm execution of replacement and startup? (y/n): ").strip().lower()
        
        if confirm not in ['y', 'yes']:
            print("Operation cancelled by user")
            return
        
        # Step 3: Execute replacement and startup
        start_malicious_agent(source_path, target_path, project_path, module_name)
        
    except KeyboardInterrupt:
        print("\nOperation interrupted by user")
    except Exception as e:
        print(f"Error occurred: {e}")
    finally:
        print("\nTest completed")
        print("Original files completely unmodified (memory-level replacement)")

if __name__ == "__main__":
    main()