package com.samples.a2a;

import jakarta.enterprise.context.ApplicationScoped;
import jakarta.enterprise.inject.Produces;
import jakarta.inject.Inject;

import java.util.List;

import io.a2a.server.agentexecution.AgentExecutor;
import io.a2a.server.agentexecution.RequestContext;
import io.a2a.server.events.EventQueue;
import io.a2a.server.tasks.TaskUpdater;
import io.a2a.spec.JSONRPCError;
import io.a2a.spec.Message;
import io.a2a.spec.Part;
import io.a2a.spec.Task;
import io.a2a.spec.TaskNotCancelableError;
import io.a2a.spec.TaskState;
import io.a2a.spec.TextPart;

/**
 * Producer for advisor search agent executor.
 * This class is final and not designed for extension.
 */
@ApplicationScoped
public final class AdvisorSearchAgentExecutorProducer {

    /**
     * The advisor search agent instance.
     */
    @Inject
    private AdvisorSearchAgent advisorSearchAgent;

    /**
     * Gets the advisor search agent.
     *
     * @return the advisor search agent
     */
    public AdvisorSearchAgent getAdvisorSearchAgent() {
        return advisorSearchAgent;
    }

    /**
     * Produces the agent executor for the advisor search agent.
     *
     * @return the configured agent executor
     */
    @Produces
    public AgentExecutor agentExecutor() {
        return new AdvisorSearchAgentExecutor(getAdvisorSearchAgent());
    }

    /**
     * Advisor search agent executor implementation.
     */
    private static class AdvisorSearchAgentExecutor implements AgentExecutor {

        /**
         * The advisor search agent instance.
         */
        private final AdvisorSearchAgent agent;

        /**
         * Constructor for AdvisorSearchAgentExecutor.
         *
         * @param advisorSearchAgentInstance the advisor search agent instance
         */
        AdvisorSearchAgentExecutor(
                final AdvisorSearchAgent advisorSearchAgentInstance) {
            this.agent = advisorSearchAgentInstance;
        }

        @Override
        public void execute(final RequestContext context,
                            final EventQueue eventQueue) throws JSONRPCError {
            final TaskUpdater updater = new TaskUpdater(context, eventQueue);

            // mark the task as submitted and start working on it
            if (context.getTask() == null) {
                updater.submit();
            }
            updater.startWork();

            // extract the text from the message
            final String assignment = extractTextFromMessage(
                    context.getMessage());

            // call the advisor search agent with the message
            final String response = agent.searchAdvisors(assignment);

            // create the response part
            final TextPart responsePart = new TextPart(response, null);
            final List<Part<?>> parts = List.of(responsePart);

            // add the response as an artifact and complete the task
            updater.addArtifact(parts, null, null, null);
            updater.complete();
        }

        private String extractTextFromMessage(final Message message) {
            final StringBuilder textBuilder = new StringBuilder();
            if (message.getParts() != null) {
                for (final Part part : message.getParts()) {
                    if (part instanceof TextPart textPart) {
                        textBuilder.append(textPart.getText());
                    }
                }
            }
            return textBuilder.toString();
        }

        @Override
        public void cancel(final RequestContext context,
                           final EventQueue eventQueue) throws JSONRPCError {
            final Task task = context.getTask();

            if (task.getStatus().state() == TaskState.CANCELED) {
                // task already cancelled
                throw new TaskNotCancelableError();
            }

            if (task.getStatus().state() == TaskState.COMPLETED) {
                // task already completed
                throw new TaskNotCancelableError();
            }

            // cancel the task
            final TaskUpdater updater = new TaskUpdater(context, eventQueue);
            updater.cancel();
        }
    }
}