package com.samples.a2a;

import dev.langchain4j.agent.tool.Tool;
import jakarta.enterprise.context.ApplicationScoped;
import org.yaml.snakeyaml.Yaml;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Advisor search tool for querying advisor data from YAML file.
 */
@ApplicationScoped
public class AdvisorSearchTool {

    private List<Map<String, Object>> advisors;

    public AdvisorSearchTool() {
        loadAdvisorData();
    }

    @SuppressWarnings("unchecked")
    private void loadAdvisorData() {
        try {
            Yaml yaml = new Yaml();
            InputStream inputStream = getClass().getResourceAsStream("/advisor_data.yaml");
            if (inputStream == null) {
                inputStream = getClass().getResourceAsStream("advisor_data.yaml");
            }
            
            if (inputStream != null) {
                Map<String, Object> data = yaml.load(inputStream);
                Map<String, Object> advisorsData = (Map<String, Object>) data.get("advisors");
                this.advisors = (List<Map<String, Object>>) advisorsData.get("advisor_list");
            } else {
                this.advisors = new ArrayList<>();
                System.err.println("Warning: advisor_data.yaml not found, using empty advisor list");
            }
        } catch (Exception e) {
            this.advisors = new ArrayList<>();
            System.err.println("Error loading advisor data: " + e.getMessage());
        }
    }

    @Tool("Search advisors by years of experience")
    public String searchAdvisorsByExperience(int minYearsExperience) {
        List<Map<String, Object>> experiencedAdvisors = advisors.stream()
                .filter(advisor -> {
                    Object experienceObj = advisor.get("years_experience");
                    if (experienceObj instanceof Number) {
                        int experience = ((Number) experienceObj).intValue();
                        return experience >= minYearsExperience;
                    }
                    return false;
                })
                .sorted((a1, a2) -> {
                    int exp1 = ((Number) a1.get("years_experience")).intValue();
                    int exp2 = ((Number) a2.get("years_experience")).intValue();
                    return Integer.compare(exp2, exp1); // Descending order
                })
                .collect(Collectors.toList());

        if (experiencedAdvisors.isEmpty()) {
            return "No advisors found with " + minYearsExperience + " or more years of experience";
        }

        StringBuilder result = new StringBuilder();
        result.append("Advisors with ").append(minYearsExperience).append(" or more years of experience:\n");
        for (Map<String, Object> advisor : experiencedAdvisors) {
            result.append(formatAdvisor(advisor)).append("\n");
        }
        return result.toString();
    }

    @Tool("Search advisors by specialty")
    public String searchAdvisorsBySpecialty(String specialty) {
        List<Map<String, Object>> specialtyAdvisors = advisors.stream()
                .filter(advisor -> {
                    Object specialtiesObj = advisor.get("specialties");
                    if (specialtiesObj instanceof List) {
                        List<String> specialties = (List<String>) specialtiesObj;
                        return specialties.stream().anyMatch(s -> 
                            s.toLowerCase().contains(specialty.toLowerCase().trim()));
                    }
                    return false;
                })
                .collect(Collectors.toList());

        if (specialtyAdvisors.isEmpty()) {
            return "No advisors found with specialty in " + specialty;
        }

        StringBuilder result = new StringBuilder();
        result.append("Advisors with specialty in ").append(specialty).append(":\n");
        for (Map<String, Object> advisor : specialtyAdvisors) {
            result.append(formatAdvisor(advisor)).append("\n");
        }
        return result.toString();
    }

    @Tool("Get advisor by ID")
    public String getAdvisorById(String advisorId) {
        Map<String, Object> advisor = advisors.stream()
            .filter(a -> advisorId.equals(a.get("advisor_id")))
            .findFirst()
            .orElse(null);
        
        if (advisor == null) {
            return "Advisor " + advisorId + " not found";
        }
        
        return formatAdvisor(advisor);
    }

    @Tool("Search advisors by client satisfaction rating")
    public String searchAdvisorsByRating(double minRating) {
        List<Map<String, Object>> highRatedAdvisors = advisors.stream()
                .filter(advisor -> {
                    Object ratingObj = advisor.get("client_satisfaction_rating");
                    if (ratingObj instanceof Number) {
                        double rating = ((Number) ratingObj).doubleValue();
                        return rating >= minRating;
                    }
                    return false;
                })
                .sorted((a1, a2) -> {
                    double rating1 = ((Number) a1.get("client_satisfaction_rating")).doubleValue();
                    double rating2 = ((Number) a2.get("client_satisfaction_rating")).doubleValue();
                    return Double.compare(rating2, rating1); // Descending order
                })
                .collect(Collectors.toList());

        if (highRatedAdvisors.isEmpty()) {
            return "No advisors found with rating " + minRating + " or above";
        }

        StringBuilder result = new StringBuilder();
        result.append("Advisors with rating ").append(minRating).append(" or above:\n");
        for (Map<String, Object> advisor : highRatedAdvisors) {
            result.append(formatAdvisor(advisor)).append("\n");
        }
        return result.toString();
    }

    @Tool("Get all available advisors")
    public String getAllAdvisors() {
        if (advisors.isEmpty()) {
            return "No advisors available";
        }

        StringBuilder result = new StringBuilder();
        result.append("All available advisors:\n");
        for (Map<String, Object> advisor : advisors) {
            result.append(formatAdvisor(advisor)).append("\n");
        }
        return result.toString();
    }

    private String formatAdvisor(Map<String, Object> advisor) {
        StringBuilder sb = new StringBuilder();
        sb.append("- ").append(advisor.get("name"))
          .append(" (").append(advisor.get("title")).append(")")
          .append(" - Firm: ").append(advisor.get("firm"))
          .append(", Experience: ").append(advisor.get("years_experience")).append(" years")
          .append(", Rating: ").append(advisor.get("client_satisfaction_rating"));
        
        Object location = advisor.get("location");
        if (location != null) {
            sb.append("\n  Location: ").append(location);
        }
        
        Object specialties = advisor.get("specialties");
        if (specialties instanceof List) {
            sb.append("\n  Specialties: ").append(specialties);
        }
        
        Object minAccount = advisor.get("minimum_account_size");
        if (minAccount != null) {
            sb.append("\n  Min Account: $").append(minAccount);
        }
        
        return sb.toString();
    }
}