package com.samples.a2a;

import jakarta.enterprise.context.ApplicationScoped;
import jakarta.enterprise.inject.Produces;
import jakarta.inject.Inject;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import io.a2a.server.agentexecution.AgentExecutor;
import io.a2a.server.agentexecution.RequestContext;
import io.a2a.server.events.EventQueue;
import io.a2a.server.tasks.TaskUpdater;
import io.a2a.spec.JSONRPCError;
import io.a2a.spec.Message;
import io.a2a.spec.Part;
import io.a2a.spec.Task;
import io.a2a.spec.TaskNotCancelableError;
import io.a2a.spec.TaskState;
import io.a2a.spec.TextPart;

/**
 * Producer for hospital search agent executor.
 * This class is final and not designed for extension.
 */
@ApplicationScoped
public final class HospitalSearchAgentExecutorProducer {

    /**
     * The hospital search agent instance.
     */
    @Inject
    private HospitalSearchAgent hospitalSearchAgent;

    /**
     * Gets the hospital search agent.
     *
     * @return the hospital search agent
     */
    public HospitalSearchAgent getHospitalSearchAgent() {
        return hospitalSearchAgent;
    }

    /**
     * Produces the agent executor for the hospital search agent.
     *
     * @return the configured agent executor
     */
    @Produces
    public AgentExecutor agentExecutor() {
        return new HospitalSearchAgentExecutor(getHospitalSearchAgent());
    }

    /**
     * Hospital search agent executor implementation.
     */
    private static class HospitalSearchAgentExecutor implements AgentExecutor {

        /**
         * The hospital search agent instance.
         */
        private final HospitalSearchAgent agent;
        
        /**
         * Timeout for agent execution in seconds.
         */
        private static final int EXECUTION_TIMEOUT_SECONDS = 90;

        /**
         * Constructor for HospitalSearchAgentExecutor.
         *
         * @param hospitalSearchAgentInstance the hospital search agent instance
         */
        HospitalSearchAgentExecutor(final HospitalSearchAgent hospitalSearchAgentInstance) {
            this.agent = hospitalSearchAgentInstance;
        }

        @Override
        public void execute(final RequestContext context, final EventQueue eventQueue) throws JSONRPCError {
            final TaskUpdater updater = new TaskUpdater(context, eventQueue);

            try {
                // Mark the task as submitted and start working on it
                if (context.getTask() == null) {
                    updater.submit();
                }
                updater.startWork();

                // Extract the text from the message
                final String assignment = extractTextFromMessage(context.getMessage());
                
                if (assignment == null || assignment.trim().isEmpty()) {
                    System.err.println("Warning: Empty or null message received");
                    final TextPart errorPart = new TextPart(
                        "I received an empty message. Please provide a hospital search query.", 
                        null
                    );
                    updater.addArtifact(List.of(errorPart), null, null, null);
                    updater.complete();
                    return;
                }

                // Call the hospital search agent with timeout and error handling
                String response = executeWithTimeout(assignment);
                
                if (response == null || response.trim().isEmpty()) {
                    response = "I apologize, but I couldn't process your hospital search request. Please try rephrasing your query with more specific details.";
                }

                // Create the response part
                final TextPart responsePart = new TextPart(response, null);
                final List<Part<?>> parts = List.of(responsePart);

                // Add the response as an artifact and complete the task
                updater.addArtifact(parts, null, null, null);
                updater.complete();
                
            } catch (Exception e) {
                System.err.println("Critical error in hospital search executor: " + e.getMessage());
                e.printStackTrace();
                
                try {
                    final TextPart errorPart = new TextPart(
                        "I apologize, but I encountered a system error while processing your hospital search request. " +
                        "Please try again with a simpler query, or contact support if the issue persists.", 
                        null
                    );
                    updater.addArtifact(List.of(errorPart), null, null, null);
                    updater.complete();
                } catch (Exception completionError) {
                    System.err.println("Failed to complete task after error: " + completionError.getMessage());
                    throw new JSONRPCError(-32603, "Failed to handle hospital search request", null);
                }
            }
        }

        /**
         * Executes the agent search with timeout handling.
         *
         * @param assignment the search query
         * @return the search response
         */
        private String executeWithTimeout(String assignment) {
            try {
                CompletableFuture<String> future = CompletableFuture.supplyAsync(() -> {
                    try {
                        return agent.searchHospitals(assignment);
                    } catch (Exception e) {
                        System.err.println("Error during agent execution: " + e.getMessage());
                        return "I encountered an error while searching for hospital information. Error details: " + e.getMessage();
                    }
                });
                
                String result = future.get(EXECUTION_TIMEOUT_SECONDS, TimeUnit.SECONDS);
                
                if (result == null) {
                    return "I received a null response from the hospital search system. Please try your query again.";
                }
                
                return result;
                
            } catch (TimeoutException e) {
                System.err.println("Hospital search request timed out after " + EXECUTION_TIMEOUT_SECONDS + " seconds");
                return "Your hospital search request took too long to process. Please try a simpler query or search for fewer hospitals at once.";
            } catch (InterruptedException e) {
                System.err.println("Hospital search request was interrupted: " + e.getMessage());
                Thread.currentThread().interrupt();
                return "Your hospital search request was interrupted. Please try again.";
            } catch (Exception e) {
                System.err.println("Unexpected error during hospital search execution: " + e.getMessage());
                return "I encountered an unexpected error while searching for hospitals. Please try rephrasing your query or contact support.";
            }
        }

        /**
         * Extracts text content from a message.
         *
         * @param message the message to extract text from
         * @return the extracted text content
         */
        private String extractTextFromMessage(final Message message) {
            try {
                final StringBuilder textBuilder = new StringBuilder();
                if (message != null && message.getParts() != null) {
                    for (final Part part : message.getParts()) {
                        if (part instanceof TextPart textPart) {
                            String text = textPart.getText();
                            if (text != null) {
                                textBuilder.append(text);
                            }
                        }
                    }
                }
                return textBuilder.toString();
            } catch (Exception e) {
                System.err.println("Error extracting text from message: " + e.getMessage());
                return "";
            }
        }

        @Override
        public void cancel(final RequestContext context, final EventQueue eventQueue) throws JSONRPCError {
            try {
                final Task task = context.getTask();

                if (task == null) {
                    throw new TaskNotCancelableError();
                }

                if (task.getStatus().state() == TaskState.CANCELED) {
                    // Task already cancelled
                    throw new TaskNotCancelableError();
                }

                if (task.getStatus().state() == TaskState.COMPLETED) {
                    // Task already completed
                    throw new TaskNotCancelableError();
                }

                // Cancel the task
                final TaskUpdater updater = new TaskUpdater(context, eventQueue);
                updater.cancel();
                
            } catch (TaskNotCancelableError e) {
                throw e; // Re-throw as expected
            } catch (Exception e) {
                System.err.println("Error canceling hospital search task: " + e.getMessage());
                throw new JSONRPCError(-32603, "Failed to cancel hospital search task", null);
            }
        }
    }
}