package com.samples.a2a;

import dev.langchain4j.agent.tool.Tool;
import jakarta.enterprise.context.ApplicationScoped;
import org.yaml.snakeyaml.Yaml;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Hospital search tool for querying hospital data from YAML file.
 */
@ApplicationScoped
public class HospitalSearchTool {

    private List<Map<String, Object>> hospitals;

    public HospitalSearchTool() {
        loadHospitalData();
    }

    @SuppressWarnings("unchecked")
    private void loadHospitalData() {
        try {
            Yaml yaml = new Yaml();
            InputStream inputStream = getClass().getResourceAsStream("/hospital_data.yaml");
            if (inputStream == null) {
                inputStream = getClass().getResourceAsStream("hospital_data.yaml");
            }
            
            if (inputStream != null) {
                Map<String, Object> data = yaml.load(inputStream);
                Map<String, Object> hospitalsData = (Map<String, Object>) data.get("hospitals");
                this.hospitals = (List<Map<String, Object>>) hospitalsData.get("hospital_list");
            } else {
                this.hospitals = new ArrayList<>();
                System.err.println("Warning: hospital_data.yaml not found, using empty hospital list");
            }
        } catch (Exception e) {
            this.hospitals = new ArrayList<>();
            System.err.println("Error loading hospital data: " + e.getMessage());
        }
    }

    @Tool("search hospitals by city")
    public String searchHospitalsByCity(String city) {
        try {
            if (city == null || city.trim().isEmpty()) {
                return "Please provide a valid city name for search";
            }
            
            List<Map<String, Object>> cityHospitals = hospitals.stream()
                    .filter(hospital -> {
                        String hospitalCity = (String) hospital.get("city");
                        return hospitalCity != null && hospitalCity.equalsIgnoreCase(city.trim());
                    })
                    .collect(Collectors.toList());

            if (cityHospitals.isEmpty()) {
                return "No hospitals found in " + city + ". Please check the city name or try a different location.";
            }

            StringBuilder result = new StringBuilder();
            result.append("Hospitals in ").append(city).append(":\n");
            for (Map<String, Object> hospital : cityHospitals) {
                result.append(formatHospital(hospital)).append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error searching hospitals by city: " + e.getMessage());
            return "An error occurred while searching for hospitals in " + city + ". Please try again.";
        }
    }

    @Tool("search hospitals by established year")
    public String searchHospitalsByEstablishedYear(int minYear, int maxYear) {
        try {
            if (minYear > maxYear) {
                return "Invalid year range: minimum year cannot be greater than maximum year";
            }
            
            if (minYear < 1000 || maxYear > 2030) {
                return "Please provide a reasonable year range (1000-2030)";
            }
            
            List<Map<String, Object>> yearHospitals = hospitals.stream()
                    .filter(hospital -> {
                        Object yearObj = hospital.get("established_year");
                        if (yearObj instanceof Number) {
                            int year = ((Number) yearObj).intValue();
                            return year >= minYear && year <= maxYear;
                        }
                        return false;
                    })
                    .sorted((h1, h2) -> {
                        int year1 = ((Number) h1.get("established_year")).intValue();
                        int year2 = ((Number) h2.get("established_year")).intValue();
                        return Integer.compare(year1, year2);
                    })
                    .collect(Collectors.toList());

            if (yearHospitals.isEmpty()) {
                return "No hospitals found established between " + minYear + " and " + maxYear;
            }

            StringBuilder result = new StringBuilder();
            result.append("Hospitals established between ").append(minYear).append(" and ").append(maxYear).append(":\n");
            for (Map<String, Object> hospital : yearHospitals) {
                result.append(formatHospital(hospital)).append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error searching hospitals by year: " + e.getMessage());
            return "An error occurred while searching for hospitals by established year. Please try again.";
        }
    }

    @Tool("search hospitals by bed capacity")
    public String searchHospitalsByBedCapacity(int minCapacity) {
        try {
            if (minCapacity <= 0) {
                return "Please provide a valid bed capacity (greater than 0)";
            }
            
            if (minCapacity > 50000) {
                return "Please provide a reasonable bed capacity (less than 50,000)";
            }
            
            List<Map<String, Object>> capacityHospitals = hospitals.stream()
                    .filter(hospital -> {
                        Object capacityObj = hospital.get("bed_capacity");
                        if (capacityObj instanceof Number) {
                            int capacity = ((Number) capacityObj).intValue();
                            return capacity >= minCapacity;
                        }
                        return false;
                    })
                    .sorted((h1, h2) -> {
                        int capacity1 = ((Number) h1.get("bed_capacity")).intValue();
                        int capacity2 = ((Number) h2.get("bed_capacity")).intValue();
                        return Integer.compare(capacity2, capacity1); // Descending order
                    })
                    .collect(Collectors.toList());

            if (capacityHospitals.isEmpty()) {
                return "No hospitals found with bed capacity of " + minCapacity + " or more";
            }

            StringBuilder result = new StringBuilder();
            result.append("Hospitals with bed capacity of ").append(minCapacity).append(" or more:\n");
            for (Map<String, Object> hospital : capacityHospitals) {
                result.append(formatHospital(hospital)).append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error searching hospitals by capacity: " + e.getMessage());
            return "An error occurred while searching for hospitals by bed capacity. Please try again.";
        }
    }

    @Tool("search hospital by name")
    public String searchHospitalByName(String hospitalName) {
        try {
            if (hospitalName == null || hospitalName.trim().isEmpty()) {
                return "Please provide a valid hospital name for search";
            }
            
            List<Map<String, Object>> foundHospitals = hospitals.stream()
                    .filter(hospital -> {
                        String name = (String) hospital.get("name");
                        return name != null && name.toLowerCase().contains(hospitalName.toLowerCase().trim());
                    })
                    .collect(Collectors.toList());

            if (foundHospitals.isEmpty()) {
                return "No hospital found with name '" + hospitalName + "'";
            }

            if (foundHospitals.size() == 1) {
                Map<String, Object> hospital = foundHospitals.get(0);
                StringBuilder result = new StringBuilder();
                result.append("Hospital found:\n\n");
                result.append("🏥 **").append(hospital.get("name")).append("**\n");
                result.append("   📍 Location: ").append(hospital.get("city")).append(", ").append(hospital.get("country")).append("\n");
                result.append("   📅 Established: ").append(hospital.get("established_year")).append("\n");
                result.append("   🛏️ Bed Capacity: ").append(hospital.get("bed_capacity")).append("\n");
                result.append("   🚨 Emergency Services: ").append(hospital.get("emergency_services")).append("\n");
                
                Object departments = hospital.get("departments");
                if (departments instanceof List) {
                    List<String> deptList = (List<String>) departments;
                    result.append("   🏥 Departments: ").append(String.join(", ", deptList)).append("\n");
                }
                return result.toString();
            }

            StringBuilder result = new StringBuilder();
            result.append("Found ").append(foundHospitals.size()).append(" hospitals matching '").append(hospitalName).append("':\n");
            for (Map<String, Object> hospital : foundHospitals) {
                result.append(formatHospital(hospital)).append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error searching hospital by name: " + e.getMessage());
            return "An error occurred while searching for hospital '" + hospitalName + "'. Please try again.";
        }
    }

    @Tool("search_hospitals_by_departments")
    public String searchHospitalsByDepartments(String department) {
        try {
            if (department == null || department.trim().isEmpty()) {
                return "Please provide a valid department name for search";
            }
            
            List<Map<String, Object>> departmentHospitals = hospitals.stream()
                    .filter(hospital -> {
                        Object departmentsObj = hospital.get("departments");
                        if (departmentsObj instanceof List) {
                            List<String> departments = (List<String>) departmentsObj;
                            return departments.stream().anyMatch(dept -> 
                                dept.toLowerCase().contains(department.toLowerCase().trim()));
                        }
                        return false;
                    })
                    .collect(Collectors.toList());

            if (departmentHospitals.isEmpty()) {
                return "No hospitals found with " + department + " department";
            }

            StringBuilder result = new StringBuilder();
            result.append("Hospitals with ").append(department).append(" department:\n");
            for (Map<String, Object> hospital : departmentHospitals) {
                result.append(formatHospital(hospital)).append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error searching hospitals by department: " + e.getMessage());
            return "An error occurred while searching for hospitals by department. Please try again.";
        }
    }

    @Tool("get_all_hospitals")
    public String getAllHospitals() {
        try {
            if (hospitals.isEmpty()) {
                return "No hospitals available in the database";
            }

            StringBuilder result = new StringBuilder();
            result.append("All available hospitals:\n");
            for (Map<String, Object> hospital : hospitals) {
                result.append(formatHospital(hospital)).append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error getting all hospitals: " + e.getMessage());
            return "An error occurred while retrieving hospital information. Please try again.";
        }
    }

    @Tool("search_hospitals_by_country")
    public String searchHospitalsByCountry(String country) {
        try {
            if (country == null || country.trim().isEmpty()) {
                return "Please provide a valid country name for search";
            }
            
            List<Map<String, Object>> countryHospitals = hospitals.stream()
                    .filter(hospital -> {
                        String hospitalCountry = (String) hospital.get("country");
                        return hospitalCountry != null && hospitalCountry.equalsIgnoreCase(country.trim());
                    })
                    .collect(Collectors.toList());

            if (countryHospitals.isEmpty()) {
                return "No hospitals found in " + country + ". Please check the country name or try a different location.";
            }

            StringBuilder result = new StringBuilder();
            result.append("Hospitals in ").append(country).append(":\n");
            for (Map<String, Object> hospital : countryHospitals) {
                result.append(formatHospital(hospital)).append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error searching hospitals by country: " + e.getMessage());
            return "An error occurred while searching for hospitals in " + country + ". Please try again.";
        }
    }

    @Tool("get_established_years_by_country")
    public String getHospitalEstablishedYearsByCountry(String country) {
        try {
            if (country == null || country.trim().isEmpty()) {
                return "Please provide a valid country name for search";
            }
            
            List<Map<String, Object>> countryHospitals = hospitals.stream()
                    .filter(hospital -> {
                        String hospitalCountry = (String) hospital.get("country");
                        return hospitalCountry != null && hospitalCountry.equalsIgnoreCase(country.trim());
                    })
                    .sorted((h1, h2) -> {
                        int year1 = ((Number) h1.get("established_year")).intValue();
                        int year2 = ((Number) h2.get("established_year")).intValue();
                        return Integer.compare(year1, year2);
                    })
                    .collect(Collectors.toList());

            if (countryHospitals.isEmpty()) {
                return "No hospitals found in " + country + ". Please check the country name.";
            }

            StringBuilder result = new StringBuilder();
            result.append("Established years of hospitals in ").append(country).append(":\n");
            for (Map<String, Object> hospital : countryHospitals) {
                result.append("- ").append(hospital.get("name"))
                    .append(": ").append(hospital.get("established_year"))
                    .append(" (").append(hospital.get("city")).append(")\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error getting established years by country: " + e.getMessage());
            return "An error occurred while retrieving hospital established years for " + country + ". Please try again.";
        }
    }

    @Tool("get_hospital_details_by_country")
    public String getHospitalDetailsByCountry(String country) {
        try {
            if (country == null || country.trim().isEmpty()) {
                return "Please provide a valid country name for search";
            }
            
            List<Map<String, Object>> countryHospitals = hospitals.stream()
                    .filter(hospital -> {
                        String hospitalCountry = (String) hospital.get("country");
                        return hospitalCountry != null && hospitalCountry.equalsIgnoreCase(country.trim());
                    })
                    .collect(Collectors.toList());

            if (countryHospitals.isEmpty()) {
                return "No hospitals found in " + country + ". Please check the country name.";
            }

            StringBuilder result = new StringBuilder();
            result.append("Detailed information for hospitals in ").append(country).append(":\n\n");
            for (Map<String, Object> hospital : countryHospitals) {
                result.append("🏥 **").append(hospital.get("name")).append("**\n");
                result.append("   📍 Location: ").append(hospital.get("city")).append(", ").append(country).append("\n");
                result.append("   📅 Established: ").append(hospital.get("established_year")).append("\n");
                result.append("   🛏️ Bed Capacity: ").append(hospital.get("bed_capacity")).append("\n");
                result.append("   🚨 Emergency Services: ").append(hospital.get("emergency_services")).append("\n");
                
                Object departments = hospital.get("departments");
                if (departments instanceof List) {
                    List<String> deptList = (List<String>) departments;
                    result.append("   🏥 Departments: ").append(String.join(", ", deptList)).append("\n");
                }
                result.append("\n");
            }
            return result.toString();
            
        } catch (Exception e) {
            System.err.println("Error getting hospital details by country: " + e.getMessage());
            return "An error occurred while retrieving detailed hospital information for " + country + ". Please try again.";
        }
    }

    @Tool("search_american_hospitals")
    public String searchAmericanHospitals() {
        return getHospitalDetailsByCountry("USA");
    }

    @Tool("get_american_hospital_years")
    public String getAmericanHospitalEstablishedYears() {
        return getHospitalEstablishedYearsByCountry("USA");
    }

    private String formatHospital(Map<String, Object> hospital) {
        try {
            StringBuilder sb = new StringBuilder();
            sb.append("- ").append(hospital.get("name"))
              .append(" (").append(hospital.get("city")).append(", ").append(hospital.get("country")).append(")")
              .append(" - Established: ").append(hospital.get("established_year"))
              .append(", Beds: ").append(hospital.get("bed_capacity"))
              .append(", Emergency: ").append(hospital.get("emergency_services"));
            
            Object departments = hospital.get("departments");
            if (departments instanceof List) {
                List<String> deptList = (List<String>) departments;
                sb.append("\n  Departments: ").append(String.join(", ", deptList));
            }
            
            Object accreditation = hospital.get("accreditation");
            if (accreditation != null) {
                sb.append("\n  Accreditation: ").append(accreditation);
            }
            
            Object contact = hospital.get("contact");
            if (contact != null) {
                sb.append("\n  Contact: ").append(contact);
            }
            
            return sb.toString();
        } catch (Exception e) {
            System.err.println("Error formatting hospital: " + e.getMessage());
            return "- " + hospital.get("name") + " (Error formatting details)";
        }
    }
}