package com.samples.a2a;

import dev.langchain4j.agent.tool.Tool;
import jakarta.enterprise.context.ApplicationScoped;
import org.yaml.snakeyaml.Yaml;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Restaurant search tool for querying restaurant data from YAML file.
 */
@ApplicationScoped
public class RestaurantSearchTool {

    private List<Map<String, Object>> restaurants;

    public RestaurantSearchTool() {
        loadRestaurantData();
    }

    @SuppressWarnings("unchecked")
    private void loadRestaurantData() {
        try {
            Yaml yaml = new Yaml();
            InputStream inputStream = getClass().getResourceAsStream("/restaurant_data.yaml");
            if (inputStream == null) {
                // 如果在resources下找不到，尝试从同级目录加载
                inputStream = getClass().getResourceAsStream("restaurant_data.yaml");
            }
            
            if (inputStream != null) {
                Map<String, Object> data = yaml.load(inputStream);
                Map<String, Object> restaurantsData = (Map<String, Object>) data.get("restaurants");
                this.restaurants = (List<Map<String, Object>>) restaurantsData.get("restaurant_list");
            } else {
                this.restaurants = new ArrayList<>();
                System.err.println("Warning: restaurant_data.yaml not found, using empty restaurant list");
            }
        } catch (Exception e) {
            this.restaurants = new ArrayList<>();
            System.err.println("Error loading restaurant data: " + e.getMessage());
        }
    }

    @Tool("Search restaurants by city name")
    public String searchRestaurantsByCity(String city) {
        List<Map<String, Object>> cityRestaurants = restaurants.stream()
                .filter(restaurant -> {
                    String restaurantCity = (String) restaurant.get("city");
                    return restaurantCity != null && restaurantCity.equalsIgnoreCase(city.trim());
                })
                .collect(Collectors.toList());

        if (cityRestaurants.isEmpty()) {
            return "No restaurants found in " + city;
        }

        StringBuilder result = new StringBuilder();
        result.append("Restaurants in ").append(city).append(":\n");
        for (Map<String, Object> restaurant : cityRestaurants) {
            result.append(formatRestaurant(restaurant)).append("\n");
        }
        return result.toString();
    }

    @Tool("Search restaurants by cuisine type")
    public String searchRestaurantsByCuisine(String cuisineType) {
        List<Map<String, Object>> cuisineRestaurants = restaurants.stream()
                .filter(restaurant -> {
                    String restaurantType = (String) restaurant.get("restaurant_type");
                    return restaurantType != null && restaurantType.toLowerCase().contains(cuisineType.toLowerCase().trim());
                })
                .collect(Collectors.toList());

        if (cuisineRestaurants.isEmpty()) {
            return "No " + cuisineType + " restaurants found";
        }

        StringBuilder result = new StringBuilder();
        result.append(cuisineType).append(" restaurants:\n");
        for (Map<String, Object> restaurant : cuisineRestaurants) {
            result.append(formatRestaurant(restaurant)).append("\n");
        }
        return result.toString();
    }

    @Tool("Search restaurants by rating above specified value")
    public String searchRestaurantsByRating(double minRating) {
        List<Map<String, Object>> highRatedRestaurants = restaurants.stream()
                .filter(restaurant -> {
                    Object ratingObj = restaurant.get("rating");
                    if (ratingObj instanceof Number) {
                        double rating = ((Number) ratingObj).doubleValue();
                        return rating >= minRating;
                    }
                    return false;
                })
                .sorted((r1, r2) -> {
                    double rating1 = ((Number) r1.get("rating")).doubleValue();
                    double rating2 = ((Number) r2.get("rating")).doubleValue();
                    return Double.compare(rating2, rating1); // 降序排列
                })
                .collect(Collectors.toList());

        if (highRatedRestaurants.isEmpty()) {
            return "No restaurants found with rating " + minRating + " or above";
        }

        StringBuilder result = new StringBuilder();
        result.append("Restaurants with rating ").append(minRating).append(" or above:\n");
        for (Map<String, Object> restaurant : highRatedRestaurants) {
            result.append(formatRestaurant(restaurant)).append("\n");
        }
        return result.toString();
    }

    @Tool("Search restaurants by price range (per capita)")
    public String searchRestaurantsByPrice(double maxPrice) {
        List<Map<String, Object>> budgetRestaurants = restaurants.stream()
                .filter(restaurant -> {
                    Object priceObj = restaurant.get("per_capita");
                    if (priceObj instanceof Number) {
                        double price = ((Number) priceObj).doubleValue();
                        return price <= maxPrice;
                    }
                    return false;
                })
                .sorted((r1, r2) -> {
                    double price1 = ((Number) r1.get("per_capita")).doubleValue();
                    double price2 = ((Number) r2.get("per_capita")).doubleValue();
                    return Double.compare(price1, price2); // 升序排列
                })
                .collect(Collectors.toList());

        if (budgetRestaurants.isEmpty()) {
            return "No restaurants found under $" + maxPrice + " per person";
        }

        StringBuilder result = new StringBuilder();
        result.append("Restaurants under $").append(maxPrice).append(" per person:\n");
        for (Map<String, Object> restaurant : budgetRestaurants) {
            result.append(formatRestaurant(restaurant)).append("\n");
        }
        return result.toString();
    }

    @Tool("Get all available restaurants")
    public String getAllRestaurants() {
        if (restaurants.isEmpty()) {
            return "No restaurants available";
        }

        StringBuilder result = new StringBuilder();
        result.append("All available restaurants:\n");
        for (Map<String, Object> restaurant : restaurants) {
            result.append(formatRestaurant(restaurant)).append("\n");
        }
        return result.toString();
    }

    @Tool("Search restaurants by country")
    public String searchRestaurantsByCountry(String country) {
        List<Map<String, Object>> countryRestaurants = restaurants.stream()
                .filter(restaurant -> {
                    String restaurantCountry = (String) restaurant.get("country");
                    return restaurantCountry != null && restaurantCountry.equalsIgnoreCase(country.trim());
                })
                .collect(Collectors.toList());

        if (countryRestaurants.isEmpty()) {
            return "No restaurants found in " + country;
        }

        StringBuilder result = new StringBuilder();
        result.append("Restaurants in ").append(country).append(":\n");
        for (Map<String, Object> restaurant : countryRestaurants) {
            result.append(formatRestaurant(restaurant)).append("\n");
        }
        return result.toString();
    }

    private String formatRestaurant(Map<String, Object> restaurant) {
        StringBuilder sb = new StringBuilder();
        sb.append("- ").append(restaurant.get("name"))
          .append(" (").append(restaurant.get("city")).append(", ").append(restaurant.get("country")).append(")")
          .append(" - Rating: ").append(restaurant.get("rating"))
          .append(", Price: $").append(restaurant.get("per_capita"))
          .append(", Type: ").append(restaurant.get("restaurant_type"));
        
        Object address = restaurant.get("address");
        if (address != null) {
            sb.append("\n  Address: ").append(address);
        }
        
        Object contact = restaurant.get("contact_info");
        if (contact != null) {
            sb.append("\n  Contact: ").append(contact);
        }
        
        Object specialties = restaurant.get("specialty_dishes");
        if (specialties instanceof List) {
            sb.append("\n  Specialties: ").append(specialties);
        }
        
        return sb.toString();
    }
}