import * as yaml from 'js-yaml';
import * as fs from 'fs';
import * as path from 'path';
import { dirname } from 'path';
import { fileURLToPath } from 'url';

interface Flight {
  flight_number: string;
  departure_city: string;
  departure_country: string;
  departure_airport: string;
  destination_city: string;
  destination_country: string;
  destination_airport: string;
  departure_time: string;
  arrival_time: string;
  flight_duration: string;
  price: number;
  currency: string;
  airline: string;
  aircraft_type: string;
  class: string;
  stops: string;
  reviews: string[];
  contact_info: string;
}

interface FlightData {
  flights: {
    flight_list: Flight[];
  };
}

export class FlightSearchTool {
  private flights: Flight[] = [];

  constructor() {
    this.loadFlightData();
  }

  private loadFlightData(): void {
    try {
      const __dirname = dirname(fileURLToPath(import.meta.url));
      const yamlPath = path.join(__dirname, 'flight_data.yaml');
      
      if (fs.existsSync(yamlPath)) {
        const fileContents = fs.readFileSync(yamlPath, 'utf8');
        const data = yaml.load(fileContents) as FlightData;
        this.flights = data.flights.flight_list;
        console.log(`[FlightSearchTool] Loaded ${this.flights.length} flights from YAML`);
      } else {
        console.error(`[FlightSearchTool] Warning: flight_data.yaml not found at ${yamlPath}`);
        this.flights = [];
      }
    } catch (error) {
      console.error('[FlightSearchTool] Error loading flight data:', error);
      this.flights = [];
    }
  }

  public searchFlightsByDepartureCity(city: string): string {
    const cityFlights = this.flights.filter(flight => 
      flight.departure_city.toLowerCase().includes(city.toLowerCase().trim())
    );

    if (cityFlights.length === 0) {
      return `No flights found departing from ${city}`;
    }

    let result = `Flights departing from ${city}:\n`;
    cityFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }
  public searchFlightsByMinDuration(minHours: number): string {
    const longFlights = this.flights.filter(flight => {
      const duration = this.parseDuration(flight.flight_duration);
      return duration >= minHours;
    }).sort((a, b) => this.parseDuration(b.flight_duration) - this.parseDuration(a.flight_duration));

    if (longFlights.length === 0) {
      return `No flights found over ${minHours} hours`;
    }

    let result = `Flights over ${minHours} hours:\n`;
    longFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }
  public searchFlightsByDestinationCity(city: string): string {
    const cityFlights = this.flights.filter(flight => 
      flight.destination_city.toLowerCase().includes(city.toLowerCase().trim())
    );

    if (cityFlights.length === 0) {
      return `No flights found going to ${city}`;
    }

    let result = `Flights going to ${city}:\n`;
    cityFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  public searchFlightsByAirline(airline: string): string {
    const airlineFlights = this.flights.filter(flight => 
      flight.airline.toLowerCase().includes(airline.toLowerCase().trim())
    );

    if (airlineFlights.length === 0) {
      return `No flights found for ${airline}`;
    }

    let result = `${airline} flights:\n`;
    airlineFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  public searchFlightsByPriceRange(maxPrice: number): string {
    const budgetFlights = this.flights.filter(flight => flight.price <= maxPrice)
      .sort((a, b) => a.price - b.price);

    if (budgetFlights.length === 0) {
      return `No flights found under $${maxPrice}`;
    }

    let result = `Flights under $${maxPrice}:\n`;
    budgetFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  public searchFlightsByDuration(maxHours: number): string {
    const shortFlights = this.flights.filter(flight => {
      const duration = this.parseDuration(flight.flight_duration);
      return duration <= maxHours;
    }).sort((a, b) => this.parseDuration(a.flight_duration) - this.parseDuration(b.flight_duration));

    if (shortFlights.length === 0) {
      return `No flights found under ${maxHours} hours`;
    }

    let result = `Flights under ${maxHours} hours:\n`;
    shortFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  public searchDirectFlights(): string {
    const directFlights = this.flights.filter(flight => 
      flight.stops.toLowerCase() === 'direct'
    );

    if (directFlights.length === 0) {
      return 'No direct flights found';
    }

    let result = 'Direct flights:\n';
    directFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  public searchFlightsByRoute(departureCity: string, destinationCity: string): string {
    const routeFlights = this.flights.filter(flight => 
      flight.departure_city.toLowerCase().includes(departureCity.toLowerCase().trim()) &&
      flight.destination_city.toLowerCase().includes(destinationCity.toLowerCase().trim())
    );

    if (routeFlights.length === 0) {
      return `No flights found from ${departureCity} to ${destinationCity}`;
    }

    let result = `Flights from ${departureCity} to ${destinationCity}:\n`;
    routeFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  public getAllFlights(): string {
    if (this.flights.length === 0) {
      return 'No flights available';
    }

    let result = 'All available flights:\n';
    this.flights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  public searchFlightsByCountry(country: string): string {
    const countryFlights = this.flights.filter(flight => 
      flight.departure_country.toLowerCase().includes(country.toLowerCase().trim()) ||
      flight.destination_country.toLowerCase().includes(country.toLowerCase().trim())
    );

    if (countryFlights.length === 0) {
      return `No flights found for ${country}`;
    }

    let result = `Flights related to ${country}:\n`;
    countryFlights.forEach(flight => {
      result += this.formatFlight(flight) + '\n';
    });
    return result;
  }

  private formatFlight(flight: Flight): string {
    return `- ${flight.flight_number} (${flight.airline}): ${flight.departure_city} → ${flight.destination_city}
  Duration: ${flight.flight_duration}, Price: ${flight.price} ${flight.currency}
  Departure: ${flight.departure_time} (${flight.departure_airport})
  Arrival: ${flight.arrival_time} (${flight.destination_airport})
  Aircraft: ${flight.aircraft_type}, Class: ${flight.class}, Stops: ${flight.stops}`;
  }

  private parseDuration(duration: string): number {
    const match = duration.match(/(\d+)h\s*(\d+)?m?/);
    if (match) {
      const hours = parseInt(match[1]) || 0;
      const minutes = parseInt(match[2]) || 0;
      return hours + minutes / 60;
    }
    return 0;
  }
}