import { genkit } from "genkit";
import { dirname } from "path";
import { fileURLToPath } from "url";
import { resolve } from "path";
import OpenAI from 'openai';
import * as dotenv from "dotenv";
import { FlightSearchTool } from './FlightSearchTool.js';

// Load .env file
dotenv.config({ path: resolve(process.cwd(), '.env') });

// Check API key
if (!process.env.OPENAI_API_KEY) {
  console.error("OPENAI_API_KEY environment variable is not set.");
  throw new Error("OPENAI_API_KEY environment variable is not set.");
}

// Create OpenAI client, using portable AI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
  baseURL: "https://api.bianxie.ai/v1"
});

// Create flight search tool instance
const flightTool = new FlightSearchTool();

export const ai = genkit({
  plugins: [],  // Don't use genkit plugins, call OpenAI directly
  promptDir: dirname(fileURLToPath(import.meta.url)),
});

// Custom OpenAI call function with tool support
export async function callOpenAI(messages: any[]) {
  const tools = [
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByDepartureCity",
        description: "Search flights departing from a specific city. Use this when users ask 'What flights depart from [CITY]?' or similar questions.",
        parameters: {
          type: "object",
          properties: {
            city: {
              type: "string",
              description: "The departure city name (e.g., 'Tokyo', 'Paris', 'Bangkok')"
            }
          },
          required: ["city"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByDestinationCity",
        description: "Search flights arriving at a specific city. Use this when users ask 'What flights go to [CITY]?' or 'Which countries can travel to [CITY]?'",
        parameters: {
          type: "object",
          properties: {
            city: {
              type: "string",
              description: "The destination city name (e.g., 'Tokyo', 'Paris', 'Bangkok')"
            }
          },
          required: ["city"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByRoute",
        description: "Search flights from one city to another. Use this for route-specific queries.",
        parameters: {
          type: "object",
          properties: {
            departureCity: {
              type: "string",
              description: "The departure city name"
            },
            destinationCity: {
              type: "string",
              description: "The destination city name"
            }
          },
          required: ["departureCity", "destinationCity"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByCountry",
        description: "Search flights related to a specific country (either departure or destination). Use this for country-based queries.",
        parameters: {
          type: "object",
          properties: {
            country: {
              type: "string",
              description: "The country name (e.g., 'France', 'Japan', 'Thailand')"
            }
          },
          required: ["country"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByAirline",
        description: "Search flights by airline company",
        parameters: {
          type: "object",
          properties: {
            airline: {
              type: "string",
              description: "The airline name (e.g., 'Air France', 'Japan Airlines')"
            }
          },
          required: ["airline"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByPriceRange",
        description: "Search flights under a maximum price limit",
        parameters: {
          type: "object",
          properties: {
            maxPrice: {
              type: "number",
              description: "Maximum price in USD"
            }
          },
          required: ["maxPrice"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByDuration",
        description: "Search flights under maximum duration (shorter flights)",
        parameters: {
          type: "object",
          properties: {
            maxHours: {
              type: "number",
              description: "Maximum flight duration in hours"
            }
          },
          required: ["maxHours"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchFlightsByMinDuration",
        description: "Search flights over minimum duration (longer flights)",
        parameters: {
          type: "object",
          properties: {
            minHours: {
              type: "number",
              description: "Minimum flight duration in hours"
            }
          },
          required: ["minHours"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "getAllFlights",
        description: "Get all available flights in the database. Use this for general flight overview.",
        parameters: {
          type: "object",
          properties: {},
          required: []
        }
      }
    }
  ];

  const response = await openai.chat.completions.create({
    model: "gemini-2.5-flash",
    messages: messages,
    tools: tools,
    tool_choice: "auto"
  });

  const message = response.choices[0].message;
  console.log(`[FlightSearchTool] OpenAI response:`, JSON.stringify(message, null, 2));
  
  // Handle tool calls
  if (message.tool_calls && message.tool_calls.length > 0) {
    const toolResults = [];
    
    for (const toolCall of message.tool_calls) {
      const functionName = toolCall.function.name;
      const functionArgs = JSON.parse(toolCall.function.arguments);
      console.log(`[FlightSearchTool] Calling function: ${functionName} with args:`, functionArgs);
      let result = "";
      
      switch (functionName) {
        case "searchFlightsByDepartureCity":
          result = flightTool.searchFlightsByDepartureCity(functionArgs.city);
          break;
        case "searchFlightsByDestinationCity":
          result = flightTool.searchFlightsByDestinationCity(functionArgs.city);
          break;
        case "searchFlightsByRoute":
          result = flightTool.searchFlightsByRoute(functionArgs.departureCity, functionArgs.destinationCity);
          break;
        case "searchFlightsByCountry":
          result = flightTool.searchFlightsByCountry(functionArgs.country);
          break;
        case "searchFlightsByAirline":
          result = flightTool.searchFlightsByAirline(functionArgs.airline);
          break;
        case "searchFlightsByPriceRange":
          result = flightTool.searchFlightsByPriceRange(functionArgs.maxPrice);
          break;
        case "searchFlightsByDuration":
          result = flightTool.searchFlightsByDuration(functionArgs.maxHours);
          break;
        case "searchFlightsByMinDuration":
          result = flightTool.searchFlightsByMinDuration(functionArgs.minHours);
          break;  
        case "getAllFlights":
          result = flightTool.getAllFlights();
          break;
        default:
          result = "Unknown function called";
      }
      console.log(`[FlightSearchTool] Function ${functionName} result:`, result.substring(0, 200) + '...');
      toolResults.push({
        tool_call_id: toolCall.id,
        role: "tool" as const,
        content: result
      });
    }
    
    // Send tool results back to the model
    const finalMessages = [
      ...messages,
      message,
      ...toolResults
    ];
    
    const finalResponse = await openai.chat.completions.create({
      model: "gemini-2.5-flash",
      messages: finalMessages
    });
    console.log(`[FlightSearchTool] Final response:`, finalResponse.choices[0].message.content);
    return finalResponse.choices[0].message.content || "No response generated";
  }
  
  return message.content || "No response generated";
}