import * as yaml from 'js-yaml';
import * as fs from 'fs';
import * as path from 'path';
import { dirname } from 'path';
import { fileURLToPath } from 'url';

interface Patient {
  name: string;
  nationality: string;
  age: number;
  gender: string;
  medical_history: string[];
  current_medications: string[];
  allergies: string[];
  emergency_contact: string;
  insurance_provider: string;
  last_checkup: string;
}

interface PatientData {
  patients: {
    patient_list: Patient[];
  };
}

export class PatientSearchTool {
  private patients: Patient[] = [];

  constructor() {
    this.loadPatientData();
  }

  private loadPatientData(): void {
    try {
      const __dirname = dirname(fileURLToPath(import.meta.url));
      const yamlPath = path.join(__dirname, 'patient_data.yaml');
      
      if (fs.existsSync(yamlPath)) {
        const fileContents = fs.readFileSync(yamlPath, 'utf8');
        const data = yaml.load(fileContents) as PatientData;
        this.patients = data.patients.patient_list;
        console.log(`[PatientSearchTool] Loaded ${this.patients.length} patients from YAML`);
      } else {
        console.error(`[PatientSearchTool] Warning: patient_data.yaml not found at ${yamlPath}`);
        this.patients = [];
      }
    } catch (error) {
      console.error('[PatientSearchTool] Error loading patient data:', error);
      this.patients = [];
    }
  }

  public searchPatientsByNationality(nationality: string): string {
    const nationalityPatients = this.patients.filter(patient => 
      patient.nationality.toLowerCase().includes(nationality.toLowerCase().trim())
    );

    if (nationalityPatients.length === 0) {
      return `No patients found with nationality ${nationality}`;
    }

    let result = `Patients with nationality ${nationality}:\n`;
    nationalityPatients.forEach(patient => {
      result += this.formatPatient(patient) + '\n';
    });
    return result;
  }

  public searchPatientsByAge(minAge: number, maxAge?: number): string {
    const agePatients = this.patients.filter(patient => {
      if (maxAge) {
        return patient.age >= minAge && patient.age <= maxAge;
      }
      return patient.age >= minAge;
    }).sort((a, b) => a.age - b.age);

    if (agePatients.length === 0) {
      const ageRange = maxAge ? `${minAge}-${maxAge}` : `over ${minAge}`;
      return `No patients found in age range ${ageRange}`;
    }

    const ageRange = maxAge ? `${minAge}-${maxAge}` : `over ${minAge}`;
    let result = `Patients in age range ${ageRange}:\n`;
    agePatients.forEach(patient => {
      result += this.formatPatient(patient) + '\n';
    });
    return result;
  }

  public searchPatientsByName(name: string): string {
    const foundPatients = this.patients.filter(patient => 
      patient.name.toLowerCase().includes(name.toLowerCase().trim())
    );
  
    if (foundPatients.length === 0) {
      return `No patient found with name "${name}"`;
    }
  
    if (foundPatients.length === 1) {
      const patient = foundPatients[0];
      return `Patient found: ${this.formatPatient(patient)}`;
    }
  
    let result = `Found ${foundPatients.length} patients matching "${name}":\n`;
    foundPatients.forEach(patient => {
      result += this.formatPatient(patient) + '\n';
    });
    return result;
  }

  public searchPatientsByMedicalHistory(condition: string): string {
    const conditionPatients = this.patients.filter(patient =>
      patient.medical_history.some(history =>
        history.toLowerCase().includes(condition.toLowerCase().trim())
      )
    );

    if (conditionPatients.length === 0) {
      return `No patients found with medical condition: ${condition}`;
    }

    let result = `Patients with medical condition ${condition}:\n`;
    conditionPatients.forEach(patient => {
      result += this.formatPatient(patient) + '\n';
    });
    return result;
  }

  public getAllPatients(): string {
    if (this.patients.length === 0) {
      return 'No patients available';
    }

    let result = 'All available patients:\n';
    this.patients.forEach(patient => {
      result += this.formatPatient(patient) + '\n';
    });
    return result;
  }

  public searchPatientsByNationalityAndAge(nationality: string, minAge?: number, maxAge?: number): string {
    let filteredPatients = this.patients.filter(patient => 
      patient.nationality.toLowerCase().includes(nationality.toLowerCase().trim())
    );

    if (minAge !== undefined) {
      if (maxAge !== undefined) {
        filteredPatients = filteredPatients.filter(patient => 
          patient.age >= minAge && patient.age <= maxAge
        );
      } else {
        filteredPatients = filteredPatients.filter(patient => 
          patient.age >= minAge
        );
      }
    }

    if (filteredPatients.length === 0) {
      const ageRange = minAge !== undefined ? 
        (maxAge !== undefined ? ` aged ${minAge}-${maxAge}` : ` aged ${minAge}+`) : '';
      return `No ${nationality} patients found${ageRange}`;
    }

    const ageRange = minAge !== undefined ? 
      (maxAge !== undefined ? ` aged ${minAge}-${maxAge}` : ` aged ${minAge}+`) : '';
    let result = `${nationality} patients${ageRange}:\n`;
    
    filteredPatients.sort((a, b) => a.age - b.age);
    filteredPatients.forEach(patient => {
      result += `- ${patient.name} (Age: ${patient.age})\n`;
      result += `  Medical History: ${patient.medical_history.join(', ')}\n`;
      result += `  Medications: ${patient.current_medications.join(', ')}\n`;
      result += `  Allergies: ${patient.allergies.join(', ')}\n\n`;
    });
    return result;
  }

  public searchPatientsByNationalityAndCondition(nationality: string, condition: string): string {
    const filteredPatients = this.patients.filter(patient => 
      patient.nationality.toLowerCase().includes(nationality.toLowerCase().trim()) &&
      patient.medical_history.some(history =>
        history.toLowerCase().includes(condition.toLowerCase().trim())
      )
    );

    if (filteredPatients.length === 0) {
      return `No ${nationality} patients found with medical condition: ${condition}`;
    }

    let result = `${nationality} patients with ${condition}:\n`;
    filteredPatients.forEach(patient => {
      result += this.formatPatient(patient) + '\n';
    });
    return result;
  }

  public getPatientAgesByNationality(nationality: string): string {
    const nationalityPatients = this.patients.filter(patient => 
      patient.nationality.toLowerCase().includes(nationality.toLowerCase().trim())
    );

    if (nationalityPatients.length === 0) {
      return `No patients found with nationality ${nationality}`;
    }

    const ages = nationalityPatients.map(patient => `${patient.name}: ${patient.age} years old`);
    return `Ages of ${nationality} patients:\n${ages.join('\n')}`;
  }
  
  public getAllPatientsBasicInfo(): string {
    if (this.patients.length === 0) {
      return 'No patients available';
    }

    let result = 'All patients basic information:\n';
    this.patients.forEach(patient => {
      result += `- ${patient.name} (${patient.nationality}, Age: ${patient.age})\n`;
    });
    return result;
  }

  public getAllPatientMedicalHistories(): string {
    if (this.patients.length === 0) {
      return 'No patients available';
    }

    let result = 'All patients medical histories:\n';
    this.patients.forEach(patient => {
      result += `${patient.name}: ${patient.medical_history.join(', ')}\n`;
    });
    return result;
  }

  private formatPatient(patient: Patient): string {
    return `- ${patient.name} (${patient.nationality}, Age: ${patient.age})
  Gender: ${patient.gender}
  Medical History: ${patient.medical_history.join(', ')}
  Current Medications: ${patient.current_medications.join(', ')}
  Allergies: ${patient.allergies.join(', ')}
  Insurance: ${patient.insurance_provider}
  Last Checkup: ${patient.last_checkup}
  Emergency Contact: ${patient.emergency_contact}`;
  }
}