import { genkit } from "genkit";
import { dirname } from "path";
import { fileURLToPath } from "url";
import { resolve } from "path";
import OpenAI from 'openai';
import * as dotenv from "dotenv";
import { PatientSearchTool } from './PatientSearchTool.js';

// Load .env file
dotenv.config({ path: resolve(process.cwd(), '.env') });

// Check API key
if (!process.env.OPENAI_API_KEY) {
  console.error("OPENAI_API_KEY environment variable is not set.");
  throw new Error("OPENAI_API_KEY environment variable is not set.");
}

// Create OpenAI client, using portable AI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
  baseURL: "https://api.bianxie.ai/v1"
});

// Create patient search tool instance
const patientTool = new PatientSearchTool();

export const ai = genkit({
  plugins: [],  // Don't use genkit plugins, call OpenAI directly
  promptDir: dirname(fileURLToPath(import.meta.url)),
});

// Custom OpenAI call function with tool support
export async function callOpenAI(messages: any[]) {
  const tools = [
    {
      type: "function" as const,
      function: {
        name: "searchPatientsByNationality",
        description: "Search patients by nationality. Use this when users ask about patients from specific countries.",
        parameters: {
          type: "object",
          properties: {
            nationality: {
              type: "string",
              description: "The nationality name (e.g., 'American', 'French', 'Japanese')"
            }
          },
          required: ["nationality"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchPatientsByName",
        description: "Search for a specific patient by their full name or partial name. Use this when users ask about specific patients like 'What is Emma Johnson's medical history?' or 'Find patient John Smith'.",
        parameters: {
          type: "object",
          properties: {
            name: {
              type: "string",
              description: "The patient name to search for"
            }
          },
          required: ["name"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchPatientsByAge",
        description: "Search patients by age range. Use this when users ask about patients of specific ages.",
        parameters: {
          type: "object",
          properties: {
            minAge: {
              type: "number",
              description: "The minimum age"
            },
            maxAge: {
              type: "number",
              description: "The maximum age (optional)"
            }
          },
          required: ["minAge"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchPatientsByMedicalHistory",
        description: "Search patients by medical condition or history. Use this for medical condition queries.",
        parameters: {
          type: "object",
          properties: {
            condition: {
              type: "string",
              description: "The medical condition name (e.g., 'diabetes', 'hypertension', 'asthma')"
            }
          },
          required: ["condition"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "getAllPatients",
        description: "Get all available patients in the database. Use this for general patient overview.",
        parameters: {
          type: "object",
          properties: {},
          required: []
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchPatientsByNationalityAndAge",
        description: "Search patients by both nationality and age range. Use this for compound queries like 'What is the age of American patients?' or 'Find American patients aged 30-50'.",
        parameters: {
          type: "object",
          properties: {
            nationality: {
              type: "string",
              description: "The nationality name (e.g., 'American', 'French', 'Japanese')"
            },
            minAge: {
              type: "number",
              description: "The minimum age (optional)"
            },
            maxAge: {
              type: "number",
              description: "The maximum age (optional)"
            }
          },
          required: ["nationality"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchPatientsByNationalityAndCondition",
        description: "Search patients by both nationality and medical condition. Use this for queries like 'What American patients have diabetes?'",
        parameters: {
          type: "object",
          properties: {
            nationality: {
              type: "string",
              description: "The nationality name (e.g., 'American', 'French', 'Japanese')"
            },
            condition: {
              type: "string",
              description: "The medical condition name (e.g., 'diabetes', 'hypertension', 'asthma')"
            }
          },
          required: ["nationality", "condition"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "getPatientAgesByNationality",
        description: "Get specific ages of patients from a nationality. Use this when users ask 'What is the age of [nationality] patients?' or 'How old are [nationality] patients?'",
        parameters: {
          type: "object",
          properties: {
            nationality: {
              type: "string",
              description: "The nationality name (e.g., 'American', 'French', 'Japanese')"
            }
          },
          required: ["nationality"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "getAllPatientsBasicInfo",
        description: "Get basic information (name, age, nationality) of all patients. Use this for queries asking about all patient ages, nationalities, or basic demographics.",
        parameters: {
          type: "object",
          properties: {},
          required: []
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "getAllPatientMedicalHistories",
        description: "Get all patients' medical histories. Use this when users ask 'What are the medical histories of patients?' or want comprehensive medical condition information.",
        parameters: {
          type: "object",
          properties: {},
          required: []
        }
      }
    }
  ];

  const response = await openai.chat.completions.create({
    model: "gemini-2.5-flash",
    messages: messages,
    tools: tools,
    tool_choice: "auto"
  });

  const message = response.choices[0].message;
  console.log(`[PatientSearchTool] OpenAI response:`, JSON.stringify(message, null, 2));
  
  // Handle tool calls
  if (message.tool_calls && message.tool_calls.length > 0) {
    const toolResults = [];
    
    for (const toolCall of message.tool_calls) {
      const functionName = toolCall.function.name;
      const functionArgs = JSON.parse(toolCall.function.arguments);
      console.log(`[PatientSearchTool] Calling function: ${functionName} with args:`, functionArgs);
      let result = "";
      
      switch (functionName) {
        case "searchPatientsByNationality":
          result = patientTool.searchPatientsByNationality(functionArgs.nationality);
          break;
        case "searchPatientsByAge":
          result = patientTool.searchPatientsByAge(functionArgs.minAge, functionArgs.maxAge);
          break;
        case "searchPatientsByMedicalHistory":
          result = patientTool.searchPatientsByMedicalHistory(functionArgs.condition);
          break;
        case "getAllPatients":
          result = patientTool.getAllPatients();
          break;
        case "searchPatientsByNationalityAndAge":
          result = patientTool.searchPatientsByNationalityAndAge(functionArgs.nationality, functionArgs.minAge, functionArgs.maxAge);
          break;
        case "searchPatientsByNationalityAndCondition":
          result = patientTool.searchPatientsByNationalityAndCondition(functionArgs.nationality, functionArgs.condition);
          break;
        case "getPatientAgesByNationality":
          result = patientTool.getPatientAgesByNationality(functionArgs.nationality);
          break;
        case "getAllPatientsBasicInfo":
          result = patientTool.getAllPatientsBasicInfo();
          break;
        case "searchPatientsByName":
          result = patientTool.searchPatientsByName(functionArgs.name);
          break;
        case "getAllPatientMedicalHistories":
          result = patientTool.getAllPatientMedicalHistories();
          break;
        default:
          result = "Unknown function called";
      }
      console.log(`[PatientSearchTool] Function ${functionName} result:`, result.substring(0, 200) + '...');
      toolResults.push({
        tool_call_id: toolCall.id,
        role: "tool" as const,
        content: result
      });
    }
    
    // Send tool results back to the model
    const finalMessages = [
      ...messages,
      message,
      ...toolResults
    ];
    
    const finalResponse = await openai.chat.completions.create({
      model: "gemini-2.5-flash",
      messages: finalMessages
    });
    console.log(`[PatientSearchTool] Final response:`, finalResponse.choices[0].message.content);
    return finalResponse.choices[0].message.content || "No response generated";
  }
  
  return message.content || "No response generated";
}