import express from "express";
import { v4 as uuidv4 } from "uuid";
import { coordinateComprehensiveMedicalPlanUpgraded } from './patient_agent_a2a_communicator.js';

import {
  AgentCard,
  Task,
  TaskStatusUpdateEvent,
  TextPart,
  Message,
} from "@a2a-js/sdk";
import {
  InMemoryTaskStore,
  TaskStore,
  AgentExecutor,
  RequestContext,
  ExecutionEventBus,
  DefaultRequestHandler,
} from "@a2a-js/sdk/server";
import { A2AExpressApp } from "@a2a-js/sdk/server/express";
import { MessageData } from "genkit";
import { ai, callOpenAI } from "./genkit.js";

import * as dotenv from "dotenv";
import { resolve } from "path";

dotenv.config({ path: resolve(process.cwd(), '.env') });

if (!process.env.OPENAI_API_KEY) {
  console.error("OPENAI_API_KEY environment variable is not set.");
  throw new Error("OPENAI_API_KEY environment variable is not set.");
}

const patientSearchPrompt = ai.prompt("patient_search");

class PatientSearchAgentExecutor implements AgentExecutor {
  private cancelledTasks = new Set<string>();

  public cancelTask = async (
    taskId: string,
    eventBus: ExecutionEventBus,
  ): Promise<void> => {
    this.cancelledTasks.add(taskId);
  };

  async execute(
    requestContext: RequestContext,
    eventBus: ExecutionEventBus,
  ): Promise<void> {
    const userMessage = requestContext.userMessage;
    const existingTask = requestContext.task;

    const taskId = existingTask?.id || uuidv4();
    const contextId =
      userMessage.contextId || existingTask?.contextId || uuidv4();

    console.log(
      `[PatientSearchAgentExecutor] Processing message ${userMessage.messageId} for task ${taskId} (context: ${contextId})`,
    );

    if (!existingTask) {
      const initialTask: Task = {
        kind: "task",
        id: taskId,
        contextId: contextId,
        status: {
          state: "submitted",
          timestamp: new Date().toISOString(),
        },
        history: [userMessage],
        metadata: userMessage.metadata,
      };
      eventBus.publish(initialTask);
    }

    // Build message history
    const historyForGenkit = existingTask?.history
      ? [...existingTask.history]
      : [];
    if (!historyForGenkit.find((m) => m.messageId === userMessage.messageId)) {
      historyForGenkit.push(userMessage);
    }

    const messages: MessageData[] = historyForGenkit
      .map((m) => ({
        role: (m.role === "agent" ? "model" : "user") as "user" | "model",
        content: m.parts
          .filter(
            (p): p is TextPart => p.kind === "text" && !!(p as TextPart).text,
          )
          .map((p) => ({
            text: (p as TextPart).text,
          })),
      }))
      .filter((m) => m.content.length > 0);

    if (messages.length === 0) {
      console.warn(
        `[PatientSearchAgentExecutor] No valid text messages found in history for task ${taskId}.`,
      );
      const failureUpdate: TaskStatusUpdateEvent = {
        kind: "status-update",
        taskId: taskId,
        contextId: contextId,
        status: {
          state: "failed",
          message: {
            kind: "message",
            role: "agent",
            messageId: uuidv4(),
            parts: [{ kind: "text", text: "No message found to process." }],
            taskId: taskId,
            contextId: contextId,
          },
          timestamp: new Date().toISOString(),
        },
        final: true,
      };
      eventBus.publish(failureUpdate);
      return;
    }

    // Check if coordination tool is needed
    const messageText = messages.map(m => m.content.map(c => c.text).join('\n')).join('\n');

    // Check if coordination tool should be used
    const coordinationKeywords = ["comprehensive", "coordinate", "integrated", "complete medical plan", "full diagnosis"];
    const needsCoordination = coordinationKeywords.some(keyword => 
        messageText.toLowerCase().includes(keyword)
    );

    if (needsCoordination) {
        try {
            const coordinationResult = await coordinateComprehensiveMedicalPlanUpgraded(messageText);

            const agentMessage: Message = {
                kind: "message",
                role: "agent",
                messageId: uuidv4(),
                parts: [{ kind: "text", text: coordinationResult }],
                taskId: taskId,
                contextId: contextId,
            };

            const finalUpdate: TaskStatusUpdateEvent = {
                kind: "status-update",
                taskId: taskId,
                contextId: contextId,
                status: {
                    state: "completed",
                    message: agentMessage,
                    timestamp: new Date().toISOString(),
                },
                final: true,
            };
            eventBus.publish(finalUpdate);
            
            console.log(`[PatientSearchAgentExecutor] Used coordination tool for task ${taskId}`);
            return;
        } catch (error) {
            console.error("Coordination tool error:", error);
            // Fall through to normal patient search
        }
    }

    // Continue with normal patient search flow
    const workingStatusUpdate: TaskStatusUpdateEvent = {
      kind: "status-update",
      taskId: taskId,
      contextId: contextId,
      status: {
        state: "working",
        message: {
          kind: "message",
          role: "agent",
          messageId: uuidv4(),
          parts: [{ kind: "text", text: "Searching patients..." }],
          taskId: taskId,
          contextId: contextId,
        },
        timestamp: new Date().toISOString(),
      },
      final: false,
    };
    eventBus.publish(workingStatusUpdate);

    try {
      const openaiMessages = messages.map(m => ({
        role: m.role === 'model' ? 'assistant' : m.role,
        content: m.content.map(c => c.text).join('\n')
      }));
      console.log(`[PatientSearchAgentExecutor] Sending messages to OpenAI:`, JSON.stringify(openaiMessages, null, 2));
      const response = await callOpenAI(openaiMessages);

      if (this.cancelledTasks.has(taskId)) {
        console.log(
          `[PatientSearchAgentExecutor] Request cancelled for task: ${taskId}`,
        );

        const cancelledUpdate: TaskStatusUpdateEvent = {
          kind: "status-update",
          taskId: taskId,
          contextId: contextId,
          status: {
            state: "canceled",
            timestamp: new Date().toISOString(),
          },
          final: true,
        };
        eventBus.publish(cancelledUpdate);
        return;
      }

      const responseText = response || "No response generated";
      
      const followupMessage = "\n\n💬 Would you like to search for more patients, or ask about different medical conditions?";
      const enhancedResponse = responseText + followupMessage;
      
      console.info(
        `[PatientSearchAgentExecutor] Prompt response: ${enhancedResponse}`,
      );

      const agentMessage: Message = {
        kind: "message",
        role: "agent",
        messageId: uuidv4(),
        parts: [{ kind: "text", text: enhancedResponse }],
        taskId: taskId,
        contextId: contextId,
      };

      const finalUpdate: TaskStatusUpdateEvent = {
        kind: "status-update",
        taskId: taskId,
        contextId: contextId,
        status: {
          state: "completed",
          message: agentMessage,
          timestamp: new Date().toISOString(),
        },
        final: true,
      };
      eventBus.publish(finalUpdate);

      console.log(
        `[PatientSearchAgentExecutor] Task ${taskId} finished with state: completed`,
      );
    } catch (error: unknown) {
      console.error(
        `[PatientSearchAgentExecutor] Error processing task ${taskId}:`,
        error,
      );
      const errorMessage =
        error instanceof Error ? error.message : "Unknown error occurred";
      const errorUpdate: TaskStatusUpdateEvent = {
        kind: "status-update",
        taskId: taskId,
        contextId: contextId,
        status: {
          state: "failed",
          message: {
            kind: "message",
            role: "agent",
            messageId: uuidv4(),
            parts: [{ kind: "text", text: `Patient search agent error: ${errorMessage}` }],
            taskId: taskId,
            contextId: contextId,
          },
          timestamp: new Date().toISOString(),
        },
        final: true,
      };
      eventBus.publish(errorUpdate);
    }
  }
}

const patientSearchAgentCard: AgentCard = {
  name: "Patient Search Agent (JS)",
  description: "An agent that can search and analyze patient records based on medical criteria.",
  url: "http://localhost:10003/",
  provider: {
    organization: "A2A Medical Samples",
    url: "https://example.com/a2a-medical-samples",
  },
  version: "1.0.0",
  capabilities: {
    streaming: true,
    pushNotifications: false,
    stateTransitionHistory: false,
  },
  securitySchemes: undefined,
  security: undefined,
  defaultInputModes: ["text"],
  defaultOutputModes: ["text"],
  skills: [
    {
      id: "patient_search",
      name: "Search patients",
      description: "Searches and analyzes patient records based on criteria like nationality, age, medical history",
      tags: ["medical", "patients", "search"],
      examples: [
        "What is Emma Johnson's medical history?",    
        "Find patient John Smith",                   
        "What are Maria Garcia's allergies?",  
        "Find patients from American nationality",
        "Search for patients over 50 years old",
        "Find patients with diabetes medical history",
      ],
      inputModes: ["text"],
      outputModes: ["text"],
    },
  ],
  supportsAuthenticatedExtendedCard: false,
};

async function main() {
  const taskStore: TaskStore = new InMemoryTaskStore();
  const agentExecutor: AgentExecutor = new PatientSearchAgentExecutor();
  const requestHandler = new DefaultRequestHandler(
    patientSearchAgentCard,
    taskStore,
    agentExecutor,
  );

  const appBuilder = new A2AExpressApp(requestHandler);
  const expressApp = appBuilder.setupRoutes(express() as any);

  const PORT = process.env.PATIENT_SEARCH_AGENT_PORT || 10003;
  expressApp.listen(PORT, () => {
    console.log(
      `[PatientSearchAgent] Server using new framework started on http://localhost:${PORT}`,
    );
    console.log(
      `[PatientSearchAgent] Agent Card: http://localhost:${PORT}/.well-known/agent-card.json`,
    );
    console.log("[PatientSearchAgent] Press Ctrl+C to stop the server");
  });
}

main().catch(console.error);