// Creating A2A communication tool for Patient Agent

interface A2AMessage {
    kind: string;
    role: string;
    messageId: string;
    contextId?: string;
    taskId?: string;
    referenceTaskIds?: string[];
    parts: Array<{kind: string, text: string}>;
}

interface A2APayload {
    message: A2AMessage;
}

class PatientAgentA2ACommunicator {
    private conversationState: Map<string, any> = new Map();
    
    async initializeAgentConnection(agentUrl: string, agentName: string): Promise<void> {
        // Get Agent Card (simulate Coordinator logic)
        try {
            const cardResponse = await fetch(`${agentUrl}/.well-known/agent-card.json`);
            const agentCard = await cardResponse.json();
            
            // Initialize state
            this.conversationState.set(agentName, {
                url: agentUrl,
                card: agentCard,
                contextId: null,
                currentTaskId: null,
                taskHistory: [],
                lastTaskState: null,
                conversationHistory: []
            });
            
            console.log(`✅ Initialized A2A connection to ${agentName} at ${agentUrl}`);
        } catch (error) {
            console.error(`❌ Failed to initialize A2A connection to ${agentUrl}:`, error);
            throw error;
        }
    }
    
    async sendMessageToAgent(
        agentName: string, 
        messageText: string, 
        continueConversation: boolean = true
    ): Promise<any> {
        const state = this.conversationState.get(agentName);
        if (!state) {
            throw new Error(`Agent ${agentName} not initialized`);
        }
        
        console.log(`📤 Sending A2A message to ${agentName}`);
        
        // 🔄 1. contextId continuity management (same as Coordinator)
        let contextId: string;
        if (continueConversation && state.contextId) {
            contextId = state.contextId;
        } else {
            contextId = this.generateUUID();
            state.contextId = contextId;
            console.log(`🆕 Created new context_id for ${agentName}: ${contextId}`);
        }
        
        // 🔄 2. referenceTaskIds management (same as Coordinator)
        const referenceTaskIds: string[] = [];
        if (state.currentTaskId && state.taskHistory.length > 0 && continueConversation) {
            referenceTaskIds.push(state.currentTaskId);
            console.log(`🔗 Referencing previous task: ${state.currentTaskId}`);
        }
        
        const messageId = this.generateUUID();
        
        // 🔄 3. Build A2A protocol compliant payload (same as Coordinator)
        const payload = {
            jsonrpc: "2.0",
            method: "message/send",
            params: {
                message: {
                    kind: 'message',
                    role: 'user',
                    messageId: messageId,
                    contextId: contextId,
                    parts: [{ kind: 'text', text: messageText }]
                } as A2AMessage
            } as A2APayload,
            id: this.generateUUID()
        };
        
        // 🔄 4. Add taskId for continuing existing task (same as Coordinator)
        if (state.currentTaskId && state.lastTaskState === 'input-required' && continueConversation) {
            payload.params.message.taskId = state.currentTaskId;
            console.log(`🔄 Continuing input-required task: ${state.currentTaskId}`);
        }
        
        // 🔄 5. Add referenceTaskIds (same as Coordinator)
        if (referenceTaskIds.length > 0) {
            payload.params.message.referenceTaskIds = referenceTaskIds;
        }
        
        try {
            const response = await fetch(state.url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const result = await response.json();
            console.log('📨 A2A Response received:', JSON.stringify(result, null, 2));
            
            // 🔄 6. Update task state management (same as Coordinator)
            if (result.result && result.result.id) {
                const newTaskId = result.result.id;
                state.currentTaskId = newTaskId;
                
                // Save to task history
                if (!state.taskHistory.includes(newTaskId)) {
                    state.taskHistory.push(newTaskId);
                    if (state.taskHistory.length > 10) {
                        state.taskHistory = state.taskHistory.slice(-10);
                    }
                }
                
                // Save task state
                if (result.result.status) {
                    state.lastTaskState = result.result.status.state;
                }
                
                console.log(`✅ A2A Task ${newTaskId} state: ${state.lastTaskState}`);
            }
            
            return result;
            
        } catch (error) {
            console.error(`❌ A2A Communication error with ${agentName}:`, error);
            throw error;
        }
    }
    
    private generateUUID(): string {
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
            const r = Math.random() * 16 | 0;
            const v = c == 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
        });
    }
}

// Upgraded coordination function
const patientAgentCommunicator = new PatientAgentA2ACommunicator();

export async function coordinateComprehensiveMedicalPlanUpgraded(medicalRequirements: string): Promise<string> {
    const keywords = ["comprehensive", "coordinate", "integrated", "complete medical plan", "full diagnosis", "doctor"];
    const hasKeyword = keywords.some(keyword => 
        medicalRequirements.toLowerCase().includes(keyword)
    );
    
    if (!hasKeyword) {
        return "This tool is only for comprehensive medical coordination requests.";
    }

    try {
        // 🚀 Initialize A2A connection (first time only)
        if (!patientAgentCommunicator.conversationState.has('Doctor Search Agent')) {
            await patientAgentCommunicator.initializeAgentConnection(
                'http://localhost:10001',
                'Doctor Search Agent'
            );
        }
        
        const coordinationMessage = `
        PATIENT COORDINATION REQUEST: I am developing a comprehensive medical plan for our patient and need your doctor expertise to optimize patient care recommendations.

        Medical Requirements: ${medicalRequirements}

        I need detailed doctor information including:
        1. Recommended specialists based on patient conditions
        2. Doctor availability and expertise matching
        3. Geographic and language compatibility considerations

        Please provide comprehensive doctor analysis so I can recommend the most suitable patient care approach.
        I will wait for your detailed response before proceeding with patient recommendations.
        `;
        
        // 🔄 Use exactly the same logic as Coordinator to send messages
        const result = await patientAgentCommunicator.sendMessageToAgent(
            'Doctor Search Agent',
            coordinationMessage,
            true  // Support conversation continuity
        );
        
        const doctorResponse = extractTextFromResponse(result);
        
        return `
        👨‍⚕️ DOCTOR AGENT RESPONSE (via A2A SDK): ${doctorResponse}
        
        🏥 PATIENT COORDINATION UPDATE: Based on the doctor information received, I am now analyzing optimal patient care plans. The A2A SDK ensures perfect context continuity and state management.
        
        ⚡ A2A FEATURES ENABLED:
        • Context continuity across conversations  
        • Task state management
        • Conversation history tracking
        • Agent Card discovery
        • Robust error handling
        
        💡 RECOMMENDATION: Continue deep coordination for optimal medical solution.
        `;
        
    } catch (error) {
        return `A2A Coordination error occurred: ${error instanceof Error ? error.message : String(error)}`;
    }
}

function extractTextFromResponse(response: any): string {
    // More robust text extraction logic
    try {
        if (response?.result?.artifacts) {
            for (const artifact of response.result.artifacts) {
                if (artifact.parts) {
                    for (const part of artifact.parts) {
                        if (part.kind === 'text' && part.text) {
                            return part.text;
                        }
                    }
                }
            }
        }
        
        if (response?.result?.status?.message?.parts) {
            const texts = [];
            for (const part of response.result.status.message.parts) {
                if (part.kind === 'text' && part.text) {
                    texts.push(part.text);
                }
            }
            if (texts.length > 0) {
                return texts.join(' ');
            }
        }
        
        return "Doctor coordination response received";
    } catch {
        return "Doctor coordination response processed";
    }
}