import * as yaml from 'js-yaml';
import * as fs from 'fs';
import * as path from 'path';
import { dirname } from 'path';
import { fileURLToPath } from 'url';

interface TradingProduct {
  product_id: string;
  symbol: string;
  name: string;
  asset_type: string;
  sector: string;
  current_price: number;
  price_change: number;
  price_change_percent: number;
  currency: string;
  risk_level: string;
  liquidity: string;
  trading_strategies: string[];
  reviews: string[];
}

interface TradingData {
  trading_products: {
    product_list: TradingProduct[];
  };
}

export class TradingSearchTool {
  private products: TradingProduct[] = [];

  constructor() {
    this.loadTradingData();
  }

  private loadTradingData(): void {
    try {
      const __dirname = dirname(fileURLToPath(import.meta.url));
      const yamlPath = path.join(__dirname, 'trading_data.yaml');
      
      if (fs.existsSync(yamlPath)) {
        const fileContents = fs.readFileSync(yamlPath, 'utf8');
        const data = yaml.load(fileContents) as TradingData;
        this.products = data.trading_products.product_list;
        console.log(`[TradingSearchTool] Loaded ${this.products.length} products from YAML`);
      } else {
        console.error(`[TradingSearchTool] Warning: trading_data.yaml not found at ${yamlPath}`);
        this.products = [];
      }
    } catch (error) {
      console.error('[TradingSearchTool] Error loading trading data:', error);
      this.products = [];
    }
  }

  public searchProductsByPrice(maxPrice: number): string {
    const affordableProducts = this.products.filter(product => 
      product.current_price <= maxPrice
    ).sort((a, b) => a.current_price - b.current_price);

    if (affordableProducts.length === 0) {
      return `No trading products found under $${maxPrice}`;
    }

    let result = `Trading products under $${maxPrice}:\n`;
    affordableProducts.forEach(product => {
      result += this.formatProduct(product) + '\n';
    });
    return result;
  }

  public searchProductsByRiskLevel(riskLevel: string): string {
    const riskProducts = this.products.filter(product => 
      product.risk_level.toLowerCase().includes(riskLevel.toLowerCase().trim())
    );

    if (riskProducts.length === 0) {
      return `No trading products found with ${riskLevel} risk level`;
    }

    let result = `${riskLevel} risk level trading products:\n`;
    riskProducts.forEach(product => {
      result += this.formatProduct(product) + '\n';
    });
    return result;
  }
  
  public getProductById(productId: string): string {
    const product = this.products.find(p => p.product_id === productId);
    
    if (!product) {
        return `Trading product ${productId} not found`;
    }
    
    return this.formatProduct(product);
  }

  public searchProductsByName(name: string): string {
    const nameProducts = this.products.filter(product => 
      product.name.toLowerCase().includes(name.toLowerCase().trim()) ||
      product.symbol.toLowerCase().includes(name.toLowerCase().trim())
    );

    if (nameProducts.length === 0) {
      return `No trading products found matching "${name}"`;
    }

    let result = `Trading products matching "${name}":\n`;
    nameProducts.forEach(product => {
      result += this.formatProduct(product) + '\n';
    });
    return result;
  }

  public searchProductsByAssetType(assetType: string): string {
    const typeProducts = this.products.filter(product => 
      product.asset_type.toLowerCase().includes(assetType.toLowerCase().trim())
    );

    if (typeProducts.length === 0) {
      return `No ${assetType} products found`;
    }

    let result = `${assetType} products:\n`;
    typeProducts.forEach(product => {
      result += this.formatProduct(product) + '\n';
    });
    return result;
  }

  public searchProductsBySector(sector: string): string {
    const sectorProducts = this.products.filter(product => 
      product.sector.toLowerCase().includes(sector.toLowerCase().trim())
    );

    if (sectorProducts.length === 0) {
      return `No products found in ${sector} sector`;
    }

    let result = `Products in ${sector} sector:\n`;
    sectorProducts.forEach(product => {
      result += this.formatProduct(product) + '\n';
    });
    return result;
  }

  public getAllProducts(): string {
    if (this.products.length === 0) {
      return 'No trading products available';
    }

    let result = 'All available trading products:\n';
    this.products.forEach(product => {
      result += this.formatProduct(product) + '\n';
    });
    return result;
  }

  private formatProduct(product: TradingProduct): string {
    return `- ${product.symbol} (${product.name}): $${product.current_price}
  Asset Type: ${product.asset_type}, Sector: ${product.sector}
  Risk Level: ${product.risk_level}, Change: ${product.price_change_percent}%
  Trading Strategies: ${product.trading_strategies.join(', ')}`;
  }
}