import { genkit } from "genkit";
import { dirname } from "path";
import { fileURLToPath } from "url";
import { resolve } from "path";
import OpenAI from 'openai';
import * as dotenv from "dotenv";
import { TradingSearchTool } from './TradingSearchTool.js';

// Load .env file
dotenv.config({ path: resolve(process.cwd(), '.env') });

// Check API key
if (!process.env.OPENAI_API_KEY) {
  console.error("OPENAI_API_KEY environment variable is not set.");
  throw new Error("OPENAI_API_KEY environment variable is not set.");
}

// Create OpenAI client, using portable AI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
  baseURL: "https://api.bianxie.ai/v1"
});

// Create trading search tool instance
const tradingTool = new TradingSearchTool();

export const ai = genkit({
  plugins: [],  // Don't use genkit plugins, call OpenAI directly
  promptDir: dirname(fileURLToPath(import.meta.url)),
});

// Custom OpenAI call function with tool support
export async function callOpenAI(messages: any[]) {
  const tools = [
    {
      type: "function" as const,
      function: {
        name: "searchProductsByPrice",
        description: "Search trading products under a maximum price limit. Use this when users ask 'What products under $[PRICE]?' or similar questions.",
        parameters: {
          type: "object",
          properties: {
            maxPrice: {
              type: "number",
              description: "The maximum price in USD (e.g., 100, 500, 1000)"
            }
          },
          required: ["maxPrice"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "getProductById",
        description: "Get specific trading product by product ID (EQ001, EQ002, BOND001, etc.). Use this when users ask about a specific product ID.",
        parameters: {
          type: "object",
          properties: {
            productId: {
              type: "string",
              description: "The product ID (e.g., 'EQ002', 'BOND001', 'CRYPTO001')"
            }
          },
          required: ["productId"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchProductsByRiskLevel",
        description: "Search trading products by risk level. Use this when users ask about risk tolerance or safety.",
        parameters: {
          type: "object",
          properties: {
            riskLevel: {
              type: "string",
              description: "The risk level (e.g., 'Low', 'Medium', 'High', 'Very High')"
            }
          },
          required: ["riskLevel"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchProductsByName",
        description: "Search trading products by name or symbol. Use this for specific product queries.",
        parameters: {
          type: "object",
          properties: {
            name: {
              type: "string",
              description: "The product name or symbol (e.g., 'AAPL', 'Bitcoin', 'SPY')"
            }
          },
          required: ["name"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchProductsByAssetType",
        description: "Search trading products by asset type",
        parameters: {
          type: "object",
          properties: {
            assetType: {
              type: "string",
              description: "The asset type (e.g., 'Equity', 'ETF', 'Bond ETF', 'Cryptocurrency')"
            }
          },
          required: ["assetType"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "searchProductsBySector",
        description: "Search trading products by sector",
        parameters: {
          type: "object",
          properties: {
            sector: {
              type: "string",
              description: "The sector name (e.g., 'Technology', 'Healthcare', 'Energy')"
            }
          },
          required: ["sector"]
        }
      }
    },
    {
      type: "function" as const,
      function: {
        name: "getAllProducts",
        description: "Get all available trading products. Use this for general product overview.",
        parameters: {
          type: "object",
          properties: {},
          required: []
        }
      }
    }
  ];

  const response = await openai.chat.completions.create({
    model: "gemini-2.5-flash",
    messages: messages,
    tools: tools,
    tool_choice: "auto"
  });

  const message = response.choices[0].message;
  console.log(`[TradingSearchTool] OpenAI response:`, JSON.stringify(message, null, 2));
  
  // Handle tool calls
  if (message.tool_calls && message.tool_calls.length > 0) {
    const toolResults = [];
    
    for (const toolCall of message.tool_calls) {
      const functionName = toolCall.function.name;
      const functionArgs = JSON.parse(toolCall.function.arguments);
      console.log(`[TradingSearchTool] Calling function: ${functionName} with args:`, functionArgs);
      let result = "";
      
      switch (functionName) {
        case "searchProductsByPrice":
          result = tradingTool.searchProductsByPrice(functionArgs.maxPrice);
          break;
        case "searchProductsByRiskLevel":
          result = tradingTool.searchProductsByRiskLevel(functionArgs.riskLevel);
          break;
        case "searchProductsByName":
          result = tradingTool.searchProductsByName(functionArgs.name);
          break;
        case "searchProductsByAssetType":
          result = tradingTool.searchProductsByAssetType(functionArgs.assetType);
          break;
        case "searchProductsBySector":
          result = tradingTool.searchProductsBySector(functionArgs.sector);
          break;
        case 'getProductById':
          result = tradingTool.getProductById(functionArgs.productId);
          break;
        case "getAllProducts":
          result = tradingTool.getAllProducts();
          break;
        default:
          result = "Unknown function called";
      }
      console.log(`[TradingSearchTool] Function ${functionName} result:`, result.substring(0, 200) + '...');
      toolResults.push({
        tool_call_id: toolCall.id,
        role: "tool" as const,
        content: result
      });
    }
    
    // Send tool results back to the model
    const finalMessages = [
      ...messages,
      message,
      ...toolResults
    ];
    
    const finalResponse = await openai.chat.completions.create({
      model: "gemini-2.5-flash",
      messages: finalMessages
    });
    console.log(`[TradingSearchTool] Final response:`, finalResponse.choices[0].message.content);
    return finalResponse.choices[0].message.content || "No response generated";
  }
  
  return message.content || "No response generated";
}