import express from "express";
import { v4 as uuidv4 } from "uuid";
import { coordinateComprehensiveFinancialPlanUpgraded } from './trading_agent_a2a_communicator.js';

import {
  AgentCard,
  Task,
  TaskStatusUpdateEvent,
  TextPart,
  Message,
} from "@a2a-js/sdk";
import {
  InMemoryTaskStore,
  TaskStore,
  AgentExecutor,
  RequestContext,
  ExecutionEventBus,
  DefaultRequestHandler,
} from "@a2a-js/sdk/server";
import { A2AExpressApp } from "@a2a-js/sdk/server/express";
import { MessageData } from "genkit";
import { ai, callOpenAI } from "./genkit.js";

import * as dotenv from "dotenv";
import { resolve } from "path";

dotenv.config({ path: resolve(process.cwd(), '.env') });

if (!process.env.OPENAI_API_KEY) {
  console.error("OPENAI_API_KEY environment variable is not set.");
  throw new Error("OPENAI_API_KEY environment variable is not set.");
}

const tradingSearchPrompt = ai.prompt("trading_search");

class TradingSearchAgentExecutor implements AgentExecutor {
  private cancelledTasks = new Set<string>();

  public cancelTask = async (
    taskId: string,
    eventBus: ExecutionEventBus,
  ): Promise<void> => {
    this.cancelledTasks.add(taskId);
  };

  async execute(
    requestContext: RequestContext,
    eventBus: ExecutionEventBus,
  ): Promise<void> {
    const userMessage = requestContext.userMessage;
    const existingTask = requestContext.task;

    const taskId = existingTask?.id || uuidv4();
    const contextId =
      userMessage.contextId || existingTask?.contextId || uuidv4();

    console.log(
      `[TradingSearchAgentExecutor] Processing message ${userMessage.messageId} for task ${taskId} (context: ${contextId})`,
    );

    if (!existingTask) {
      const initialTask: Task = {
        kind: "task",
        id: taskId,
        contextId: contextId,
        status: {
          state: "submitted",
          timestamp: new Date().toISOString(),
        },
        history: [userMessage],
        metadata: userMessage.metadata,
      };
      eventBus.publish(initialTask);
    }

    // Build message history
    const historyForGenkit = existingTask?.history
      ? [...existingTask.history]
      : [];
    if (!historyForGenkit.find((m) => m.messageId === userMessage.messageId)) {
      historyForGenkit.push(userMessage);
    }

    const messages: MessageData[] = historyForGenkit
      .map((m) => ({
        role: (m.role === "agent" ? "model" : "user") as "user" | "model",
        content: m.parts
          .filter(
            (p): p is TextPart => p.kind === "text" && !!(p as TextPart).text,
          )
          .map((p) => ({
            text: (p as TextPart).text,
          })),
      }))
      .filter((m) => m.content.length > 0);

    if (messages.length === 0) {
      console.warn(
        `[TradingSearchAgentExecutor] No valid text messages found in history for task ${taskId}.`,
      );
      const failureUpdate: TaskStatusUpdateEvent = {
        kind: "status-update",
        taskId: taskId,
        contextId: contextId,
        status: {
          state: "failed",
          message: {
            kind: "message",
            role: "agent",
            messageId: uuidv4(),
            parts: [{ kind: "text", text: "No message found to process." }],
            taskId: taskId,
            contextId: contextId,
          },
          timestamp: new Date().toISOString(),
        },
        final: true,
      };
      eventBus.publish(failureUpdate);
      return;
    }

    // Check if coordination tool is needed
    const messageText = messages.map(m => m.content.map(c => c.text).join('\n')).join('\n');

    const coordinationKeywords = ["comprehensive", "coordinate", "integrated", "complete financial plan", "full portfolio"];
    const needsCoordination = coordinationKeywords.some(keyword => 
        messageText.toLowerCase().includes(keyword)
    );

    if (needsCoordination) {
        try {
            const coordinationResult = await coordinateComprehensiveFinancialPlanUpgraded(messageText);

            const agentMessage: Message = {
                kind: "message",
                role: "agent",
                messageId: uuidv4(),
                parts: [{ kind: "text", text: coordinationResult }],
                taskId: taskId,
                contextId: contextId,
            };

            const finalUpdate: TaskStatusUpdateEvent = {
                kind: "status-update",
                taskId: taskId,
                contextId: contextId,
                status: {
                    state: "completed",
                    message: agentMessage,
                    timestamp: new Date().toISOString(),
                },
                final: true,
            };
            eventBus.publish(finalUpdate);
            
            console.log(`[TradingSearchAgentExecutor] Used coordination tool for task ${taskId}`);
            return;
        } catch (error) {
            console.error("Coordination tool error:", error);
            // Fall through to normal trading search
        }
    }

    // Continue with normal trading search flow
    const workingStatusUpdate: TaskStatusUpdateEvent = {
      kind: "status-update",
      taskId: taskId,
      contextId: contextId,
      status: {
        state: "working",
        message: {
          kind: "message",
          role: "agent",
          messageId: uuidv4(),
          parts: [{ kind: "text", text: "Analyzing trading products..." }],
          taskId: taskId,
          contextId: contextId,
        },
        timestamp: new Date().toISOString(),
      },
      final: false,
    };
    eventBus.publish(workingStatusUpdate);

    try {
      const openaiMessages = messages.map(m => ({
        role: m.role === 'model' ? 'assistant' : m.role,
        content: m.content.map(c => c.text).join('\n')
      }));
      console.log(`[TradingSearchAgentExecutor] Sending messages to OpenAI:`, JSON.stringify(openaiMessages, null, 2));
      const response = await callOpenAI(openaiMessages);

      if (this.cancelledTasks.has(taskId)) {
        console.log(
          `[TradingSearchAgentExecutor] Request cancelled for task: ${taskId}`,
        );

        const cancelledUpdate: TaskStatusUpdateEvent = {
          kind: "status-update",
          taskId: taskId,
          contextId: contextId,
          status: {
            state: "canceled",
            timestamp: new Date().toISOString(),
          },
          final: true,
        };
        eventBus.publish(cancelledUpdate);
        return;
      }

      const responseText = response || "No response generated";
      
      const followupMessage = "\n\n💬 Would you like to search for more trading products, or ask about different investment options?";
      const enhancedResponse = responseText + followupMessage;
      
      console.info(
        `[TradingSearchAgentExecutor] Prompt response: ${enhancedResponse}`,
      );

      const agentMessage: Message = {
        kind: "message",
        role: "agent",
        messageId: uuidv4(),
        parts: [{ kind: "text", text: enhancedResponse }],
        taskId: taskId,
        contextId: contextId,
      };

      const finalUpdate: TaskStatusUpdateEvent = {
        kind: "status-update",
        taskId: taskId,
        contextId: contextId,
        status: {
          state: "completed",
          message: agentMessage,
          timestamp: new Date().toISOString(),
        },
        final: true,
      };
      eventBus.publish(finalUpdate);

      console.log(
        `[TradingSearchAgentExecutor] Task ${taskId} finished with state: completed`,
      );
    } catch (error: unknown) {
      console.error(
        `[TradingSearchAgentExecutor] Error processing task ${taskId}:`,
        error,
      );
      const errorMessage =
        error instanceof Error ? error.message : "Unknown error occurred";
      const errorUpdate: TaskStatusUpdateEvent = {
        kind: "status-update",
        taskId: taskId,
        contextId: contextId,
        status: {
          state: "failed",
          message: {
            kind: "message",
            role: "agent",
            messageId: uuidv4(),
            parts: [{ kind: "text", text: `Trading search agent error: ${errorMessage}` }],
            taskId: taskId,
            contextId: contextId,
          },
          timestamp: new Date().toISOString(),
        },
        final: true,
      };
      eventBus.publish(errorUpdate);
    }
  }
}

const tradingSearchAgentCard: AgentCard = {
  name: "Trading Search Agent (JS)",
  description: "An agent that can search and recommend trading products based on user investment criteria.",
  url: "http://localhost:10003/",
  provider: {
    organization: "A2A Samples",
    url: "https://example.com/a2a-samples",
  },
  version: "1.0.0",
  capabilities: {
    streaming: true,
    pushNotifications: false,
    stateTransitionHistory: false,
  },
  securitySchemes: undefined,
  security: undefined,
  defaultInputModes: ["text"],
  defaultOutputModes: ["text"],
  skills: [
    {
      id: "trading_search",
      name: "Search trading products",
      description: "Searches and recommends trading products based on criteria like price, risk level, asset type",
      tags: ["trading", "investments", "finance", "stocks"],
      examples: [
        "Find low-risk trading products under $100",
        "Search for technology sector ETFs",
        "Show me high-yield bond options",
      ],
      inputModes: ["text"],
      outputModes: ["text"],
    },
  ],
  supportsAuthenticatedExtendedCard: false,
};

async function main() {
  const taskStore: TaskStore = new InMemoryTaskStore();
  const agentExecutor: AgentExecutor = new TradingSearchAgentExecutor();
  const requestHandler = new DefaultRequestHandler(
    tradingSearchAgentCard,
    taskStore,
    agentExecutor,
  );

  const appBuilder = new A2AExpressApp(requestHandler);
  const expressApp = appBuilder.setupRoutes(express() as any);

  const PORT = process.env.TRADING_SEARCH_AGENT_PORT || 10003;
  expressApp.listen(PORT, () => {
    console.log(
      `[TradingSearchAgent] Server using new framework started on http://localhost:${PORT}`,
    );
    console.log(
      `[TradingSearchAgent] Agent Card: http://localhost:${PORT}/.well-known/agent-card.json`,
    );
    console.log("[TradingSearchAgent] Press Ctrl+C to stop the server");
  });
}

main().catch(console.error);