"""
A2A Agent-to-Agent Communication Utility for Client Management
Using the same high-level A2A SDK communication method as Coordinator
"""

import asyncio
import uuid
from typing import Dict, Any, Optional
import httpx

from a2a.client import A2AClient, A2ACardResolver
from a2a.types import (
    AgentCard,
    MessageSendParams,
    SendMessageRequest,
    SendMessageResponse,
    SendMessageSuccessResponse,
    Task,
)


class A2AAgentCommunicator:
    """
    High-level A2A Agent communication tool for Client Management, using the same logic as Coordinator
    """
    
    def __init__(self):
        self.agent_connections: Dict[str, 'AgentConnection'] = {}
        self.conversation_state: Dict[str, Dict[str, Any]] = {}
        
    async def initialize_agent_connection(self, agent_url: str, agent_name: Optional[str] = None) -> str:
        """
        Initialize connection with remote Agent (same logic as Coordinator)
        
        Args:
            agent_url: Agent URL
            agent_name: Optional Agent name, if not provided it will be retrieved from Agent Card
            
        Returns:
            Agent name
        """
        try:
            async with httpx.AsyncClient(timeout=60) as client:
                # 1. Get Agent Card (same as Coordinator)
                card_resolver = A2ACardResolver(client, agent_url)
                card = await card_resolver.get_agent_card()
                
                # 2. Create connection (same as Coordinator)
                connection = AgentConnection(agent_card=card, agent_url=agent_url)
                
                # 3. Store connection
                actual_agent_name = agent_name or card.name
                self.agent_connections[actual_agent_name] = connection
                
                # 4. Initialize session state
                if actual_agent_name not in self.conversation_state:
                    self.conversation_state[actual_agent_name] = {
                        'context_id': None,
                        'current_task_id': None,
                        'task_history': [],
                        'last_task_state': None,
                        'conversation_history': []
                    }
                
                print(f'✅ Initialized A2A connection to {actual_agent_name} at {agent_url}')
                return actual_agent_name
                
        except Exception as e:
            print(f'❌ Failed to initialize A2A connection to {agent_url}: {e}')
            raise
    
    async def send_message_to_agent(
        self, 
        agent_name: str, 
        message_text: str,
        continue_conversation: bool = True
    ) -> Optional[Task]:
        """
        Send message to specified Agent (fully using Coordinator logic)
        
        Args:
            agent_name: Target Agent name
            message_text: Message content to send
            continue_conversation: Whether to continue existing conversation
            
        Returns:
            Task object or None
        """
        if agent_name not in self.agent_connections:
            raise ValueError(f'Agent {agent_name} not initialized. Call initialize_agent_connection first.')
        
        connection = self.agent_connections[agent_name]
        state = self.conversation_state[agent_name]
        
        print(f'📤 Sending A2A message to {agent_name}')
        
        # 🔄 1. contextId continuity management (same as Coordinator)
        if continue_conversation and state['context_id']:
            context_id = state['context_id']
        else:
            context_id = str(uuid.uuid4())
            state['context_id'] = context_id
            print(f'🆕 Created new context_id for {agent_name}: {context_id}')
        
        # 🔄 2. taskId continuity management (same as Coordinator)
        current_task_id = state['current_task_id']
        
        # 🔄 3. referenceTaskIds management (same as Coordinator)
        reference_task_ids = []
        if current_task_id and len(state['task_history']) > 0 and continue_conversation:
            reference_task_ids = [current_task_id]
            print(f'🔗 Referencing previous task: {current_task_id}')
        
        # Generate message ID
        message_id = str(uuid.uuid4())
        
        # 🔄 4. Build A2A protocol compliant payload (same as Coordinator)
        payload = {
            'message': {
                'role': 'user',
                'parts': [
                    {'kind': 'text', 'text': message_text}
                ],
                'messageId': message_id,
                'contextId': context_id,
            },
        }
        
        # 🔄 5. Add taskId for continuing existing task (same as Coordinator)
        last_task_state = state['last_task_state']
        
        if current_task_id and last_task_state == 'input-required' and continue_conversation:
            payload['message']['taskId'] = current_task_id
            print(f'🔄 Continuing input-required task: {current_task_id}')
        elif current_task_id and last_task_state == 'completed':
            print(f'🆕 Creating new task (previous completed): {current_task_id}')
        
        # 🔄 6. Add referenceTaskIds (same as Coordinator)
        if reference_task_ids:
            payload['message']['referenceTaskIds'] = reference_task_ids
        
        try:
            # Send request (same as Coordinator)
            message_request = SendMessageRequest(
                id=message_id, params=MessageSendParams.model_validate(payload)
            )
            
            send_response: SendMessageResponse = await connection.send_message(message_request)
            
            print('📨 A2A Response received:', send_response.model_dump_json(exclude_none=True, indent=2))
            
            # Validate response (same as Coordinator)
            if not isinstance(send_response.root, SendMessageSuccessResponse):
                print('❌ Received non-success response')
                return None
            
            if not isinstance(send_response.root.result, Task):
                print('❌ Received non-task response')
                return None
            
            task_result = send_response.root.result
            
            # 🔄 7. Update task state management (same as Coordinator)
            new_task_id = task_result.id
            state['current_task_id'] = new_task_id
            
            # Save to task history
            if new_task_id not in state['task_history']:
                state['task_history'].append(new_task_id)
                if len(state['task_history']) > 10:
                    state['task_history'] = state['task_history'][-10:]
            
            # 🔄 8. Save task state for future reference (same as Coordinator)
            state['last_task_state'] = task_result.status.state
            
            # 🔄 9. Save conversation history (same as Coordinator)
            user_message = {
                'role': 'user',
                'text': message_text,
                'messageId': message_id,
                'taskId': new_task_id,
                'contextId': context_id,
                'timestamp': task_result.status.timestamp or str(uuid.uuid4())
            }
            state['conversation_history'].append(user_message)
            
            # Add agent response to history
            if task_result.status.message:
                agent_message = {
                    'role': 'agent',
                    'text': self._extract_message_text(task_result.status.message.parts) if task_result.status.message.parts else '',
                    'messageId': getattr(task_result.status.message, 'messageId', str(uuid.uuid4())),
                    'taskId': new_task_id,
                    'contextId': context_id,
                    'timestamp': task_result.status.timestamp,
                    'state': task_result.status.state
                }
                state['conversation_history'].append(agent_message)
            
            # Keep only recent history
            if len(state['conversation_history']) > 20:
                state['conversation_history'] = state['conversation_history'][-20:]
            
            print(f'✅ A2A Task {new_task_id} state: {task_result.status.state}')
            return task_result
            
        except Exception as e:
            print(f'❌ A2A Communication error with {agent_name}: {e}')
            raise
    
    def _extract_message_text(self, parts) -> str:
        """Extract message text (same logic as Coordinator)"""
        if not parts:
            return ''
        
        texts = []
        for part in parts:
            if hasattr(part, 'kind') and getattr(part, 'kind', None) == 'text':
                texts.append(getattr(part, 'text', ''))
            elif hasattr(part, 'type') and getattr(part, 'type', None) == 'text':
                texts.append(getattr(part, 'text', ''))
        
        return ' '.join(texts)
    
    def get_conversation_history(self, agent_name: str) -> list:
        """Get conversation history with specified Agent"""
        if agent_name in self.conversation_state:
            return self.conversation_state[agent_name]['conversation_history']
        return []


class AgentConnection:
    """Agent Connection class (same as RemoteAgentConnections)"""
    
    def __init__(self, agent_card: AgentCard, agent_url: str):
        self._httpx_client = httpx.AsyncClient(timeout=30)
        self.agent_client = A2AClient(self._httpx_client, agent_card, url=agent_url)
        self.card = agent_card
    
    async def send_message(self, message_request: SendMessageRequest) -> SendMessageResponse:
        return await self.agent_client.send_message(message_request)
    
    def get_agent_card(self) -> AgentCard:
        return self.card