import json
import logging

import click
import uvicorn

from a2a.server.apps import A2AStarletteApplication
from a2a.server.request_handlers import DefaultRequestHandler
from a2a.server.tasks import InMemoryTaskStore
from a2a.types import (
    AgentCapabilities,
    AgentCard,
    AgentSkill,
)
from doctor_search_agent import root_agent as doctor_search_agent  # Changed import
from agent_executor import ADKAgentExecutor
from dotenv import load_dotenv

load_dotenv()

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class MissingAPIKeyError(Exception):
    """Exception for missing API key."""

@click.command()
@click.option("--host", default="localhost")
@click.option("--port", default=10001)
@click.option("--enable-followup/--disable-followup", default=True, help="Enable multi-turn conversations")
@click.option("--json-debug/--no-json-debug", default=False, help="Show A2A JSON communication")
def main(host, port, enable_followup, json_debug):

    # If JSON debugging is enabled, only show A2A related logs, suppress LiteLLM
    if json_debug:
        # Suppress detailed LiteLLM logs
        logging.getLogger('LiteLLM').setLevel(logging.WARNING)
        logging.getLogger('openai').setLevel(logging.WARNING)
        logging.getLogger('litellm').setLevel(logging.WARNING)
        print("🔍 A2A JSON Debug: Enabled")

    agent_card = AgentCard(
        name='Doctor Search Agent',  # Changed name
        description=doctor_search_agent.description + " Supports contextual multi-turn conversations.",  # Updated reference
        url=f'http://{host}:{port}',
        version="1.0.0",
        protocolVersion="0.3.0",
        preferredTransport="JSONRPC",
        defaultInputModes=["text", "text/plain"],
        defaultOutputModes=["text", "text/plain"],
        capabilities=AgentCapabilities(
            streaming=True,
            pushNotifications=False,
            stateTransitionHistory=True,
        ),
        skills=[
            AgentSkill(
                id="doctor_search",  # Changed skill ID
                name="Doctor Information Search",  # Changed skill name
                description="Search and retrieve doctor information from local database including specialties, nationalities, hospitals and experience. Supports contextual conversations and follow-up queries.",  # Updated description
                tags=["search", "doctors", "medical", "healthcare", "contextual"],  # Updated tags
                examples=[  # Updated examples
                    "Find doctors with Cardiology specialty",
                    "Search for doctors from American nationality",
                    "Get doctors working at Mayo Clinic",
                    "Show me doctors with over 15 years experience",
                    "Tell me more about the first doctor",
                    "What about pediatric specialists?",
                ],
            )
        ],
    )

    request_handler = DefaultRequestHandler(
        agent_executor=ADKAgentExecutor(
            agent=doctor_search_agent,  # Updated reference
            status_message="Processing your doctor search request...",  # Updated message
            artifact_name="doctor_search_results",  # Updated artifact name
            enable_followup=enable_followup,
            debug_json=json_debug,  # Pass debug parameter
        ),
        task_store=InMemoryTaskStore(),
    )

    server = A2AStarletteApplication(
        agent_card=agent_card, http_handler=request_handler
    )

    print(f"🚀 Doctor Search Agent starting on {host}:{port}")  # Updated print message
    if enable_followup:
        print(f"✅ Multi-turn conversations: Enabled")
        print(f"✅ Context ID support: Enabled") 
        print(f"✅ Reference Task IDs support: Enabled")
        print(f"✅ Input-required state support: Enabled")
    else:
        print(f"ℹ️  Single-turn mode: Enabled")
    
    uvicorn.run(server.build(), host=host, port=port)

if __name__ == "__main__":
    main()