import json
import logging

import click
import uvicorn

from a2a.server.apps import A2AStarletteApplication
from a2a.server.request_handlers import DefaultRequestHandler
from a2a.server.tasks import InMemoryTaskStore
from a2a.types import (
    AgentCapabilities,
    AgentCard,
    AgentSkill,
)
from hotel_search_agent import root_agent as hotel_search_agent
from agent_executor import ADKAgentExecutor
from dotenv import load_dotenv

load_dotenv()

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class MissingAPIKeyError(Exception):
    """Exception for missing API key."""

@click.command()
@click.option("--host", default="localhost")
@click.option("--port", default=10001)
@click.option("--enable-followup/--disable-followup", default=True, help="Enable multi-turn conversations")
@click.option("--json-debug/--no-json-debug", default=False, help="Show A2A JSON communication")
def main(host, port, enable_followup, json_debug):

    # If JSON debugging is enabled, only show A2A related logs, suppress LiteLLM
    if json_debug:
        # Suppress detailed LiteLLM logs
        logging.getLogger('LiteLLM').setLevel(logging.WARNING)
        logging.getLogger('openai').setLevel(logging.WARNING)
        logging.getLogger('litellm').setLevel(logging.WARNING)
        print("🔍 A2A JSON Debug: Enabled")

    agent_card = AgentCard(
        name='Hotel Search Agent',
        description=hotel_search_agent.description + " Supports contextual multi-turn conversations.",
        url=f'http://{host}:{port}',
        version="1.0.0",
        protocolVersion="0.3.0",
        preferredTransport="JSONRPC",
        defaultInputModes=["text", "text/plain"],
        defaultOutputModes=["text", "text/plain"],
        capabilities=AgentCapabilities(
            streaming=True,
            pushNotifications=False,
            stateTransitionHistory=True,
        ),
        skills=[
            AgentSkill(
                id="hotel_search",
                name="Hotel Information Search",
                description="Search and retrieve hotel information from local database including ratings, prices, locations and amenities. Supports contextual conversations and follow-up queries.",
                tags=["search", "hotels", "travel", "accommodation", "contextual"],
                examples=[
                    "Find hotels in Paris with rating above 4.0",
                    "Search for hotels under $200 per night",
                    "Get details for City Hub hotel",
                    "Show me luxury hotels in Tokyo",
                    "Tell me more about the first hotel",
                    "What about budget options?",
                ],
            )
        ],
    )

    request_handler = DefaultRequestHandler(
        agent_executor=ADKAgentExecutor(
            agent=hotel_search_agent,
            status_message="Processing your hotel search request...",
            artifact_name="hotel_search_results",
            enable_followup=enable_followup,
            debug_json=json_debug,  # Pass debug parameter
        ),
        task_store=InMemoryTaskStore(),
    )

    server = A2AStarletteApplication(
        agent_card=agent_card, http_handler=request_handler
    )

    print(f"🚀 Hotel Search Agent starting on {host}:{port}")
    if enable_followup:
        print(f"✅ Multi-turn conversations: Enabled")
        print(f"✅ Context ID support: Enabled") 
        print(f"✅ Reference Task IDs support: Enabled")
        print(f"✅ Input-required state support: Enabled")
    else:
        print(f"ℹ️  Single-turn mode: Enabled")
    
    uvicorn.run(server.build(), host=host, port=port)

if __name__ == "__main__":
    main()