"""client_utils.py
A2A client utilities for testing.
Based on number_guessing_game/utils/protocol_wrappers.py
"""

from __future__ import annotations

import asyncio
import uuid

from a2a.client import ClientConfig, ClientFactory, minimal_agent_card
from a2a.client.client_task_manager import ClientTaskManager
from a2a.types import Message, Role, Task, TaskIdParams, TextPart
from a2a.utils.message import get_message_text


_client_factory = ClientFactory(ClientConfig())


async def send_text_async(
    port: int,
    text: str,
    *,
    context_id: str | None = None,
    reference_task_ids: list[str] | None = None,
    timeout: float = 60.0,
):
    """Send text to the agent via A2A message/send operation."""
    client = _client_factory.create(
        minimal_agent_card(f'http://localhost:{port}')
    )
    msg = Message(
        kind='message',
        role=Role.user,
        message_id=uuid.uuid4().hex,
        context_id=context_id,
        reference_task_ids=reference_task_ids or [],
        parts=[TextPart(text=text)],
    )

    task_manager = ClientTaskManager()
    last_message: Message | None = None
    event_count = 0

    print(f"🔍 [CLIENT] Starting to send message to port {port} (timeout: {timeout}s)")
    
    try:
        # ✅ Add timeout control
        async with asyncio.timeout(timeout):
            async for event in client.send_message(msg):  # type: ignore[attr-defined]
                event_count += 1
                print(f"🔍 [CLIENT] Received event {event_count}: {type(event).__name__}")
                
                if isinstance(event, tuple):
                    event = event[0]
                await task_manager.process(event)
                if isinstance(event, Message):
                    last_message = event
                    print(f"🔍 [CLIENT] Received message: {get_message_text(event)[:50]}...")

        print(f"🔍 [CLIENT] Event loop ended, processed {event_count} events total")
        
    except asyncio.TimeoutError:
        print(f"⏰ [CLIENT] Request timeout ({timeout}s)")
        raise
    except Exception as e:
        print(f"❌ [CLIENT] Async loop exception: {e}")
        raise

    task = task_manager.get_task()
    if task:
        print(f"✅ [CLIENT] Got task, status: {task.status.state if hasattr(task, 'status') else 'unknown'}")
        return task
    if last_message is not None:
        print(f"✅ [CLIENT] Returning last message")
        return last_message
    
    print(f"❌ [CLIENT] No response received")
    raise RuntimeError('No response from agent')


def send_text(port: int, text: str, **kwargs):
    """Synchronous helper for send_text_async."""
    try:
        return asyncio.run(send_text_async(port, text, **kwargs))
    except RuntimeError:
        loop = asyncio.get_event_loop()
        if loop.is_running():
            return loop.run_until_complete(send_text_async(port, text, **kwargs))
        raise


def send_followup(port: int, resp_task: Task, text: str):
    """Send a follow-up message that maintains context and task reference."""
    return send_text(
        port,
        text,
        context_id=resp_task.context_id,
        reference_task_ids=[resp_task.id],
    )


def extract_text(obj: Task | Message):
    """Extract plain text from a Task or Message."""
    if isinstance(obj, Message):
        return get_message_text(obj)

    if isinstance(obj, Task) and obj.artifacts:
        for artifact in reversed(obj.artifacts):
            if artifact.parts:
                for part in reversed(artifact.parts):
                    if hasattr(part, 'root') and hasattr(part.root, 'text'):
                        return part.root.text  # type: ignore[attr-defined]
    return ''