import ray
from .base import ToolEngine, _get_free_port
import asyncio
from contextlib import asynccontextmanager
import fastapi
import os
import uvicorn
from sandbox.database import get_databases
from sandbox.runners import CODE_RUNNERS, CodeRunArgs
import re


@ray.remote(num_cpus=1)
class CodeEngine(ToolEngine):
    def __init__(self, config):
        super().__init__(
            f"/tool/code",
            batch_size=config.execution_batch_size,
            max_wait_time=config.max_wait_time,
        )
        self.config = config
        self.semaphore = asyncio.Semaphore(32)

    def parse_request(self, req_json):
        if "language" not in req_json:
            req_json["language"] = "python"
        return req_json

    def format_response(self, result):
        """
        Optional: Subclasses can override to customize response format.
        """
        return {"code_result": result}

    async def run_code(self, request):
        try:
            language = request.pop("language")
            result = await CODE_RUNNERS[language](CodeRunArgs(**request))

            if result.run_result.return_code == 0 or len(result.run_result.stdout):
                return result.run_result.stdout
            else:
                error_str = ""
                if result.compile_result is not None:
                    if result.compile_result.return_code == 0:
                        error_str += "Compile result: Successful!"
                    else:
                        error_str += f"Compile result: {result.compile_result.stderr}"

                error_str += f"\nRun result: {result.run_result.stderr}"

                pattern = r'File "/tmp/[^"]+\.py",'
                cleaned_error_str = re.sub(pattern, "", error_str)
                return cleaned_error_str
        except Exception as e:
            return str(e)

    async def process_batch(self, batch_inputs):
        async def run_single(input_obj):
            async with self.semaphore:
                # 你的代码执行逻辑
                return await self.run_code(input_obj)
        tasks = [run_single(x) for x in batch_inputs]
        return await asyncio.gather(*tasks)
        # tasks = [self.run_code(req) for req in batch_inputs]
        # results = await asyncio.gather(*tasks)

        # return results


if __name__ == "__main__":
    ray.init()

    class DummyConfig:
        execution_batch_size = 2
        max_wait_time = 0.1

    # 实例化并启动CodeEngine
    engine = CodeEngine.options(num_cpus=2).remote(DummyConfig())
    # 等待FastAPI服务ready

    batch_inputs = [
        {"code": "print('hello world')", "language": "python"},
        {"code": "print('hello again')", "language": "python"},
    ]
    results = ray.get(engine.process_batch.remote(batch_inputs))
    print("Batch results:", results)
