from dataclasses import dataclass
from pathlib import Path

import pandas as pd


@dataclass
class Result:
    folder: Path
    df: pd.DataFrame
    sort_by: str

    @staticmethod
    def from_folder(folder: Path | str, sort_by: str = "entropies") -> "Result":
        if isinstance(folder, str):
            folder = Path(folder)
        dfs = []
        nnc_file = folder / "nncodec-results.csv"
        if nnc_file.exists():
            df = pd.read_csv(nnc_file)
            df = df.assign(bits=0.0, method="NNCodec")
            dfs.append(df)

        for subfolder in folder.iterdir():
            if subfolder.is_dir():
                b = float(subfolder.name)

                # Define file names and their corresponding variable names
                file_mappings = {
                    "gptq.csv": "GPTQ",
                    "alpha_inv.csv": "GPTQ-RD 1/a",
                    "uniform.csv": "GPTQ-RD Unif.",
                    "rtn.csv": "RTN",
                    "direct_rd.csv": "Direct RD",
                    "alpha_inv_tr.csv": "GPTQ-RD Trace",
                    "alpha_inv_tr_scale.csv": "GPTQ-RD Trace Scaled",
                    "alpha_inv_tr_scale_sq.csv": "GPTQ-RD Trace^2",
                    "alpha_inv_std.csv": "GPTQ-RD std",
                    "alpha_inv_tr_unscaled.csv": "GPTQ-RD Trace Unscaled",
                    "alpha_inv_tr_rescaled.csv": "GPTQ-RD Trace Rescaled",
                }
                dataframes = {}
                for file_name, method in file_mappings.items():
                    file_path = subfolder / file_name
                    dataframes[method] = (
                        pd.read_csv(file_path) if file_path.exists() else None
                    )
                for method, df in dataframes.items():
                    if df is None:
                        continue
                    df = df.assign(bits=b, method=method)
                    dfs.append(df)
        pd.concat(dfs)
        return Result(folder=folder, df=pd.concat(dfs), sort_by=sort_by)

    def _retrieve(self, method, bits=None):
        method_df = self.df[self.df.method == method]
        if bits is None:
            bits_df = method_df
        else:
            bits_df = method_df[method_df.bits == bits]
        return bits_df.sort_values(self.sort_by)

    def gptq(self, bits=None):
        return self._retrieve("GPTQ", bits)

    def rd_unif(self, bits=None):
        return self._retrieve("GPTQ-RD Unif.", bits)

    def rd_alpha_inv(self, bits=None):
        return self._retrieve("GPTQ-RD 1/a", bits)

    def rtn(self, bits=None):
        return self._retrieve("RTN", bits)

    def rd_direct(self, bits=None):
        return self._retrieve("Direct RD", bits)

    def rd_tr(self, bits=None):
        return self._retrieve("GPTQ-RD Trace", bits)

    def rd_tr_scale(self, bits=None):
        return self._retrieve("GPTQ-RD Trace Scaled", bits)

    def rd_std(self, bits=None):
        return self._retrieve("GPTQ-RD std", bits)

    def rd_tr_sq(self, bits=None):
        return self._retrieve("GPTQ-RD Trace^2", bits)

    def rd_tr_unscaled(self, bits=None):
        return self._retrieve("GPTQ-RD Trace Unscaled", bits)

    def rd_tr_rescaled(self, bits=None):
        return self._retrieve("GPTQ-RD Trace Rescaled", bits)
