# Copyright (c) Facebook, Inc. and its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import os
import argparse
import json
from pathlib import Path

import torch
from torch import nn
import torch.distributed as dist
import torch.backends.cudnn as cudnn
from torchvision import datasets
from torchvision import transforms as pth_transforms
from torchvision import models as torchvision_models

import dino_vit.utils as utils
import dino_vit.vision_transformer as vits
from cifar10_embeddings.cifar10_dataset import CIFAR10EmbeddingEval


def eval_linear(args):
    # utils.init_distributed_mode(args)
    # print("git:\n  {}\n".format(utils.get_sha()))
    print("\n".join("%s: %s" % (k, str(v)) for k, v in sorted(dict(vars(args)).items())))
    cudnn.benchmark = True

    # ============ building network ... ============
    # if the network is a Vision Transformer (i.e. vit_tiny, vit_small, vit_base)
    model = None
    if not args.API_embedding:
        model = vits.__dict__[args.arch](patch_size=args.patch_size, num_classes=0)
        # optionally load from a checkpoint
        checkpoint_file = None
        if args.load_victim:
            checkpoint_file = f"/checkpoint/jiangm/checkpoints/{args.victim}_checkpoint.pth"
        elif args.load_file:
            if os.path.isfile(args.load_file):
                checkpoint_file = args.load_file
            else:
                print("=> no checkpoint found at '{}'".format(args.load_file))
        elif args.simple_load:
            print("Simple load: ")
            checkpoint_file = f"logs/{args.victim}/{args.experiment_folder}/checkpoint_{args.datasetsteal}_{args.losstype}_{args.num_queries}_lr0.001_bs64_n4.pth.tar"
            print(checkpoint_file)
        else:
            raise ValueError

        try:
            try:
                checkpoint = torch.load(checkpoint_file)
                model.load_state_dict(checkpoint)
            except:
                state_dict = torch.load(checkpoint_file)['state_dict']
                for k in list(state_dict.keys()):
                    # retain only encoder up to before the embedding layer
                    if k.startswith('module.'):
                        # remove prefix
                        state_dict[k[len("module."):]] = state_dict[k]
                    # delete renamed or unused k
                    del state_dict[k]
                model.load_state_dict(state_dict)
            print(f"Load backbone model from {checkpoint_file}")
        except:
            whole_model = utils.MultiCropWrapper(
                model,
                vits.DINOHead(model.embed_dim, args.out_dim, False, nlayers=args.head_layer),
            )
            model = utils.load_pretrained_teacher(whole_model, ckp_path=checkpoint_file)
            print(f"Load teacher model from the whole checkpoint {checkpoint_file}")

        embed_dim = model.embed_dim * (args.n_last_blocks + int(args.avgpool_patchtokens))

        model.cuda()
        model.eval()
        # load weights to evaluate
        # utils.load_pretrained_weights(model, args.pretrained_weights, args.checkpoint_key, args.arch, args.patch_size)
        print(f"Model {args.arch} built.")

    # ============ preparing data ... ============
    # Data loading code
    if args.API_embedding == 'cifar10':
        transform = pth_transforms.Compose([
            pth_transforms.ToTensor(), ])
        train_dataset = CIFAR10EmbeddingEval(root='/checkpoint/jiangm/data/cifar10-embeddings/', train=True, transform=transform)
        val_dataset = CIFAR10EmbeddingEval(root='/checkpoint/jiangm/data/cifar10-embeddings/', train=False, transform=transform)
        num_labels = 10
        embed_dim = 768

    elif args.downstreamdata == 'imagenet':
        normalize = pth_transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                         std=[0.229, 0.224, 0.225])
        train_dataset = datasets.ImageNet(
            root="/scratch/ssd002/datasets/imagenet256/",
            split = "train",
            transform=pth_transforms.Compose([
                pth_transforms.RandomResizedCrop(224),
                pth_transforms.RandomHorizontalFlip(),
                pth_transforms.ToTensor(),
                normalize,
            ]))

        val_dataset = datasets.ImageNet(
            root = "/scratch/ssd002/datasets/imagenet256/",
            split = "val",
            transform=pth_transforms.Compose([pth_transforms.Resize(256),
                                              pth_transforms.CenterCrop(224),
                                              pth_transforms.ToTensor(),
                                          normalize,
                                          ]))
        num_labels = 1000

    elif args.downstreamdata == 'cifar10':
        transform_train = pth_transforms.Compose([
            pth_transforms.Resize(224),
            # transforms.RandomCrop(32, padding=4),
            pth_transforms.RandomHorizontalFlip(),
            pth_transforms.ToTensor(),
            pth_transforms.Normalize((0.4914, 0.4822, 0.4465),
                                 (0.2023, 0.1994, 0.2010)),
        ])

        transform_test = pth_transforms.Compose([
            pth_transforms.Resize(224),
            pth_transforms.ToTensor(),
            pth_transforms.Normalize((0.4914, 0.4822, 0.4465),
                                 (0.2023, 0.1994, 0.2010)),
        ])

        train_dataset = datasets.CIFAR10(
            root=f"/scratch/ssd002/datasets/cifar10/", train=True,
            download=True, transform=transform_train)
        val_dataset = datasets.CIFAR10(
            root=f"/scratch/ssd002/datasets/cifar10/", train=False,
            download=True, transform=transform_test)
        num_labels = 10

    elif args.downstreamdata == 'cifar100':
        transform_train = pth_transforms.Compose([
            pth_transforms.Resize(224),
            # transforms.RandomCrop(32, padding=4),
            pth_transforms.RandomHorizontalFlip(),
            pth_transforms.ToTensor(),
            pth_transforms.Normalize((0.4914, 0.4822, 0.4465),
                                 (0.2023, 0.1994, 0.2010)),
        ])

        transform_test = pth_transforms.Compose([
            pth_transforms.Resize(224),
            pth_transforms.ToTensor(),
            pth_transforms.Normalize((0.4914, 0.4822, 0.4465),
                                 (0.2023, 0.1994, 0.2010)),
        ])

        train_dataset = datasets.CIFAR100(
            root=f"/scratch/ssd002/datasets/cifar100/", train=True,
            download=True, transform=transform_train)

        val_dataset = datasets.CIFAR100(
            root=f"/scratch/ssd002/datasets/cifar100/", train=False,
            download=True, transform=transform_test)

        num_labels = 100

    elif args.downstreamdata == 'svhn':
        transform_svhn = pth_transforms.Compose([
            pth_transforms.Resize(224),
            pth_transforms.ToTensor(),
        ])

        train_dataset = datasets.SVHN(
            root="/scratch/ssd002/datasets/svhn/", split='train',
            download=False, transform=transform_svhn)
        val_dataset = datasets.SVHN(
            root="/scratch/ssd002/datasets/svhn/", split='test', download=False,
            transform=transform_svhn)

        num_labels = 10

    elif args.downstreamdata == "stl10":
        transform_train = pth_transforms.Compose([
            pth_transforms.Resize(224),
            # transforms.RandomCrop(32, padding=4),
            pth_transforms.RandomHorizontalFlip(),
            pth_transforms.ToTensor(),
            pth_transforms.Normalize((0.4914, 0.4822, 0.4465),
                                 (0.2023, 0.1994, 0.2010)),
        ])

        transform_test = pth_transforms.Compose([
            pth_transforms.Resize(224),
            pth_transforms.ToTensor(),
        ])
        train_dataset = datasets.STL10(
            '/checkpoint/jiangm/data', split='train',
            download=True,transform=transform_train)
        val_dataset = datasets.STL10('/checkpoint/jiangm/data', split='test', download=True,
                                      transform=transform_test)

        num_labels = 10

    else:
        raise Exception(f"Unknown args.datasetsteal: {args.datasetsteal}.")

    train_sampler = torch.utils.data.distributed.DistributedSampler(
            train_dataset) if args.distributed else None
    train_loader = torch.utils.data.DataLoader(
        train_dataset,
        sampler=train_sampler,
        batch_size=args.batch_size_per_gpu,
        num_workers=args.num_workers,
        pin_memory=True,
    )
    val_loader = torch.utils.data.DataLoader(val_dataset, batch_size=args.batch_size_per_gpu,
                                             num_workers=args.workers, drop_last=False, shuffle=False, pin_memory=True,)
    print(f"Data loaded with {len(train_dataset)} train and {len(val_dataset)} val imgs.")

    linear_classifier = LinearClassifier(embed_dim, num_labels=num_labels)
    linear_classifier = linear_classifier.cuda()
    # linear_classifier = nn.parallel.DistributedDataParallel(linear_classifier, device_ids=[args.gpu])
    linear_classifier = nn.DataParallel(linear_classifier)
    print(f"Linear classifier dim is ({embed_dim}, {num_labels})")

    # if args.evaluate:
    #     utils.load_pretrained_linear_weights(linear_classifier, args.arch, args.patch_size)
    #     test_stats = validate_network(val_loader, model, linear_classifier, args.n_last_blocks, args.avgpool_patchtokens)
    #     print(f"Accuracy of the network on the {len(test_dataset)} test images: {test_stats['acc1']:.1f}%")
    #     return

    # optimizer = torch.optim.SGD(
    #     linear_classifier.parameters(),
    #     args.lr * (args.batch_size_per_gpu * utils.get_world_size()) / 256., # linear scaling rule
    #     momentum=0.9,
    #     weight_decay=0, # we do not apply weight decay
    # )
    optimizer = torch.optim.AdamW(
        linear_classifier.parameters(),
        args.lr * (args.batch_size_per_gpu * utils.get_world_size()) / 256., # linear scaling rule
        weight_decay=0, # we do not apply weight decay
    )
    scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(optimizer, args.epochs, eta_min=1e-4)

    # Optionally resume from a checkpoint
    real_output_dir = os.path.join(args.output_dir, f"{args.n_last_blocks}blocks", args.victim, args.experiment_folder)
    if not os.path.exists(real_output_dir):
        os.makedirs(real_output_dir)
    save_prefix = None
    softmax_str = ''
    if args.use_softmax:
        softmax_str = '_softmax'
    if args.API_embedding:
        save_prefix = f'API_{args.API_embedding}_n{args.n_last_blocks}_ds{args.downstreamdata}{softmax_str}'
    elif args.load_victim:
        save_prefix = f'victim_{args.victim}_n{args.n_last_blocks}_ds{args.downstreamdata}{softmax_str}'
    elif args.load_file:
        save_prefix = f'{args.load_file}_ds{args.downstreamdata}{softmax_str}'
    elif args.simple_load:
        save_prefix = f"lineval_{args.arch}_{os.path.basename(checkpoint_file)}_ds{args.downstreamdata}{softmax_str}"

    to_restore = {"epoch": 0, "best_acc": 0.}
    # utils.restart_from_checkpoint(
    #     os.path.join(real_output_dir, f"{save_prefix}_checkpoint.pth.tar"),
    #     run_variables=to_restore,
    #     state_dict=linear_classifier,
    #     optimizer=optimizer,
    #     scheduler=scheduler,
    # )
    start_epoch = to_restore["epoch"]
    best_acc = to_restore["best_acc"]

    for epoch in range(start_epoch, args.epochs):
        # train_loader.sampler.set_epoch(epoch)

        train_stats = train(model, linear_classifier, optimizer, train_loader, epoch, args.n_last_blocks, args.avgpool_patchtokens)
        scheduler.step()

        log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
                     'epoch': epoch}
        if epoch % args.val_freq == 0 or epoch == args.epochs - 1:
            test_stats = validate_network(val_loader, model, linear_classifier, args.n_last_blocks, args.avgpool_patchtokens)
            print(f"Accuracy at epoch {epoch} of the network on the {len(val_dataset)} test images: {test_stats['acc1']:.1f}%")
            best_acc = max(best_acc, test_stats["acc1"])
            print(f'Max accuracy so far: {best_acc:.2f}%')
            log_stats = {**{k: v for k, v in log_stats.items()},
                         **{f'test_{k}': v for k, v in test_stats.items()}}
        if utils.is_main_process():
            json.dumps(log_stats)
            # with (Path(real_output_dir) / f"{save_prefix}_log.txt").open("a") as f:
            #     f.write(json.dumps(log_stats) + "\n")
            # save_dict = {
            #     "epoch": epoch + 1,
            #     "state_dict": linear_classifier.state_dict(),
            #     "optimizer": optimizer.state_dict(),
            #     "scheduler": scheduler.state_dict(),
            #     "best_acc": best_acc,
            # }
            # torch.save(save_dict, os.path.join(real_output_dir, f"{save_prefix}_checkpoint.pth.tar"))
    print("Training of the supervised linear classifier on frozen features completed.\n"
          "Top-1 test accuracy: {acc:.1f}".format(acc=best_acc))

    with (Path(f"summary_lin_eval_log_{args.victim}.txt")).open("a") as f:
        f.write(f"{save_prefix} Top-1 test accuracy: {best_acc} on downstream data {args.downstreamdata}. Results in {real_output_dir}. With {args.use_softmax} softmax." + "\n")
    with (Path(real_output_dir) / f"{save_prefix}_log.txt").open("a") as f:
        f.write(f"Best accuracy is {best_acc}" + "\n")

def train(model, linear_classifier, optimizer, loader, epoch, n, avgpool):
    linear_classifier.train()
    metric_logger = utils.MetricLogger(delimiter="  ")
    metric_logger.add_meter('lr', utils.SmoothedValue(window_size=1, fmt='{value:.6f}'))
    header = 'Epoch: [{}]'.format(epoch)
    for (inp, target) in metric_logger.log_every(loader, 20, header):
        # move to gpu
        inp = inp.cuda(non_blocking=True)
        target = target.cuda(non_blocking=True)

        # forward
        with torch.no_grad():
            if not args.API_embedding:
                output = model.get_image_representation(x=inp, n=n, avgpool=avgpool)
            else:
                output = inp
            if args.use_softmax:
                output = torch.nn.functional.softmax(output, dim=1)
        output = linear_classifier(output)

        # compute cross entropy loss
        loss = nn.CrossEntropyLoss()(output, target)

        # compute the gradients
        optimizer.zero_grad()
        loss.backward()

        # step
        optimizer.step()

        # log
        torch.cuda.synchronize()
        metric_logger.update(loss=loss.item())
        metric_logger.update(lr=optimizer.param_groups[0]["lr"])
    # gather the stats from all processes
    metric_logger.synchronize_between_processes()
    print("Averaged stats:", metric_logger)
    return {k: meter.global_avg for k, meter in metric_logger.meters.items()}


@torch.no_grad()
def validate_network(val_loader, model, linear_classifier, n, avgpool):
    linear_classifier.eval()
    metric_logger = utils.MetricLogger(delimiter="  ")
    header = 'Test:'
    for inp, target in metric_logger.log_every(val_loader, 20, header):
        # move to gpu
        inp = inp.cuda(non_blocking=True)
        target = target.cuda(non_blocking=True)

        # forward
        with torch.no_grad():
            if not args.API_embedding:
                output = model.get_image_representation(x=inp, n=n, avgpool=avgpool)
            else:
                output = inp
            if args.use_softmax:
                output = torch.nn.functional.softmax(output, dim=1)
        output = linear_classifier(output)
        loss = nn.CrossEntropyLoss()(output, target)

        if linear_classifier.module.num_labels >= 5:
            acc1, acc5 = utils.accuracy(output, target, topk=(1, 5))
        else:
            acc1, = utils.accuracy(output, target, topk=(1,))

        batch_size = inp.shape[0]
        metric_logger.update(loss=loss.item())
        metric_logger.meters['acc1'].update(acc1.item(), n=batch_size)
        if linear_classifier.module.num_labels >= 5:
            metric_logger.meters['acc5'].update(acc5.item(), n=batch_size)
    if linear_classifier.module.num_labels >= 5:
        print('* Acc@1 {top1.global_avg:.3f} Acc@5 {top5.global_avg:.3f} loss {losses.global_avg:.3f}'
              .format(top1=metric_logger.acc1, top5=metric_logger.acc5, losses=metric_logger.loss))
    else:
        print('* Acc@1 {top1.global_avg:.3f} loss {losses.global_avg:.3f}'
              .format(top1=metric_logger.acc1, losses=metric_logger.loss))
    return {k: meter.global_avg for k, meter in metric_logger.meters.items()}


class LinearClassifier(nn.Module):
    """Linear layer to train on top of frozen features"""
    def __init__(self, dim, num_labels=1000):
        super(LinearClassifier, self).__init__()
        self.num_labels = num_labels
        self.linear = nn.Linear(dim, num_labels)
        self.linear.weight.data.normal_(mean=0.0, std=0.01)
        self.linear.bias.data.zero_()

    def forward(self, x):
        # flatten
        x = x.view(x.size(0), -1)

        # linear layer
        return self.linear(x)


if __name__ == '__main__':
    parser = argparse.ArgumentParser('Evaluation with linear classification on ImageNet')
    parser.add_argument('--n_last_blocks', default=4, type=int, help="""Concatenate [CLS] tokens
        for the `n` last blocks. We use `n=4` when evaluating ViT-Small and `n=1` with ViT-Base.""")
    parser.add_argument('--avgpool_patchtokens', default=False, type=bool,
                        help="""Whether ot not to concatenate the global average pooled features to the [CLS] token.
        We typically set this to False for ViT-Small and to True with ViT-Base.""")
    parser.add_argument('--arch', default='vit_small', type=str, help='Architecture')
    parser.add_argument('--patch_size', default=16, type=int, help='Patch resolution of the model.')
    parser.add_argument('--pretrained_weights', default='', type=str, help="Path to pretrained weights to evaluate.")
    parser.add_argument("--checkpoint_key", default="teacher", type=str, help='Key to use in the checkpoint (example: "teacher")')
    parser.add_argument('-j', '--workers', default=20, type=int, metavar='N',
                        help='number of data loading workers (default: 32)')
    parser.add_argument('--epochs', default=100, type=int, help='Number of epochs of training.')
    parser.add_argument("--lr", default=0.01, type=float, help="""Learning rate at the beginning of
        training (highest LR used during training). The learning rate is linearly scaled
        with the batch size, and specified here for a reference batch size of 256.
        We recommend tweaking the LR depending on the checkpoint evaluated.""")
    parser.add_argument('--batch_size_per_gpu', default=128, type=int, help='Per-GPU batch-size')
    parser.add_argument("--dist_url", default="env://", type=str, help="""url used to set up
        distributed training; see https://pytorch.org/docs/stable/distributed.html""")
    parser.add_argument("--local_rank", default=0, type=int, help="Please ignore and do not set this argument.")
    parser.add_argument('--downstreamdata', default='cifar10', choices=['cifar10', 'cifar100', 'imagenet', 'svhn', 'stl10'], type=str)
    parser.add_argument('--experiment_folder', default=None, type=str)
    parser.add_argument('--num_workers', default=10, type=int, help='Number of data loading workers per GPU.')
    parser.add_argument('--val_freq', default=1, type=int, help="Epoch frequency for validation.")
    parser.add_argument('--output_dir', default="linear_eval", help='Path to save logs and dino_deitsmall16_pretrain')
    parser.add_argument('--evaluate', dest='evaluate', action='store_true', help='evaluate model on validation set')
    parser.add_argument('--simple_load', action='store_true',
                        help='Whether resume the checkpoint same as the save '
                             'checkpoint path.')
    parser.add_argument('--load_file', default=None, type=str)
    parser.add_argument('--load_victim', default=False, type=bool)
    parser.add_argument('--victim', default=None, type=str)
    parser.add_argument('--losstype', default='mse', choices=['infonce', 'softnn', 'mse'], type=str)
    parser.add_argument('--API_embedding', default=None, choices=['cifar10', None, 'svhn', 'stl10'], type=str)
    parser.add_argument('--num_queries', default=50000, type=int)
    parser.add_argument('--head_layer', default=3, type=int)
    parser.add_argument('--out_dim', default=65536, type=int)
    parser.add_argument('--use_softmax', action='store_true', default=False)
    parser.add_argument('--datasetsteal', default='cifar10', choices=['cifar10', 'cifar100', 'imagenet', 'svhn', 'stl10'], type=str)
    # parser.add_argument('--temperature', default=0.07, type=float,
    #                     help='softmax temperature (default: 0.07)')

    args = parser.parse_args()
    # args.batch_size = args.batch_size_per_gpu
    # if args.losstype == "mse":
    #     criterion = nn.MSELoss().cuda()
    # elif args.losstype == "infonce":
    #     criterion = nn.CrossEntropyLoss().cuda()

    # args.world_size = utils.get_world_size()
    # args.distributed = args.world_size > 1
    args.world_size = 1
    args.distributed = False

    eval_linear(args)
