import torch
from torch import nn

from einops import rearrange, repeat
from einops.layers.torch import Rearrange


# helpers

def pair(t):
    return t if isinstance(t, tuple) else (t, t)


# classes

class FeedForward(nn.Module):
    def __init__(self, dim, hidden_dim, dropout=0.):
        super().__init__()
        self.net = nn.Sequential(
            nn.LayerNorm(dim),
            nn.Linear(dim, hidden_dim),
            nn.GELU(),
            nn.Dropout(dropout),
            nn.Linear(hidden_dim, dim),
            nn.Dropout(dropout)
        )

    def forward(self, x):
        return self.net(x)


class Attention(nn.Module):
    def __init__(self, dim, heads=8, dim_head=64, dropout=0.):
        super().__init__()
        inner_dim = dim_head * heads
        project_out = not (heads == 1 and dim_head == dim)

        self.heads = heads
        self.scale = dim_head ** -0.5

        self.norm = nn.LayerNorm(dim)

        self.attend = nn.Softmax(dim=-1)
        self.dropout = nn.Dropout(dropout)

        self.to_qkv = nn.Linear(dim, inner_dim * 3, bias=False)

        self.to_out = nn.Sequential(
            nn.Linear(inner_dim, dim),
            nn.Dropout(dropout)
        ) if project_out else nn.Identity()

    def forward(self, x):
        x = self.norm(x)

        qkv = self.to_qkv(x).chunk(3, dim=-1)
        q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h=self.heads), qkv)

        dots = torch.matmul(q, k.transpose(-1, -2)) * self.scale

        attn = self.attend(dots)
        attn = self.dropout(attn)

        out = torch.matmul(attn, v)
        out = rearrange(out, 'b h n d -> b n (h d)')
        return self.to_out(out)


class Transformer(nn.Module):
    def __init__(self, dim, depth, heads, dim_head, mlp_dim, dropout=0.):
        super().__init__()
        self.norm = nn.LayerNorm(dim)
        self.layers = nn.ModuleList([])
        for _ in range(depth):
            self.layers.append(nn.ModuleList([
                Attention(dim, heads=heads, dim_head=dim_head, dropout=dropout),
                FeedForward(dim, mlp_dim, dropout=dropout)
            ]))

    def forward(self, x):
        for attn, ff in self.layers:
            x = attn(x) + x
            x = ff(x) + x

        return self.norm(x)


class ViT(nn.Module):
    def __init__(self, image_size, p_in, dim, depth, heads, mlp_dim, dim_in, dim_out, p_out, dim_head=64, dropout=0.,
                 emb_dropout=0., patch_in_out=(True, True), maxlen=1000):
        super().__init__()
        self.patch_in_out = patch_in_out
        image_height, image_width = pair(image_size)
        p_h_in, p_w_in = pair(p_in)
        p_h_out, p_w_out = pair(p_out)

        assert image_height % p_h_in == 0 and image_width % p_w_in == 0, 'Image dimensions must be divisible by the patch size.'

        if patch_in_out[0]:
            patch_dim_in = dim_in * p_h_in * p_w_in
            num_patches = (image_height // p_h_in) * (image_width // p_w_in)
            self.pos_embedding = nn.Parameter(torch.randn(1, num_patches, dim))
            self.to_patch_embedding = nn.Sequential(
                Rearrange('b c (h p1) (w p2) -> b (h w) (p1 p2 c)', p1=p_h_in, p2=p_w_in),
                nn.LayerNorm(patch_dim_in),
                nn.Identity() if patch_dim_in == dim else nn.Linear(patch_dim_in, dim),
                nn.LayerNorm(dim)
            )
        else:
            self.pos_embedding = nn.Parameter(torch.randn(1, maxlen, dim))
            self.proj_in = nn.Linear(dim_in, dim)

        if patch_in_out[1]:
            patch_dim_out = dim_out * p_h_out * p_w_out
            self.from_patch_embedding = nn.Sequential(
                nn.Identity() if patch_dim_out == dim else nn.Linear(dim, patch_dim_out),
                Rearrange('b (h w) (p1 p2 c)-> b c (h p1) (w p2)', h=image_height // p_h_in,
                          w=image_width // p_w_in, p1=p_h_out,
                          p2=p_w_out),
            )
        else:
            self.proj_out = nn.Linear(dim, dim_out)

        self.dropout = nn.Dropout(emb_dropout)
        self.transformer = Transformer(dim, depth, heads, dim_head, mlp_dim, dropout)

    def forward(self, x):
        if self.patch_in_out[0]:
            x = self.to_patch_embedding(x)
        else:
            x = self.proj_in(x)

        x += self.pos_embedding[:, :x.size(1), :]
        x = self.dropout(x)
        x = self.transformer(x)

        if self.patch_in_out[1]:
            x = self.from_patch_embedding(x)
        else:
            x = self.proj_out(x)
        return x
