import torch
import torch.nn as nn
import torch.nn.functional as F

__all__ = ['resnet18_under_d2_bn','resnet18_under_d4_bn','resnet18_under_d8_bn','resnet18_under_d16_bn']

#https://github.com/kuangliu/pytorch-cifar/blob/master/models/resnet.py

class BasicBlock_bn(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock_bn, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck_bn(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck_bn, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet_bn_d(nn.Module):
    def __init__(self, block, num_blocks, divide, num_classes=10):
        super(ResNet_bn_d, self).__init__()
        self.in_planes = int(64/divide)

        self.conv1 = nn.Conv2d(3, int(64/divide), kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(int(64/divide))
        self.layer1 = self._make_layer(block, int(64/divide), num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, int(128/divide), num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, int(256/divide), num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, int(512/divide), num_blocks[3], stride=2)
        self.linear = nn.Linear(int(512/divide)*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out

    def get_features(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        return out

def resnet18_under_d2_bn(**kwargs):
    return ResNet_bn_d(BasicBlock_bn, [2, 2, 2, 2], 2, **kwargs)

def resnet18_under_d4_bn(**kwargs):
    return ResNet_bn_d(BasicBlock_bn, [2, 2, 2, 2], 4, **kwargs)

def resnet18_under_d8_bn(**kwargs):
    return ResNet_bn_d(BasicBlock_bn, [2, 2, 2, 2], 8, **kwargs)

def resnet18_under_d16_bn(**kwargs):
    return ResNet_bn_d(BasicBlock_bn, [2, 2, 2, 2], 16, **kwargs)
