import torch
from torch.optim.optimizer import Optimizer 
import copy
 

class SGDGD(Optimizer):
    

    def __init__(self, params, lr, lrsgd, momentum=0, dampening=0,
                 weight_decay=0, nesterov=False):
      
        defaults = dict(lr=lr, lrsgd=lrsgd, momentum=momentum, dampening=dampening,
                        weight_decay=weight_decay, nesterov=nesterov)
        if nesterov and (momentum <= 0 or dampening != 0):
            raise ValueError("Nesterov momentum requires a momentum and zero dampening")
        super(SGDGD, self).__init__(params, defaults)

    def __setstate__(self, state):
        super(SGDGD, self).__setstate__(state)
        for group in self.param_groups:
            group.setdefault('nesterov', False)

    @torch.no_grad()
    def step(self, grads2, closure=None):
      
        loss = None
        if closure is not None:
            with torch.enable_grad():
                loss = closure()

        for group in self.param_groups:
            weight_decay = group['weight_decay']
            momentum = group['momentum']
            dampening = group['dampening']
            nesterov = group['nesterov']

            for p, g2 in zip(group['params'],grads2):
                if p.grad is None:
                    continue
                d_p = p.grad
                if weight_decay != 0:
                    d_p = d_p.add(p, alpha=weight_decay)
                    
                sgd_noise=g2-copy.deepcopy(d_p)

                if momentum != 0:
                    param_state = self.state[p]
                    if 'momentum_buffer' not in param_state:
                        buf = param_state['momentum_buffer'] = torch.clone(d_p).detach()
                    else:
                        buf = param_state['momentum_buffer']
                        buf.mul_(momentum).add_(d_p, alpha=1 - dampening)
                    if nesterov:
                        d_p = d_p.add(buf, alpha=momentum)
                    else:
                        d_p = buf
                
               # print(d_p.shape)
               # print(dict_model2[nam].shape)
                
                
                p.add_(d_p, alpha=-group['lr'])
                
                
                
                p.add_(sgd_noise, alpha=-group['lrsgd'])

        return loss





class GD_momnoise(Optimizer):
    

    def __init__(self, params, lr, lrsgd, momentum=0, dampening=0,
                 weight_decay=0, nesterov=False):
      
        defaults = dict(lr=lr, lrsgd=lrsgd, momentum=momentum, dampening=dampening,
                        weight_decay=weight_decay, nesterov=nesterov)
        if nesterov and (momentum <= 0 or dampening != 0):
            raise ValueError("Nesterov momentum requires a momentum and zero dampening")
        super(GD_momnoise, self).__init__(params, defaults)

    def __setstate__(self, state):
        super(GD_momnoise, self).__setstate__(state)
        for group in self.param_groups:
            group.setdefault('nesterov', False)

    @torch.no_grad()
    def step(self, grads2, closure=None):
      
        loss = None
        if closure is not None:
            with torch.enable_grad():
                loss = closure()

        for group in self.param_groups:
            weight_decay = group['weight_decay']
            momentum = group['momentum']
            dampening = group['dampening']
            nesterov = group['nesterov']

            for p, g2 in zip(group['params'],grads2):
                if p.grad is None:
                    continue
                d_p = p.grad
                if weight_decay != 0:
                    d_p = d_p.add(p, alpha=weight_decay)
                    
                sgd_noise=g2-copy.deepcopy(d_p)

                if momentum != 0:
                    param_state = self.state[p]
                    if 'momentum_buffer' not in param_state:
                        buf = param_state['momentum_buffer'] = torch.clone(sgd_noise).detach()
                    else:
                        buf = param_state['momentum_buffer']
                        buf.mul_(momentum).add_(sgd_noise, alpha=1 - dampening)
                    if nesterov:
                        sgd_noise = sgd_noise.add(buf, alpha=momentum)
                    else:
                        sgd_noise = buf
                
               # print(d_p.shape)
               # print(dict_model2[nam].shape)
                
                
                p.add_(d_p, alpha=-group['lr'])
                
                
                
                p.add_(sgd_noise, alpha=-group['lrsgd'])

        return loss








