import math
import os
import warnings
from dataclasses import dataclass
from typing import List, Optional, Tuple, Union

import torch
import torch.utils.checkpoint
from torch import nn
from torch.nn import BCEWithLogitsLoss, CrossEntropyLoss, MSELoss

from transformers.modeling_outputs import BaseModelOutputWithPoolingAndCrossAttentions, SequenceClassifierOutput

from transformers.models.bert.modeling_bert import BertPreTrainedModel, BertEmbeddings, BertEncoder, BertPooler

from transformers.utils import (ModelOutput,
    add_code_sample_docstrings,
    add_start_docstrings,
    add_start_docstrings_to_model_forward,
    logging,
    replace_return_docstrings,)



_CHECKPOINT_FOR_DOC = "bert-base-uncased"
_CONFIG_FOR_DOC = "BertConfig"
_TOKENIZER_FOR_DOC = "BertTokenizer"

## TokenClassification docstring
#_CHECKPOINT_FOR_TOKEN_CLASSIFICATION = "dbmdz/bert-large-cased-finetuned-conll03-english"
#_TOKEN_CLASS_EXPECTED_OUTPUT = (
#    "['O', 'I-ORG', 'I-ORG', 'I-ORG', 'O', 'O', 'O', 'O', 'O', 'I-LOC', 'O', 'I-LOC', 'I-LOC'] "
#)
#_TOKEN_CLASS_EXPECTED_LOSS = 0.01
#
## QuestionAnswering docstring
#_CHECKPOINT_FOR_QA = "deepset/bert-base-cased-squad2"
#_QA_EXPECTED_OUTPUT = "'a nice puppet'"
#_QA_EXPECTED_LOSS = 7.41
#_QA_TARGET_START_INDEX = 14
#_QA_TARGET_END_INDEX = 15
#
## SequenceClassification docstring
#_CHECKPOINT_FOR_SEQUENCE_CLASSIFICATION = "textattack/bert-base-uncased-yelp-polarity"
#_SEQ_CLASS_EXPECTED_OUTPUT = "'LABEL_1'"
#_SEQ_CLASS_EXPECTED_LOSS = 0.01
#
#
#BERT_START_DOCSTRING = r"""
#    This model inherits from [`PreTrainedModel`]. Check the superclass documentation for the generic methods the
#    library implements for all its model (such as downloading or saving, resizing the input embeddings, pruning heads
#    etc.)
#    This model is also a PyTorch [torch.nn.Module](https://pytorch.org/docs/stable/nn.html#torch.nn.Module) subclass.
#    Use it as a regular PyTorch Module and refer to the PyTorch documentation for all matter related to general usage
#    and behavior.
#    Parameters:
#        config ([`BertConfig`]): Model configuration class with all the parameters of the model.
#            Initializing with a config file does not load the weights associated with the model, only the
#            configuration. Check out the [`~PreTrainedModel.from_pretrained`] method to load the model weights.
#"""
#
#BERT_INPUTS_DOCSTRING = r"""
#    Args:
#        input_ids (`torch.LongTensor` of shape `({0})`):
#            Indices of input sequence tokens in the vocabulary.
#            Indices can be obtained using [`BertTokenizer`]. See [`PreTrainedTokenizer.encode`] and
#            [`PreTrainedTokenizer.__call__`] for details.
#            [What are input IDs?](../glossary#input-ids)
#        attention_mask (`torch.FloatTensor` of shape `({0})`, *optional*):
#            Mask to avoid performing attention on padding token indices. Mask values selected in `[0, 1]`:
#            - 1 for tokens that are **not masked**,
#            - 0 for tokens that are **masked**.
#            [What are attention masks?](../glossary#attention-mask)
#        token_type_ids (`torch.LongTensor` of shape `({0})`, *optional*):
#            Segment token indices to indicate first and second portions of the inputs. Indices are selected in `[0,
#            1]`:
#            - 0 corresponds to a *sentence A* token,
#            - 1 corresponds to a *sentence B* token.
#            [What are token type IDs?](../glossary#token-type-ids)
#        position_ids (`torch.LongTensor` of shape `({0})`, *optional*):
#            Indices of positions of each input sequence tokens in the position embeddings. Selected in the range `[0,
#            config.max_position_embeddings - 1]`.
#            [What are position IDs?](../glossary#position-ids)
#        head_mask (`torch.FloatTensor` of shape `(num_heads,)` or `(num_layers, num_heads)`, *optional*):
#            Mask to nullify selected heads of the self-attention modules. Mask values selected in `[0, 1]`:
#            - 1 indicates the head is **not masked**,
#            - 0 indicates the head is **masked**.
#        inputs_embeds (`torch.FloatTensor` of shape `({0}, hidden_size)`, *optional*):
#            Optionally, instead of passing `input_ids` you can choose to directly pass an embedded representation. This
#            is useful if you want more control over how to convert `input_ids` indices into associated vectors than the
#            model's internal embedding lookup matrix.
#        output_attentions (`bool`, *optional*):
#            Whether or not to return the attentions tensors of all attention layers. See `attentions` under returned
#            tensors for more detail.
#        output_hidden_states (`bool`, *optional*):
#            Whether or not to return the hidden states of all layers. See `hidden_states` under returned tensors for
#            more detail.
#        return_dict (`bool`, *optional*):
#            Whether or not to return a [`~utils.ModelOutput`] instead of a plain tuple.
#"""
#
#
#@add_start_docstrings(
#    "The bare Bert Model transformer outputting raw hidden-states without any specific head on top.",
#    BERT_START_DOCSTRING,
#)

class BertNoEmbed(BertPreTrainedModel):
    """
    The model can behave as an encoder (with only self-attention) as well as a decoder, in which case a layer of
    cross-attention is added between the self-attention layers, following the architecture described in [Attention is
    all you need](https://arxiv.org/abs/1706.03762) by Ashish Vaswani, Noam Shazeer, Niki Parmar, Jakob Uszkoreit,
    Llion Jones, Aidan N. Gomez, Lukasz Kaiser and Illia Polosukhin.
    To behave as an decoder the model needs to be initialized with the `is_decoder` argument of the configuration set
    to `True`. To be used in a Seq2Seq model, the model needs to initialized with both `is_decoder` argument and
    `add_cross_attention` set to `True`; an `encoder_hidden_states` is then expected as an input to the forward pass.
    """

    def __init__(self, config, add_pooling_layer=True):
        super().__init__(config)
        self.config = config

        #self.embeddings = BertEmbeddings(config)
        self.encoder = BertEncoder(config)

        self.pooler = BertPooler(config) if add_pooling_layer else None

        classifier_dropout = (
            self.config.classifier_dropout if self.config.classifier_dropout is not None else self.config.hidden_dropout_prob
        )

        self.dropout = nn.Dropout(classifier_dropout)
        self.classifier = nn.Linear(self.config.hidden_size, self.config.num_labels)
        self.num_labels = self.config.num_labels

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.embeddings.word_embeddings

    def set_input_embeddings(self, value):
        self.embeddings.word_embeddings = value

    def _prune_heads(self, heads_to_prune):
        """
        Prunes heads of the model. heads_to_prune: dict of {layer_num: list of heads to prune in this layer} See base
        class PreTrainedModel
        """
        for layer, heads in heads_to_prune.items():
            self.encoder.layer[layer].attention.prune_heads(heads)

#    @add_start_docstrings_to_model_forward(BERT_INPUTS_DOCSTRING.format("batch_size, sequence_length"))
#    @add_code_sample_docstrings(
#        processor_class=_TOKENIZER_FOR_DOC,
#        checkpoint=_CHECKPOINT_FOR_DOC,
#        output_type=BaseModelOutputWithPoolingAndCrossAttentions,
#        config_class=_CONFIG_FOR_DOC,
#    )

#    classifier_dropout = (
#            config.classifier_dropout if config.classifier_dropout is not None else config.hidden_dropout_prob
#        )
#
#    self.dropout = nn.Dropout(classifier_dropout)
#    self.classifier = nn.Linear(config.hidden_size, config.num_labels) 
#    self.post_init()


    def forward(
        self,
        input_ids: Optional[torch.Tensor] = None,
        attention_mask: Optional[torch.Tensor] = None,
        token_type_ids: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.Tensor] = None,
        head_mask: Optional[torch.Tensor] = None,
        inputs_embeds: Optional[torch.Tensor] = None,
        labels: Optional[torch.Tensor] = None,
        encoder_hidden_states: Optional[torch.Tensor] = None,
        encoder_attention_mask: Optional[torch.Tensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple[torch.Tensor], BaseModelOutputWithPoolingAndCrossAttentions]:
        r"""
        encoder_hidden_states  (`torch.FloatTensor` of shape `(batch_size, sequence_length, hidden_size)`, *optional*):
            Sequence of hidden-states at the output of the last layer of the encoder. Used in the cross-attention if
            the model is configured as a decoder.
        encoder_attention_mask (`torch.FloatTensor` of shape `(batch_size, sequence_length)`, *optional*):
            Mask to avoid performing attention on the padding token indices of the encoder input. This mask is used in
            the cross-attention if the model is configured as a decoder. Mask values selected in `[0, 1]`:
            - 1 for tokens that are **not masked**,
            - 0 for tokens that are **masked**.
        past_key_values (`tuple(tuple(torch.FloatTensor))` of length `config.n_layers` with each tuple having 4 tensors of shape `(batch_size, num_heads, sequence_length - 1, embed_size_per_head)`):
            Contains precomputed key and value hidden states of the attention blocks. Can be used to speed up decoding.
            If `past_key_values` are used, the user can optionally input only the last `decoder_input_ids` (those that
            don't have their past key value states given to this model) of shape `(batch_size, 1)` instead of all
            `decoder_input_ids` of shape `(batch_size, sequence_length)`.
        use_cache (`bool`, *optional*):
            If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding (see
            `past_key_values`).
        """
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        if self.config.is_decoder:
            use_cache = use_cache if use_cache is not None else self.config.use_cache
        else:
            use_cache = False

#        if input_ids is not None and inputs_embeds is not None:
#            raise ValueError("You cannot specify both input_ids and inputs_embeds at the same time")
#        elif input_ids is not None:
#            input_shape = input_ids.size()
#        elif inputs_embeds is not None:
#            input_shape = inputs_embeds.size()[:-1]
#        else:
#            raise ValueError("You have to specify either input_ids or inputs_embeds")
 
        input_shape = inputs_embeds.size()[:-1] 
  
        batch_size, seq_length = input_shape
        device = input_ids.device if input_ids is not None else inputs_embeds.device

        # past_key_values_length
        past_key_values_length = past_key_values[0][0].shape[2] if past_key_values is not None else 0

        if attention_mask is None:
            attention_mask = torch.ones(((batch_size, seq_length + past_key_values_length)), device=device)
        #token_type_ids = torch.zeros(input_shape, dtype=torch.long, device=device)


#        if token_type_ids is None:
#            if hasattr(self.embeddings, "token_type_ids"):
#                buffered_token_type_ids = self.embeddings.token_type_ids[:, :seq_length]
#                buffered_token_type_ids_expanded = buffered_token_type_ids.expand(batch_size, seq_length)
#                token_type_ids = buffered_token_type_ids_expanded
#            else:
#                token_type_ids = torch.zeros(input_shape, dtype=torch.long, device=device)

        # We can provide a self-attention mask of dimensions [batch_size, from_seq_length, to_seq_length]
        # ourselves in which case we just need to make it broadcastable to all heads.
        extended_attention_mask: torch.Tensor = self.get_extended_attention_mask(attention_mask, input_shape)


        encoder_extended_attention_mask = None


        # If a 2D or 3D attention mask is provided for the cross-attention
        # we need to make broadcastable to [batch_size, num_heads, seq_length, seq_length]
#        if self.config.is_decoder and encoder_hidden_states is not None:
#            encoder_batch_size, encoder_sequence_length, _ = encoder_hidden_states.size()
#            encoder_hidden_shape = (encoder_batch_size, encoder_sequence_length)
#            if encoder_attention_mask is None:
#                encoder_attention_mask = torch.ones(encoder_hidden_shape, device=device)
#            encoder_extended_attention_mask = self.invert_attention_mask(encoder_attention_mask)
#        else:
#            encoder_extended_attention_mask = None

        # Prepare head mask if needed
        # 1.0 in head_mask indicate we keep the head
        # attention_probs has shape bsz x n_heads x N x N
        # input head_mask has shape [num_heads] or [num_hidden_layers x num_heads]
        # and head_mask is converted to shape [num_hidden_layers x batch x num_heads x seq_length x seq_length]
        head_mask = self.get_head_mask(head_mask, self.config.num_hidden_layers)

#        embedding_output = self.embeddings(
#            input_ids=input_ids,
#            position_ids=position_ids,
#            token_type_ids=token_type_ids,
#            inputs_embeds=inputs_embeds,
#            past_key_values_length=past_key_values_length,
#        )
#embedding_output,
        encoder_outputs = self.encoder(
            inputs_embeds, 
            attention_mask=extended_attention_mask,
            head_mask=head_mask,
            encoder_hidden_states=encoder_hidden_states,
            encoder_attention_mask=encoder_extended_attention_mask,
            past_key_values=past_key_values,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        sequence_output = encoder_outputs[0]
        pooled_output = self.pooler(sequence_output) if self.pooler is not None else None
        
        if not return_dict:
            return (sequence_output, pooled_output) + encoder_outputs[1:]

        outputs= BaseModelOutputWithPoolingAndCrossAttentions(
            last_hidden_state=sequence_output,
            pooler_output=pooled_output,
            past_key_values=encoder_outputs.past_key_values,
            hidden_states=encoder_outputs.hidden_states,
            attentions=encoder_outputs.attentions,
            cross_attentions=encoder_outputs.cross_attentions,
        )

        pooled_output = outputs[1]
        pooled_output = self.dropout(pooled_output)

        #print(f"HELP {pooled_output.shape}")
            

        logits = self.classifier(pooled_output)

        loss = None
        if labels is not None:
            if self.config.problem_type is None:
                if self.num_labels == 1:
                    self.config.problem_type = "regression"
                elif self.num_labels > 1 and (labels.dtype == torch.long or labels.dtype == torch.int):
                    self.config.problem_type = "single_label_classification"
                else:
                    self.config.problem_type = "multi_label_classification"

            if self.config.problem_type == "regression":
                loss_fct = MSELoss()
                if self.num_labels == 1:
                    loss = loss_fct(logits.squeeze(), labels.squeeze())
                else:
                    loss = loss_fct(logits, labels)
            elif self.config.problem_type == "single_label_classification":
                loss_fct = CrossEntropyLoss()
                loss = loss_fct(logits.view(-1, self.num_labels), labels.view(-1))
            elif self.config.problem_type == "multi_label_classification":
                loss_fct = BCEWithLogitsLoss()
                loss = loss_fct(logits, labels)

        return SequenceClassifierOutput(
            loss=loss,
            logits=logits,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )


    def get_features(
        self,
        input_ids: Optional[torch.Tensor] = None,
        attention_mask: Optional[torch.Tensor] = None,
        token_type_ids: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.Tensor] = None,
        head_mask: Optional[torch.Tensor] = None,
        inputs_embeds: Optional[torch.Tensor] = None,
        labels: Optional[torch.Tensor] = None,
        encoder_hidden_states: Optional[torch.Tensor] = None,
        encoder_attention_mask: Optional[torch.Tensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple[torch.Tensor], BaseModelOutputWithPoolingAndCrossAttentions]:
        r"""
        encoder_hidden_states  (`torch.FloatTensor` of shape `(batch_size, sequence_length, hidden_size)`, *optional*):
            Sequence of hidden-states at the output of the last layer of the encoder. Used in the cross-attention if
            the model is configured as a decoder.
        encoder_attention_mask (`torch.FloatTensor` of shape `(batch_size, sequence_length)`, *optional*):
            Mask to avoid performing attention on the padding token indices of the encoder input. This mask is used in
            the cross-attention if the model is configured as a decoder. Mask values selected in `[0, 1]`:
            - 1 for tokens that are **not masked**,
            - 0 for tokens that are **masked**.
        past_key_values (`tuple(tuple(torch.FloatTensor))` of length `config.n_layers` with each tuple having 4 tensors of shape `(batch_size, num_heads, sequence_length - 1, embed_size_per_head)`):
            Contains precomputed key and value hidden states of the attention blocks. Can be used to speed up decoding.
            If `past_key_values` are used, the user can optionally input only the last `decoder_input_ids` (those that
            don't have their past key value states given to this model) of shape `(batch_size, 1)` instead of all
            `decoder_input_ids` of shape `(batch_size, sequence_length)`.
        use_cache (`bool`, *optional*):
            If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding (see
            `past_key_values`).
        """
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        if self.config.is_decoder:
            use_cache = use_cache if use_cache is not None else self.config.use_cache
        else:
            use_cache = False

#        if input_ids is not None and inputs_embeds is not None:
#            raise ValueError("You cannot specify both input_ids and inputs_embeds at the same time")
#        elif input_ids is not None:
#            input_shape = input_ids.size()
#        elif inputs_embeds is not None:
#            input_shape = inputs_embeds.size()[:-1]
#        else:
#            raise ValueError("You have to specify either input_ids or inputs_embeds")
 
        input_shape = inputs_embeds.size()[:-1] 
  
        batch_size, seq_length = input_shape
        device = input_ids.device if input_ids is not None else inputs_embeds.device

        # past_key_values_length
        past_key_values_length = past_key_values[0][0].shape[2] if past_key_values is not None else 0

        if attention_mask is None:
            attention_mask = torch.ones(((batch_size, seq_length + past_key_values_length)), device=device)
        #token_type_ids = torch.zeros(input_shape, dtype=torch.long, device=device)


#        if token_type_ids is None:
#            if hasattr(self.embeddings, "token_type_ids"):
#                buffered_token_type_ids = self.embeddings.token_type_ids[:, :seq_length]
#                buffered_token_type_ids_expanded = buffered_token_type_ids.expand(batch_size, seq_length)
#                token_type_ids = buffered_token_type_ids_expanded
#            else:
#                token_type_ids = torch.zeros(input_shape, dtype=torch.long, device=device)

        # We can provide a self-attention mask of dimensions [batch_size, from_seq_length, to_seq_length]
        # ourselves in which case we just need to make it broadcastable to all heads.
        extended_attention_mask: torch.Tensor = self.get_extended_attention_mask(attention_mask, input_shape)


        encoder_extended_attention_mask = None


        # If a 2D or 3D attention mask is provided for the cross-attention
        # we need to make broadcastable to [batch_size, num_heads, seq_length, seq_length]
#        if self.config.is_decoder and encoder_hidden_states is not None:
#            encoder_batch_size, encoder_sequence_length, _ = encoder_hidden_states.size()
#            encoder_hidden_shape = (encoder_batch_size, encoder_sequence_length)
#            if encoder_attention_mask is None:
#                encoder_attention_mask = torch.ones(encoder_hidden_shape, device=device)
#            encoder_extended_attention_mask = self.invert_attention_mask(encoder_attention_mask)
#        else:
#            encoder_extended_attention_mask = None

        # Prepare head mask if needed
        # 1.0 in head_mask indicate we keep the head
        # attention_probs has shape bsz x n_heads x N x N
        # input head_mask has shape [num_heads] or [num_hidden_layers x num_heads]
        # and head_mask is converted to shape [num_hidden_layers x batch x num_heads x seq_length x seq_length]
        head_mask = self.get_head_mask(head_mask, self.config.num_hidden_layers)

#        embedding_output = self.embeddings(
#            input_ids=input_ids,
#            position_ids=position_ids,
#            token_type_ids=token_type_ids,
#            inputs_embeds=inputs_embeds,
#            past_key_values_length=past_key_values_length,
#        )
#embedding_output,
        encoder_outputs = self.encoder(
            inputs_embeds, 
            attention_mask=extended_attention_mask,
            head_mask=head_mask,
            encoder_hidden_states=encoder_hidden_states,
            encoder_attention_mask=encoder_extended_attention_mask,
            past_key_values=past_key_values,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        sequence_output = encoder_outputs[0]
        pooled_output = self.pooler(sequence_output) if self.pooler is not None else None
         
        if not return_dict:
            return (sequence_output, pooled_output) + encoder_outputs[1:]

        outputs= BaseModelOutputWithPoolingAndCrossAttentions(
            last_hidden_state=sequence_output,
            pooler_output=pooled_output,
            past_key_values=encoder_outputs.past_key_values,
            hidden_states=encoder_outputs.hidden_states,
            attentions=encoder_outputs.attentions,
            cross_attentions=encoder_outputs.cross_attentions,
        )

        pooled_output = outputs[1]
        pooled_output = self.dropout(pooled_output)


        return pooled_output


