"""
Utilities for plotting the Lending environment results.
"""

from typing import Any

from fair_gym.utils import plotting


def plot_lending(results: dict[str:Any], plot_dir: str) -> None:
    """
    Plot the lending environment results.

    Args:
        results (dict[str, Any]): The results dictionary.
        plot_dir (str): The directory to save the plots.
    """
    # Initial observable and latent credit score distribution
    plotting.plot_credit_score_distribution(
        results["initial_credit_distribution"],
        title="initial_credit_score",
        path=plot_dir,
    )
    plotting.plot_credit_score_distribution(
        results["initial_latent_credit_distribution"],
        title="initial_latent_credit_score",
        path=plot_dir,
    )

    # Final observable and latent credit score distribution
    plotting.plot_credit_score_distribution(
        results["final_credit_distribution"],
        title="final_credit_score",
        path=plot_dir,
    )
    plotting.plot_credit_score_distribution(
        results["final_latent_credit_distribution"],
        title="final_latent_credit_score",
        path=plot_dir,
    )

    # Plot the conscientiousness distribution
    plotting.plot_distribution(
        results["cons_distribution"],
        title="conscientiousness",
        path=plot_dir,
    )

    # Cumulative loan
    plotting.plot_cumulative_metric(
        results["cumulative_loan"],
        title="cumulative_loan",
        path=plot_dir,
    )


def plot_college_admission(results: dict[str, Any], plot_dir: str) -> None:
    """
    Plot the college admission environment results.

    Args:
        results (dict[str, Any]): The results dictionary.
        plot_dir (str): The directory to save the plots.
    """
    # Plot the initial score and budget distributions
    plotting.plot_distribution(
        results["initial_score_distribution"],
        title="initial_score",
        path=plot_dir,
    )
    plotting.plot_distribution(
        results["initial_budget_distribution"],
        title="initial_budget",
        path=plot_dir,
    )

    # Plot the current score and budget distributions
    plotting.plot_distribution(
        results["final_score_distribution"],
        title="final_score",
        path=plot_dir,
    )
    plotting.plot_distribution(
        results["final_budget_distribution"],
        title="final_budget",
        path=plot_dir,
    )

    # Plot cumulative addmisions per step
    plotting.plot_cumulative_metric(
        results["cumulative_admissions"],
        title="cumulative_admissions",
        path=plot_dir,
    )

    # Plot the accepted distribution
    plotting.plot_acceptance_ratio(
        results["acceptance_ratio"],
        title="acceptance_ratio",
        path=plot_dir,
    )
