"""
An example script for running the Lending Gym environment and plotting the results.
"""

import numpy as np

import gymnasium
import fair_gym

from fair_gym.utils import plotting
from fair_gym import LendingMetrics

# Define the environment parameters
env_kwargs = {
    "n_groups": 2,
    "success_func": "latent_credit",
    "group_distribution": (0.5, 0.5),
    "credit_score_distribution": (
        (0.0, 0.0, 0.0, 0.2, 0.3, 0.3, 0.2),
        (0.2, 0.2, 0.3, 0.2, 0.1, 0.0, 0.0),
    ),
    "success_probability": (0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9),
    "cons_distribution": "normal",
    "cons_mean": 0.5,
    "cons_std": 0.1,
    "population_size": 1000,
    "max_credit": 7,
    "loan_amount": 1,
    "interest_rate": 0.3,
    "bank_starting_cash": 1000000,
}
episode_length = 10000

# Create the environment and the metrics
np.random.seed(0)

env = gymnasium.make(
    "fair_gym/LendingEnv", max_episode_steps=episode_length, **env_kwargs
)
metrics = LendingMetrics(env)

# Run the environment
obs, info = env.reset(seed=0)
terminated, truncated = False, False
time_step = 0

while not terminated and not truncated:
    time_step += 1
    action = env.action_space.sample()
    obs, reward, terminated, truncated, info = env.step(action)

# Get the metrics
results = metrics.get_all_metrics()

# Plot the initial observable and latent credit score distributions
plotting.plot_credit_score_distribution(
    results["initial_credit_distribution"],
    title="initial_credit_score",
    path="plots",
)
plotting.plot_credit_score_distribution(
    results["initial_latent_credit_distribution"],
    title="initial_latent_credit_score",
    path="plots",
)

# Plot the final observable and latent credit score distributions
plotting.plot_credit_score_distribution(
    results["final_credit_distribution"],
    title="final_credit_score",
    path="plots",
)
plotting.plot_credit_score_distribution(
    results["final_latent_credit_distribution"],
    title="final_latent_credit_score",
    path="plots",
)

# Plot the conscientiousness distributions
plotting.plot_distribution(
    results["cons_distribution"],
    title="conscientiousness",
    path="plots",
)

# Plot cumulative loans per step
cumulative_loans = metrics.get_cumulative_loan()
plotting.plot_cumulative_metric(
    results["cumulative_loan"],
    title="cumulative_loans",
    path="plots",
)

# Plot the loans given, loans repaid, loans defaulted distributions
plotting.plot_distribution(
    metrics.get_loans_given_distribution(),
    title="loans_given",
    path="plots",
)
plotting.plot_distribution(
    metrics.get_loans_repaid_distribution(),
    title="loans_repaid",
    path="plots",
)
plotting.plot_distribution(
    metrics.get_loans_defaulted_distribution(),
    title="loans_defaulted",
    path="plots",
)

print(f"Time step: {time_step}")
# Print the bank's profit rate
print(f"Bank's profit rate: {results['profit_rate']}")
# Print the agent's recall and precision for each group
print(f"Agent's recall: {results['recall']}")
print(f"Agent's precision: {results['precision']}")
# Print the W distances between the initial credit score distributions
print(f"Initial W distances: {results['initial_w_distances']}")
# Print the W distances between the final credit score distributions
print(f"Final W distances: {results['final_w_distances']}")
# Print the W distances between the initial latent credit score distributions
print(f"Initial latent W distances: {results['initial_latent_w_distances']}")
# Print the W distances between the final latent credit score distributions
print(f"Final latent W distances: {results['final_latent_w_distances']}")
