import numpy as np


def one_hot_encode(value: int, size: int) -> np.ndarray:
    """
    One-hot encode the given value.

    Args:
        value (int): The value to one-hot encode.
        size (int): The size of the one-hot encoding.

    Returns:
        np.ndarray: The one-hot encoding.
    """
    one_hot = np.zeros(size, dtype=int)
    one_hot[value] = 1
    return one_hot


def compute_recall(
    true_positives: list[int],
    false_negatives: list[int],
) -> list[float]:
    """
    Compute the recall.

    Args:
        true_positives (list[int]): The number of true positives.
        false_negatives (list[int]): The number of false negatives.

    Returns:
        list[float]: The recall.
    """
    n = len(true_positives)

    return [
        (
            true_positives[i] / (true_positives[i] + false_negatives[i])
            if (true_positives[i] + false_negatives[i]) != 0
            else 0
        )
        for i in range(n)
    ]


def compute_precision(
    true_positives: list[int],
    false_positives: list[int],
) -> list[float]:
    """
    Compute the precision.

    Args:
        true_positives (list[int]): The number of true positives.
        false_positives (list[int]): The number of false positives.

    Returns:
        list[float]: The precision.
    """
    n = len(true_positives)

    return [
        (
            true_positives[i] / (true_positives[i] + false_positives[i])
            if (true_positives[i] + false_positives[i]) != 0
            else 0
        )
        for i in range(n)
    ]
