#!/usr/bin/env python3
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import copy
import math
import os
import sys
import time
import pickle as pkl
import tqdm
import wandb
import datetime

# from logger import Logger
from replay_buffer import ReplayBuffer
from reward_model import RewardModel
from collections import deque
from agent.sac import SACAgent
from agent.critic import DoubleQCritic
from utils.seqrank_utils import MetaOptim

import utils
import hydra


class Workspace(object):
    def __init__(self, cfg):
        self.work_dir = os.getcwd()
        print(f'workspace: {self.work_dir}')

        self.cfg = cfg
        self.logger = None

        utils.set_seed_everywhere(cfg.seed)
        cfg.device = 'cuda:0' if torch.cuda.is_available() else 'cpu'
        self.device = torch.device(cfg.device)
        self.log_success = False
        self.step = 0

        # make env
        if 'metaworld' in cfg.env:
            self.env = utils.make_metaworld_env(cfg)
            self.log_success = True
        else:
            self.env = utils.make_env(cfg)

        cfg.agent.agent.params.obs_dim = self.env.observation_space.shape[0]
        cfg.agent.agent.params.action_dim = self.env.action_space.shape[0]
        cfg.agent.agent.params.action_range = [
            float(self.env.action_space.low.min()),
            float(self.env.action_space.high.max())
        ]
        cfg.agent.agent.params.critic_cfg.params.obs_dim = cfg.agent.agent.params.obs_dim
        cfg.agent.agent.params.critic_cfg.params.action_dim = cfg.agent.agent.params.action_dim
        cfg.agent.agent.params.actor_cfg.params.obs_dim = cfg.agent.agent.params.obs_dim
        cfg.agent.agent.params.actor_cfg.params.action_dim = cfg.agent.agent.params.action_dim

        config_dict = {
            'obs_dim': cfg.agent.agent.params.obs_dim,
            'action_dim': cfg.agent.agent.params.action_dim,
            'action_range': cfg.agent.agent.params.action_range,
            'device': self.device,
            'critic_cfg': cfg.agent.agent.params.critic_cfg,
            'actor_cfg': cfg.agent.agent.params.actor_cfg,
            'discount': cfg.agent.agent.params.discount,
            'init_temperature': cfg.agent.agent.params.init_temperature,
            'alpha_lr': cfg.agent.agent.params.alpha_lr,
            'alpha_betas': cfg.agent.agent.params.alpha_betas,
            'actor_lr': cfg.agent.agent.params.actor_lr,
            'actor_betas': cfg.agent.agent.params.actor_betas,
            'actor_update_frequency': cfg.agent.agent.params.actor_update_frequency,
            'critic_lr': cfg.agent.agent.params.critic_lr,
            'critic_betas': cfg.agent.agent.params.critic_betas,
            'critic_tau': cfg.agent.agent.params.critic_tau,
            'critic_target_update_frequency': cfg.agent.agent.params.critic_target_update_frequency,
            'batch_size': cfg.agent.agent.params.batch_size,
            'learnable_temperature': cfg.agent.agent.params.learnable_temperature,
            'wandb_use': cfg.wandb.use,
        }
        self.agent = SACAgent(config_dict)

        self.replay_buffer = ReplayBuffer(
            self.env.observation_space.shape,
            self.env.action_space.shape,
            int(cfg.replay_buffer_capacity),
            self.device
        )
        meta_file = os.path.join(self.work_dir, 'metadata.pkl')
        pkl.dump({'cfg': self.cfg}, open(meta_file, "wb"))

    def evaluate(self):
        average_episode_reward = 0
        average_true_episode_reward = 0
        if self.log_success:
            success_rate = 0

        for episode in range(self.cfg.num_eval_episodes):
            obs = self.env.reset()
            self.agent.reset()
            done = False
            episode_reward = 0
            true_episode_reward = 0
            if self.log_success:
                episode_success = 0

            episode_step = 0
            while not done:
                with utils.eval_mode(self.agent):
                    action = self.agent.act(obs, sample=False)
                try:
                    obs, reward, done, truncated, extra = self.env.step(action)
                except:
                    obs, reward, done, extra = self.env.step(action)

                episode_step += 1
                done_no_max = 0 if episode_step + 1 == self.env._max_episode_steps else done
                if 'metaworld' in self.cfg.env and episode_step + 1 == self.env._max_episode_steps:
                    done = 1
                    episode_step = 0

                episode_reward += reward
                true_episode_reward += reward
                if self.log_success:
                    episode_success = max(episode_success, extra['success'])

            average_episode_reward += episode_reward
            average_true_episode_reward += true_episode_reward
            if self.log_success:
                success_rate += episode_success

        average_episode_reward /= self.cfg.num_eval_episodes
        average_true_episode_reward /= self.cfg.num_eval_episodes
        if self.log_success:
            success_rate /= self.cfg.num_eval_episodes
            success_rate *= 100.0

        if self.cfg.wandb.use:
            wandb.log({
                'eval/episode_reward': average_episode_reward,
                'eval/true_episode_reward': average_true_episode_reward,
            }, step=self.step)

        if self.log_success:
            if self.cfg.wandb.use:
                wandb.log({
                    'eval/success_rate': success_rate,
                    'eval/true_episode_success': success_rate,
                }, step=self.step)

    def learn_reward(self, first_flag=0):
        # get feedbacks
        labeled_queries, noisy_queries = 0, 0
        if first_flag == 1:
            # if it is first time to get feedback, need to use random sampling
            labeled_queries = self.reward_model.uniform_sampling()
        else:
            if self.cfg.feed_type == 0:
                labeled_queries = self.reward_model.uniform_sampling()
            elif self.cfg.feed_type == 1:
                labeled_queries = self.reward_model.disagreement_sampling()
            elif self.cfg.feed_type == 2:
                labeled_queries = self.reward_model.entropy_sampling()
            elif self.cfg.feed_type == 3:
                labeled_queries = self.reward_model.kcenter_sampling()
            elif self.cfg.feed_type == 4:
                labeled_queries = self.reward_model.kcenter_disagree_sampling()
            elif self.cfg.feed_type == 5:
                labeled_queries = self.reward_model.kcenter_entropy_sampling()
            else:
                raise NotImplementedError

        self.total_feedback += self.reward_model.mb_size
        self.labeled_feedback += labeled_queries

        if self.labeled_feedback > 0:
            # update reward
            for epoch in range(self.cfg.reward_update):
                if self.cfg.label_margin > 0 or self.cfg.teacher_eps_equal > 0:
                    train_acc = self.reward_model.train_soft_reward()
                else:
                    train_acc = self.reward_model.train_reward()
                total_acc = np.mean(train_acc)

                if total_acc > 0.97:
                    break

        print("Reward function is updated!! ACC: " + str(total_acc))

    def r_hat_critic_old(self, x):
        batch_size, segment_length, obsact = x.shape  # _ = obs+act
        assert obsact == self.env.observation_space.shape[0] + self.env.action_space.shape[0]
        obs = x[:, 0, :self.env.observation_space.shape[0]].reshape(batch_size, self.env.observation_space.shape[0])
        act = x[:, 0, self.env.observation_space.shape[0]:].reshape(batch_size, self.env.action_space.shape[0])
        obs = torch.from_numpy(obs).float().to(self.device)
        act = torch.from_numpy(act).float().to(self.device)

        q1, q2 = self.agent.critic_old(obs, act)
        assert q1.shape == (batch_size, 1)

        return q1, q2

    def bilevel_update(self):
        # sample from replay buffer and get meta reward from reward model (with grad)
        obs, action, reward, next_obs, not_done, not_done_no_max = self.replay_buffer.sample(self.agent.batch_size)

        inputs = np.concatenate([obs.cpu(), action.cpu()], axis=-1)
        reward = self.reward_model.r_hat_batch_grad(inputs)

        if self.cfg.wandb.use:
            wandb.log({
                'train/batch_reward': reward.detach().cpu().numpy().mean(),
            }, step=self.step)

        # load parameters of critic_old from current critic
        critic_cfg = self.cfg.agent.agent.params.critic_cfg
        self.agent.critic_old = DoubleQCritic(obs_dim=critic_cfg.params.obs_dim, 
                        action_dim=critic_cfg.params.action_dim,
                        hidden_dim=critic_cfg.params.hidden_dim,
                        hidden_depth=critic_cfg.params.hidden_depth,
                        ).to(self.device)
        self.agent.update_critic_old()

        # calculate target_Q for critic_old
        dist = self.agent.actor(next_obs)
        next_action = dist.rsample()
        log_prob = dist.log_prob(next_action).sum(-1, keepdim=True)
        target_Q1, target_Q2 = self.agent.critic_target(next_obs, next_action)
        target_V = torch.min(target_Q1, target_Q2) - self.agent.alpha.detach() * log_prob
        target_V = target_V.detach()
        target_Q = reward + (not_done * self.agent.discount * target_V)

        # get Q estimates of critic_old
        current_Q1, current_Q2 = self.agent.critic_old(obs, action)
        critic_old_loss = F.mse_loss(current_Q1, target_Q) + F.mse_loss(current_Q2, target_Q)

        # optimize the critic_old
        pseudo_grads = torch.autograd.grad(critic_old_loss, self.agent.critic_old.parameters(), create_graph=True)
        critic_old_optimizer = MetaOptim(self.agent.critic_old, self.agent.critic_old.parameters(), lr=self.cfg.agent.agent.params.critic_lr)
        critic_old_optimizer.load_state_dict(self.agent.critic_optimizer.state_dict())
        critic_old_optimizer.meta_step(pseudo_grads)
        del pseudo_grads

        # calculate loss using trajectory preferences
        index = self.reward_model.buffer_index
        num_eval_pref = self.cfg.reward_batch
        if index < self.cfg.reward_batch:
            idxs = np.append(np.arange(index), np.arange(self.reward_model.capacity - num_eval_pref + index, self.reward_model.capacity))
        else:
            idxs = np.arange(index - num_eval_pref, index)
        np.random.shuffle(idxs)

        sa_t_1 = self.reward_model.buffer_seg1[idxs]  # (B x len_segment x (obs+act))
        sa_t_2 = self.reward_model.buffer_seg2[idxs]  # (B x len_segment x (obs+act))
        labels = self.reward_model.buffer_label[idxs]  # (B x 1)
        labels = torch.from_numpy(labels.flatten()).long().to(self.device)  # (B) [1, 0, 0, 1, 0, 1, 0, 0, 1, 1]

        # get r_hat estimates from critic_old
        r_hat_critic1_q1, r_hat_critic1_q2 = self.r_hat_critic_old(sa_t_1)  # (B x 1)
        r_hat_critic2_q1, r_hat_critic2_q2 = self.r_hat_critic_old(sa_t_2)  # (B x 1)
        r_hat_critic_q1 = torch.cat([r_hat_critic1_q1, r_hat_critic2_q1], axis=-1)  # (B x 2)
        r_hat_critic_q2 = torch.cat([r_hat_critic1_q2, r_hat_critic2_q2], axis=-1)  # (B x 2)

        # compute loss CE((B x 2), (B)) + CE((B x 2), (B))
        outer_loss = (F.cross_entropy(r_hat_critic_q1, labels) + F.cross_entropy(r_hat_critic_q2, labels)) * self.cfg.outer_weight

        # optimize the reward function
        self.reward_model.opt.zero_grad()
        outer_loss.backward()
        self.reward_model.opt.step()

        # calculate target_Q for critic
        reward = self.reward_model.r_hat_batch(inputs)
        reward = torch.as_tensor(reward, device=self.device)
        target_Q = (reward + (not_done * self.agent.discount * target_V)).detach()
        current_Q1, current_Q2 = self.agent.critic(obs, action)
        critic_loss = F.mse_loss(current_Q1, target_Q) + F.mse_loss(current_Q2, target_Q)

        # optimize the critic
        self.agent.critic.zero_grad()
        critic_loss.backward()
        self.agent.critic_optimizer.step()

        if self.cfg.wandb.use:
            wandb.log({'train_critic/loss': critic_loss})

        # update actor and alpha
        dist = self.agent.actor(obs)
        action = dist.rsample()
        log_prob = dist.log_prob(action).sum(-1, keepdim=True)
        actor_Q1, actor_Q2 = self.agent.critic(obs, action)

        actor_Q = torch.min(actor_Q1, actor_Q2)
        actor_loss = (self.agent.alpha.detach() * log_prob - actor_Q).mean()

        if self.cfg.wandb.use:
            wandb.log({'train_actor/loss': actor_loss, 'train_actor/target_entropy': self.agent.target_entropy,
                       'train_actor/entropy': -log_prob.mean()}, step=self.step)

        # optimize the actor
        self.agent.actor_optimizer.zero_grad()
        actor_loss.backward()
        self.agent.actor_optimizer.step()

        if self.agent.learnable_temperature:
            self.agent.log_alpha_optimizer.zero_grad()
            alpha_loss = (self.agent.alpha * (-log_prob - self.agent.target_entropy).detach()).mean()

            if self.cfg.wandb.use:
                wandb.log({'train_alpha/loss': alpha_loss, 'train_alpha/value': self.agent.alpha}, step=self.step)

            alpha_loss.backward()
            self.agent.log_alpha_optimizer.step()

        if self.step % self.agent.critic_target_update_frequency == 0:  # critic_target_update_frequency = 2
            utils.soft_update_params(self.agent.critic, self.agent.critic_target, self.agent.critic_tau)

    def run(self):
        episode, episode_reward, done = 0, 0, True
        if self.log_success:
            episode_success = 0
        true_episode_reward = 0

        # store train returns of recent 10 episodes
        avg_train_true_return = deque([], maxlen=10)
        start_time = time.time()

        while self.step < self.cfg.num_train_steps:
            if done:
                if self.step > 0:
                    start_time = time.time()
                    if self.cfg.wandb.use:
                        wandb.log({
                            'train/duration': time.time() - start_time
                        }, step=self.step)

                # evaluate agent periodically
                if self.step > 0 and self.step % self.cfg.eval_frequency == 0:
                    if self.cfg.wandb.use:
                        wandb.log({
                            'eval/episode': episode
                        }, step=self.step)
                    self.evaluate()

                if self.cfg.wandb.use:
                    wandb.log({
                        'train/episode_reward': episode_reward,
                        'train/true_episode_reward': true_episode_reward,
                    }, step=self.step)

                if self.log_success:
                    if self.cfg.wandb.use:
                        wandb.log({
                            'train/episode_success': episode_success
                        }, step=self.step)

                obs = self.env.reset()
                self.agent.reset()
                done = False
                episode_reward = 0
                avg_train_true_return.append(true_episode_reward)
                true_episode_reward = 0
                if self.log_success:
                    episode_success = 0
                episode_step = 0
                episode += 1

                if self.cfg.wandb.use:
                    wandb.log({
                        'train/episode': episode
                    }, step=self.step)

            # sample action for data collection
            if self.step < self.cfg.num_seed_steps:
                action = self.env.action_space.sample()
            else:
                with utils.eval_mode(self.agent):
                    action = self.agent.act(obs, sample=True)

            # run training update
            if self.step == (self.cfg.num_seed_steps + self.cfg.num_unsup_steps) and self.cfg.num_unsup_steps > 0:
                # reset Q due to unsupervised exploration
                self.agent.reset_critic()
                # update agent
                self.agent.update_after_reset(
                    self.replay_buffer, self.logger, self.step, 
                    gradient_update=self.cfg.reset_update, 
                    policy_update=True
                )

            elif self.step > self.cfg.num_seed_steps + self.cfg.num_unsup_steps:
                self.agent.update(self.replay_buffer, self.logger, self.step)

            # unsupervised exploration
            elif self.step > self.cfg.num_seed_steps:
                self.agent.update_state_ent(self.replay_buffer, self.logger, self.step, gradient_update=1,
                                            K=self.cfg.topK)

            try:
                next_obs, reward, done, truncated, extra = self.env.step(action)
            except:
                next_obs, reward, done, extra = self.env.step(action)
            # allow infinite bootstrap
            done = float(done)
            done_no_max = 0 if episode_step + 1 == self.env._max_episode_steps else done
            if 'metaworld' in self.cfg.env and episode_step + 1 == self.env._max_episode_steps:
                done = 1
            episode_reward += reward
            true_episode_reward += reward

            if self.log_success:
                episode_success = max(episode_success, extra['success'])

            self.replay_buffer.add(obs, action, reward, next_obs, done, done_no_max)

            obs = next_obs
            episode_step += 1
            self.step += 1

        save_dir = os.path.join(self.work_dir, self.cfg.wandb.version)
        if not os.path.exists(save_dir):
            os.makedirs(save_dir)
        self.agent.save(save_dir, self.step)
        self.reward_model.save(save_dir, self.step)

@hydra.main(config_path="config", config_name="train")
def main(cfg):
    # wandb initialize
    if 'debug' in cfg.wandb.tag:
        datetime_now = str(datetime.datetime.today()).split(".")[0].replace(" ","_")
        cfg.wandb.tag = datetime_now + "_" + cfg.wandb.tag
    print(cfg.wandb.version)
    print(type(cfg.wandb.version))
    cfg.wandb.version += "_" + cfg.wandb.tag
    if cfg.wandb.use:
        wandb.init(project="{}".format(cfg.env), entity=cfg.wandb.username, reinit=True)
        wandb.run.name = cfg.wandb.version
        wandb.config.update(cfg)
        
    workspace = Workspace(cfg)
    workspace.run()


if __name__ == '__main__':
    main()
