from .graphics import *
from .entities import RectangleEntity, CircleEntity, RingEntity

from PIL import Image, ImageDraw

import pdb

class Visualizer:
    def __init__(self, width: float, height: float, ppm: int):
        # pdb.set_trace()
        self.ppm = ppm
        self.display_width, self.display_height = int(width * ppm), int(height * ppm)
        self.visualized_imgs = []
        # Initialize an empty image canvas
        self.canvas = Image.new("RGB", (self.display_width, self.display_height), "gray80")
        self.draw = ImageDraw.Draw(self.canvas)
        
    def create_canvas(self, bg_color: str = 'gray80'):
        # Re-create the canvas if needed
        self.canvas = Image.new("RGB", (self.display_width, self.display_height), bg_color)
        self.draw = ImageDraw.Draw(self.canvas)
        self.visualized_imgs = []

    def remove_agents(self, agents: list):
        # Simply clear the image if needed, or reinitialize the canvas
        self.create_canvas(bg_color="gray80")

    def update_agents(self, agents: list):
        # Clear movable agents from the canvas by re-initializing the draw object
        self.create_canvas(bg_color="gray80")

        for agent in agents:
            if isinstance(agent, RectangleEntity):
                C = [self.ppm * c for c in agent.corners]
                # Draw the polygon rectangle on the canvas
                points = [(c.x, self.display_height - c.y) for c in C]
                self.draw.polygon(points, fill=agent.color)
            elif isinstance(agent, CircleEntity):
                # Draw the circle on the canvas
                x, y = self.ppm * agent.center.x, self.display_height - self.ppm * agent.center.y
                r = self.ppm * agent.radius
                self.draw.ellipse((x - r, y - r, x + r, y + r), fill=agent.color)
            elif isinstance(agent, RingEntity):
                # Draw the ring as two circles to simulate the ring effect
                x, y = self.ppm * agent.center.x, self.display_height - self.ppm * agent.center.y
                outer_r, inner_r = self.ppm * agent.outer_radius, self.ppm * agent.inner_radius
                self.draw.ellipse((x - outer_r, y - outer_r, x + outer_r, y + outer_r), outline=agent.color)
                self.draw.ellipse((x - inner_r, y - inner_r, x + inner_r, y + inner_r), fill="gray80")  # background color
            else:
                raise NotImplementedError

    def save_fig(self, save_path):
        self.canvas.save(save_path)

    def close(self):
        self.visualized_imgs = []