from copy import deepcopy
import collections


class AttributeDict(dict):
    """ A dict which allows attribute access to its keys."""

    def __getattr__(self, *args, **kwargs):
        try:
            return self.__getitem__(*args, **kwargs)
        except KeyError as e:
            raise AttributeError(e)

    def __deepcopy__(self, memo):
        """ In order to support deepcopy"""
        return self.__class__(
            [(deepcopy(k, memo=memo), deepcopy(v, memo=memo)) for k, v in self.items()])

    def __setattr__(self, key, value):
        self.__setitem__(key, value)

    def __delattr__(self, item):
        self.__delitem__(item)


class ImmutableAttributeDict(AttributeDict):
    """ A dict which allows attribute access to its keys. Forced immutable."""

    def __delattr__(self, item):
        raise TypeError("Setting object not mutable after settings are fixed!")

    def __delitem__(self, item):
        raise TypeError("Setting object not mutable after settings are fixed!")

    def __setattr__(self, key, value):
        raise TypeError("Setting object not mutable after settings are fixed!")

    def __setitem__(self, key, value):
        raise TypeError("Setting object not mutable after settings are fixed!")

    def __reduce__(self):
        return (AttributeDict, (self.__getstate__(),))

    def __getstate__(self):
        return self


def recursive_objectify(nested_dict):
    return ImmutableAttributeDict({k: recursive_objectify(v) if isinstance(v, collections.Mapping) else v for k, v in deepcopy(nested_dict).items()})


def recursive_dictify(nested_attr_dict):
    return dict({k: recursive_dictify(v) if isinstance(v, collections.Mapping) else v for k, v in deepcopy(nested_attr_dict).items()})
