from __future__ import annotations

from collections import defaultdict
from datetime import datetime
from typing import Any, Dict, List

from ..utils.io import read_json, write_json


def add_accumulated_news(
    input_file: str,
    output_file: str,
) -> None:
    """Given a list produced by reformatting (one row per question_id/start_time with best_news),
    add an `accumulated_news` field to each row containing the timeline up to that point for its question_id.
    """
    data: List[Dict[str, Any]] = read_json(input_file)

    grouped: Dict[Any, List[Dict[str, Any]]] = defaultdict(list)
    for entry in data:
        grouped[entry["question_id"]].append(entry)

    for _, entries in grouped.items():
        # sort by start_time ascending
        entries.sort(key=lambda x: datetime.strptime(x["start_time"], "%Y-%m-%d"))
        acc: List[Dict[str, Any]] = []
        for e in entries:
            acc.append({"start_time": e["start_time"], "best_news": e["best_news"]})
            e["accumulated_news"] = list(acc)

    write_json(output_file, data, indent=2)
