from __future__ import annotations

from typing import Any, Dict, List

from ..utils.dates import timestamp_to_date
from ..utils.io import read_json, write_json


def attach_history_from_binary(
    accumulated_input_file: str,
    binary_questions_file: str,
    output_file: str,
) -> None:
    """Merge simplified daily history from a raw binary questions dump into each accumulated entry.

    For each entry in accumulated_input_file (which contains question_id), find matching question in
    binary_questions_file, extract last history entry per day with fields start_time (YYYY-MM-DD) and mean,
    then attach as `history` list to the entry, and write to output_file.
    """
    acc_data: List[Dict[str, Any]] = read_json(accumulated_input_file)
    bin_data = read_json(binary_questions_file)

    if isinstance(bin_data, list):
        bin_lookup: Dict[Any, Dict[str, Any]] = {q.get("id"): q for q in bin_data}
    else:
        bin_lookup = bin_data

    output: List[Dict[str, Any]] = []
    for entry in acc_data:
        qid = entry.get("question_id")
        bin_entry = bin_lookup.get(qid)
        if not bin_entry:
            continue
        history = (
            (bin_entry.get("question", {}) or {})
            .get("aggregations", {})
            .get("recency_weighted", {})
            .get("history", [])
        )
        date_to_entry: Dict[str, Dict[str, Any]] = {}
        for h in history:
            start_time = h.get("start_time")
            means = h.get("means")
            if start_time is not None and means and len(means) > 0:
                date_str = timestamp_to_date(start_time)
                # Keep only the last entry of the date
                date_to_entry[date_str] = {"start_time": date_str, "mean": means[0]}
        simplified_history = list(date_to_entry.values())
        simplified_history.sort(key=lambda x: x["start_time"])  # chronological
        e = dict(entry)
        e["history"] = simplified_history
        output.append(e)

    write_json(output_file, output, indent=2)
