from __future__ import annotations

import os
from dataclasses import dataclass
from typing import Optional


@dataclass
class PipelineConfig:
    # Data paths (directories or files)
    input_path: Optional[str] = None
    output_path: Optional[str] = None
    cache_path: Optional[str] = None
    log_path: Optional[str] = None

    # External services / API keys
    metaculus_api_key: Optional[str] = None
    metaculus_api_url: str = "https://www.metaculus.com/api"

    google_api_key: Optional[str] = None
    google_search_engine_id: Optional[str] = None
    google_search_url: str = "https://www.googleapis.com/customsearch/v1"

    device: Optional[str] = None  # e.g. "cuda" or "cpu"


def load_config_from_env(prefix: str = "ICLR_") -> PipelineConfig:
    """Load a PipelineConfig from environment variables.

    Known variables (all optional):
    - ICLR_INPUT_PATH
    - ICLR_OUTPUT_PATH
    - ICLR_CACHE_PATH
    - ICLR_LOG_PATH
    - ICLR_METACULUS_API_KEY
    - ICLR_METACULUS_API_URL
    - ICLR_GOOGLE_API_KEY
    - ICLR_GOOGLE_SEARCH_ENGINE_ID
    - ICLR_GOOGLE_SEARCH_URL
    - ICLR_DEVICE
    """
    env = os.environ.get
    return PipelineConfig(
        input_path=env(f"{prefix}INPUT_PATH"),
        output_path=env(f"{prefix}OUTPUT_PATH"),
        cache_path=env(f"{prefix}CACHE_PATH"),
        log_path=env(f"{prefix}LOG_PATH"),
        metaculus_api_key=env(f"{prefix}METACULUS_API_KEY"),
        metaculus_api_url=env(f"{prefix}METACULUS_API_URL", "https://www.metaculus.com/api"),
        google_api_key=env(f"{prefix}GOOGLE_API_KEY"),
        google_search_engine_id=env(f"{prefix}GOOGLE_SEARCH_ENGINE_ID"),
        google_search_url=env(f"{prefix}GOOGLE_SEARCH_URL", "https://www.googleapis.com/customsearch/v1"),
        device=env(f"{prefix}DEVICE"),
    )
