from __future__ import annotations

from datetime import datetime
from typing import Optional, Union


def format_date_iso(value: Union[int, float, str]) -> Optional[str]:
    """Convert a timestamp or an ISO-like string to YYYY-MM-DD.

    Returns None if parse fails.
    """
    if isinstance(value, (int, float)):
        return datetime.utcfromtimestamp(value).strftime("%Y-%m-%d")

    if not isinstance(value, str):
        return None

    # Try multiple formats commonly seen in the raw scripts
    for fmt in (
        "%Y-%m-%dT%H:%M:%S.%fZ",
        "%Y-%m-%dT%H:%M:%SZ",
        "%Y-%m-%d",
    ):
        try:
            return datetime.strptime(value, fmt).strftime("%Y-%m-%d")
        except ValueError:
            continue
    return None


def timestamp_to_date(ts: Union[int, float]) -> str:
    return datetime.utcfromtimestamp(ts).strftime("%Y-%m-%d")
